/* HyperCubeFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs.standard;

import java.util.HashMap;
import java.util.Map;

import org.grinvin.factories.FactoryParameterException;
import org.grinvin.factories.graphs.AbstractGraphFactory;
import org.grinvin.graphs.Annotation;
import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.Graph;
import org.grinvin.graphs.Vertex;

/**
 * Factory for cyclic graphs of given order.
 */
public class HyperCubeFactory extends AbstractGraphFactory {
    
    //
    @Override
    protected void createGraph(Graph graph, Embedding embedding, Annotation annotation) {
        int dimension = ((Integer)values[0]).intValue();
        boolean showCoordinates = ((Boolean)values[1]).booleanValue();
        long order = 1 << dimension;
        embedding.setDimension(2);
        Map<String, Vertex> vertices = new HashMap<String, Vertex>();
        for(int i = 0; i < order; i++) {
            double angle = 2 * i * Math.PI / order;
            double[] coords = new double[2];
            coords[0] = Math.cos(angle);
            coords[1] = Math.sin(angle);
            String coordinate = padWithZeroes(Integer.toBinaryString(i),dimension);
            Vertex v = graph.addNewVertex();
            if (showCoordinates)
                annotation.setAnnotation(v, coordinate);
            vertices.put(coordinate,v);
            embedding.setCoordinates(v, coords);
        }
        
        for (int i=0; i < order-1; i++){
            String coordinate = padWithZeroes(Integer.toBinaryString(i),dimension);
            for(int j=0; j<dimension; j++){
                if(coordinate.charAt(j)=='0'){
                    String secondCoordinate = padWithZeroes(Integer.toBinaryString(i + (1 << (dimension - 1 - j))),dimension);
                    graph.addNewEdge(vertices.get(coordinate), vertices.get(secondCoordinate));
                }
            }
        }
    }
    
    //
    @Override
    protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        int dimension = ((Integer)values[0]).intValue();
        if (dimension < 0 || dimension > 31)
            throw new FactoryParameterException("Dimension should be between 0 and 31.");
    }
    
    //
    private static final String LOTS_OF_ZEROES = "000000000000000000000000";
    
    private String padWithZeroes(String s, int d){
        return LOTS_OF_ZEROES.substring(s.length(), d) + s;
    }
}
