/* DefaultGraphListModel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list.graphs;

import be.ugent.caagt.swirl.lists.ArrayListModel;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.grinvin.graphs.GraphURI;
import org.grinvin.invariants.Invariant;

/**
 * Default implementation of {@link GraphListModel}.
 */
public class DefaultGraphListModel extends ArrayListModel<GraphListElement> implements GraphListModel, GraphListElementListener {
    
    //
    private URI uri;
    
    //
    private List<GraphListModelListener> listeners = new ArrayList<GraphListModelListener>();
    
    //
    private List<GraphListModelContentsListener> contentsListeners = new ArrayList<GraphListModelContentsListener>();
    
    //
    public void add(int index, GraphListElement element) {
        element.addGraphListElementListener(this);
        super.add(index, element);
    }
    
    //
    public boolean add(GraphListElement element) {
        element.addGraphListElementListener(this);
        return super.add(element);
    }
    
    //
    public boolean addAll(Collection<? extends GraphListElement> collection) {
        for(GraphListElement element : collection) {
            element.addGraphListElementListener(this);
        }
        return super.addAll(collection);
    }
    
    //
    public boolean addAll(int index, Collection<? extends GraphListElement> collection) {
        for(GraphListElement element : collection) {
            element.addGraphListElementListener(this);
        }
        return super.addAll(index, collection);
    }

    //
    public void clear() {
        super.clear();
    }
    
    //
    public boolean remove(Object obj) {
        return super.remove(obj);
    }
    
    //
    public GraphListElement remove(int index) {
        return super.remove(index);
    }
    
    //
    protected void removeRange(int fromIndex, int toIndex) {
        super.removeRange(fromIndex, toIndex);
    }
    
    //
    public GraphListElement set(int index, GraphListElement element) {
        return super.set(index, element);
    }
    
    public void fireInvariantComputed(int index, Invariant invariant) {
        for (GraphListModelContentsListener listener : contentsListeners) {
            listener.graphInvariantComputed(index, invariant);
        }
    }
    
    /**
     * Get the uri.
     * @return the uri, or null
     */
    public URI getURI() {
        return uri;
    }
    
    /**
     * Set the URI of this GraphListElement. Use this to set the URI when the
     * bundle has been saved.
     * @param uri the URI
     */
    public void setURI(URI uri) {
        this.uri = uri;
    }
    
    //
    public boolean containsSessionElements() {
        for (GraphListElement element : this) {
            if (GraphURI.isSession(element.getURI()))
                return true;
        }
        return false;
    }

    //
    public void addGraphListModelListener(GraphListModelListener listener) {
        listeners.add(listener);
        this.addListDataListener(listener);
    }

    //
    public void addGraphListModelContentsListener(GraphListModelContentsListener listener) {
        contentsListeners.add(listener);
        addGraphListModelListener(listener);
    }
    
    //
    public void removeGraphListModelListener(GraphListModelListener listener) {
        this.removeListDataListener(listener);
        listeners.remove(listener);
    }

    // implements GraphListElementListener
    public void graphListElementURIChanged(GraphListElement element) {
        //repaint icon
        int index = indexOf(element);
        fireContentsChanged(this, index, index);
        // make sure the list is saved again
    }

    // implements GraphListElementListener
    public void graphListElementNameChanged(GraphListElement element) {
        //repaint name
        int index = indexOf(element);
        fireContentsChanged(this, index, index);
    }

    // implements GraphListElementListener
    public void graphListElementInvariantComputed(GraphListElement element, Invariant invariant) {
        int index = indexOf(element);
        fireInvariantComputed(index, invariant);
    }

    // implements GraphListElementListener
    public void graphListElementGraphChanged(GraphListElement element) {
        int index = indexOf(element);
        fireContentsChanged(this, index, index);
    }

}
