/*  This file is part of the KDE libraries
    Copyright (C) 1999 Waldo Bastian (bastian@kde.org)
    Modified by Toan T Nguyen (nguyenthetoan@gmail.com) to allow for
    listbox selection

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; version 2
    of the License.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <qcheckbox.h>
#include <qguardedptr.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qvbox.h>
#include <qvgroupbox.h>
#include <qstylesheet.h>
#include <qsimplerichtext.h>
#include <qpushbutton.h>
#include <qlayout.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include <kguiitem.h>
#include <klistbox.h>
#include <klocale.h>
//#include <kmessagebox.h>
#include <knotifyclient.h>
#include <kstdguiitem.h>
#include <kactivelabel.h>
#include <kiconloader.h>
#include <kglobalsettings.h>

#ifdef Q_WS_X11
#include <X11/Xlib.h>
#endif

#include "kxmessagebox.h"

 /**
  * Easy MessageBox Dialog.
  *
  * Provides convenience functions for some i18n'ed standard dialogs,
  * as well as audible notification via @ref KNotifyClient
  *
  * @author Waldo Bastian (bastian@kde.org)
  */

static bool KxmameMessageBox_queue = false;

static QPixmap themedMessageBoxIcon(QMessageBox::Icon icon)
{
    QString icon_name;

    switch(icon)
    {
    case QMessageBox::NoIcon:
        return QPixmap();
        break;
    case QMessageBox::Information:
        icon_name = "messagebox_info";
        break;
    case QMessageBox::Warning:
        icon_name = "messagebox_warning";
        break;
    case QMessageBox::Critical:
        icon_name = "messagebox_critical";
        break;
    default:
        break;
    }

   QPixmap ret = KApplication::kApplication()->iconLoader()->loadIcon(icon_name, KIcon::NoGroup, KIcon::SizeMedium, KIcon::DefaultState, 0, true);

   if (ret.isNull())
       return QMessageBox::standardIcon(icon);
   else
       return ret;
}

static void sendNotification( QString message,
                              const QStringList& strlist,
                              QMessageBox::Icon icon,
                              WId parent_id )
{
    // create the message for KNotify
    QString messageType;
    switch ( icon )
    {
        case QMessageBox::Warning:
            messageType = "messageWarning";
            break;
        case QMessageBox::Critical:
            messageType = "messageCritical";
            break;
        case QMessageBox::Question:
            messageType = "messageQuestion";
            break;
        default:
            messageType = "messageInformation";
            break;
    }

    if ( !strlist.isEmpty() )
    {
        for ( QStringList::ConstIterator it = strlist.begin(); it != strlist.end(); ++it )
            message += "\n" + *it;
    }

    if ( !message.isEmpty() )
        KNotifyClient::event( (int)parent_id, messageType, message );
}

static QString qrichtextify( const QString& text )
{
  if ( text.isEmpty() || text[0] == '<' )
    return text;

  QStringList lines = QStringList::split('\n', text);
  for(QStringList::Iterator it = lines.begin(); it != lines.end(); ++it)
  {
    *it = QStyleSheet::convertFromPlainText( *it, QStyleSheetItem::WhiteSpaceNormal );
  }

  return lines.join(QString::null);
}

int KxmameMessageBox::createKxmameMessageBox(int *listitemSelected, KDialogBase *dialog, QMessageBox::Icon icon,
                             const QString &text, const QStringList &strlist,
                             const QString &ask, bool *checkboxReturn,
                             int options, const QString &details)
{
	return createKxmameMessageBox(listitemSelected, dialog, themedMessageBoxIcon(icon), text, strlist,
                      ask, checkboxReturn, options, details, icon);
}

int KxmameMessageBox::createKxmameMessageBox(int *listitemSelected, KDialogBase *dialog, QPixmap icon,
                             const QString &text, const QStringList &strlist,
                             const QString &ask, bool *checkboxReturn, int options,
                             const QString &details, QMessageBox::Icon notifyType)
{
    QVBox *topcontents = new QVBox (dialog);
    topcontents->setSpacing(KDialog::spacingHint()*2);
    topcontents->setMargin(KDialog::marginHint());

    QWidget *contents = new QWidget(topcontents);
    QHBoxLayout * lay = new QHBoxLayout(contents);
    lay->setSpacing(KDialog::spacingHint());

    QLabel *label1 = new QLabel( contents);

    if (!icon.isNull())
       label1->setPixmap(icon);

    lay->addWidget( label1, 0, Qt::AlignCenter );
    lay->addSpacing(KDialog::spacingHint());
    // Enforce <p>text</p> otherwise the word-wrap doesn't work well
    QString qt_text = qrichtextify( text );

    int pref_width = 0;
    int pref_height = 0;
    // Calculate a proper size for the text.
    {
       QSimpleRichText rt(qt_text, dialog->font());
       QRect d = KGlobalSettings::desktopGeometry(dialog);

       pref_width = d.width() / 3;
       rt.setWidth(pref_width);
       int used_width = rt.widthUsed();
       pref_height = rt.height();
       if (3*pref_height > 2*d.height())
       {
          // Very high dialog.. make it wider
          pref_width = d.width() / 2;
          rt.setWidth(pref_width);
          used_width = rt.widthUsed();
          pref_height = rt.height();
       }
       if (used_width <= pref_width)
       {
          while(true)
          {
             int new_width = (used_width * 9) / 10;
             rt.setWidth(new_width);
             int new_height = rt.height();
             if (new_height > pref_height)
                break;
             used_width = rt.widthUsed();
             if (used_width > new_width)
                break;
          }
          pref_width = used_width;
       }
       else
       {
          if (used_width > (pref_width *2))
             pref_width = pref_width *2;
          else
             pref_width = used_width;
       }
    }
    KActiveLabel *label2 = new KActiveLabel( qt_text, contents );
    if (!(options & KxmameMessageBox::AllowLink))
    {
       QObject::disconnect(label2, SIGNAL(linkClicked(const QString &)),
                  label2, SLOT(openLink(const QString &)));
    }

    // We add 10 pixels extra to compensate for some KActiveLabel margins.
    // TODO: find out why this is 10.
    label2->setFixedSize(QSize(pref_width+10, pref_height));
    lay->addWidget( label2 );
    lay->addStretch();

    KListBox *listbox = 0;
    if (!strlist.isEmpty())
    {
       listbox=new KListBox( topcontents );
       listbox->insertStringList( strlist );
       //listbox->setSelectionMode( QListBox::NoSelection );
       listbox->setCurrentItem(0);
       topcontents->setStretchFactor(listbox, 1);
    }

    QGuardedPtr<QCheckBox> checkbox = 0;
    if (!ask.isEmpty())
    {
       checkbox = new QCheckBox(ask, topcontents);
       if (checkboxReturn)
         checkbox->setChecked(*checkboxReturn);
    }

    if (!details.isEmpty())
    {
       QVGroupBox *detailsGroup = new QVGroupBox( i18n("Details"), dialog);
       if ( details.length() < 512 ) {
         KActiveLabel *label3 = new KActiveLabel(qrichtextify(details),
                                                 detailsGroup);
         label3->setMinimumSize(label3->sizeHint());
	 if (!(options & KxmameMessageBox::AllowLink))
         {
           QObject::disconnect(label3, SIGNAL(linkClicked(const QString &)),
                               label3, SLOT(openLink(const QString &)));
         }
       } else {
         QTextEdit* te = new QTextEdit(details, QString::null, detailsGroup);
         te->setReadOnly( true );
         te->setMinimumHeight( te->fontMetrics().lineSpacing() * 11 );
       }
       dialog->setDetailsWidget(detailsGroup);
    }

    dialog->setMainWidget(topcontents);
    dialog->enableButtonSeparator(false);
    if (!listbox)
       dialog->disableResize();

    const KDialogBase::ButtonCode buttons[] = {
        KDialogBase::Help,
        KDialogBase::Default,
        KDialogBase::Ok,
        KDialogBase::Apply,
        KDialogBase::Try,
        KDialogBase::Cancel,
        KDialogBase::Close,
        KDialogBase::User1,
        KDialogBase::User2,
        KDialogBase::User3,
        KDialogBase::No,
        KDialogBase::Yes,
        KDialogBase::Details };
    for( unsigned int i = 0;
	 i < sizeof( buttons )/sizeof( buttons[ 0 ] );
	 ++i )
	if( QPushButton* btn = dialog->actionButton( buttons[ i ] ))
	    if( btn->isDefault())
		btn->setFocus();

    if ( (options & KxmameMessageBox::Notify) )
        sendNotification( text, strlist, notifyType, dialog->topLevelWidget()->winId());

    if (KxmameMessageBox_queue)
    {
       KDialogQueue::queueDialog(dialog);
       return KxmameMessageBox::Cancel; // We have to return something.
    }

    if ( (options & KxmameMessageBox::NoExec) )
    {
	    return KxmameMessageBox::Cancel; // We have to return something.
    }

    // We use a QGuardedPtr because the dialog may get deleted
    // during exec() if the parent of the dialog gets deleted.
    // In that case the guarded ptr will reset to 0.
    QGuardedPtr<KDialogBase> guardedDialog = dialog;

    int result = guardedDialog->exec();
    if(listbox)
	    *listitemSelected = listbox->currentItem();
    if (checkbox && checkboxReturn)
       *checkboxReturn = checkbox->isChecked();
    delete (KDialogBase *) guardedDialog;
    return result;
}

int
KxmameMessageBox::warningContinueCancel(QWidget *parent,
			const QString &text,
			const QString &caption,
			const KGuiItem &buttonContinue,
			const QString &dontAskAgainName,
			int options)
{
	int unused;
	return warningContinueCancelList(&unused, parent, text, QStringList(), caption,
                                 buttonContinue, dontAskAgainName, options);
}
 

int KxmameMessageBox::warningContinueCancelList(int *listitemSelected, QWidget *parent, const QString &text,
			const QStringList &strlist,
			const QString &caption,
			const KGuiItem &buttonContinue,
			const QString &dontAskAgainName,
			int options)
{
	return warningContinueCancelListWId(listitemSelected, parent ? parent->winId() : 0, text, strlist,
        caption, buttonContinue, dontAskAgainName, options );
}

int KxmameMessageBox::warningContinueCancelListWId(int *listitemSelected, WId parent_id, const QString &text,
			const QStringList &strlist,
			const QString &caption,
			const KGuiItem &buttonContinue,
			const QString &dontAskAgainName,
			int options)
{
//    if ( !shouldBeShownContinue(dontAskAgainName) )
//        return Continue;

    QWidget* parent = QWidget::find( parent_id );
    KDialogBase *dialog= new KDialogBase(
                       caption.isEmpty() ? i18n("Warning") : caption,
                       KDialogBase::Yes | KDialogBase::No,
                       KDialogBase::Yes, KDialogBase::No,
                       parent, "warningYesNo", true, true,
                       buttonContinue, KStdGuiItem::cancel() );
    if( options & PlainCaption )
        dialog->setPlainCaption( caption );
#ifdef Q_WS_X11
    if( parent == NULL && parent_id )
        XSetTransientForHint( qt_xdisplay(), dialog->winId(), parent_id );
#endif

    bool checkboxResult = false;
    int result = createKxmameMessageBox(listitemSelected, dialog, QMessageBox::Warning, text, strlist,
                       dontAskAgainName.isEmpty() ? QString::null : i18n("Do not ask again"),
                       &checkboxResult, options);

    if ( result==KDialogBase::No )
        return Cancel;
//    if (checkboxResult)
//        saveDontShowAgainContinue(dontAskAgainName);
    return Continue;
}

void
KxmameMessageBox::warning(QWidget *parent,  const QString &text,
			const QString &caption, int options)
{
	return warningListWId( parent ? parent->winId() : 0, text, QStringList(), caption, options );
}

void
KxmameMessageBox::warningListWId(WId parent_id,  const QString &text, const QStringList &strlist,
		       const QString &caption, int options)
{
	QWidget* parent = QWidget::find( parent_id );
	KDialogBase *dialog= new KDialogBase(
			caption.isEmpty() ? i18n("Error") : caption,
	KDialogBase::Yes,
	KDialogBase::Yes, KDialogBase::Yes,
	parent, "error", true, true,
	KStdGuiItem::ok() );
	if( options & PlainCaption )
		dialog->setPlainCaption( caption );
#ifdef Q_WS_X11
    if( parent == NULL && parent_id )
        XSetTransientForHint( qt_xdisplay(), dialog->winId(), parent_id );
#endif

    int unused;
    createKxmameMessageBox(&unused, dialog, QMessageBox::Warning, text, strlist, QString::null, 0, options);
}

