/***************************************************************************
 *   Copyright (C) 2003-2005 by Kevin Hessels                              *
 *   khessels@shaw.ca                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "kfoldingconfigdata.h"
#include "kfoldingprocess.h"
#include "kfoldingprocessiface.h"

#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qobject.h>
#include <qregexp.h>

#include <dcopclient.h>
#include <dcopobject.h>
#include <kaction.h>
#include <kdebug.h>
#include <kdirwatch.h>
#include <klocale.h>
#include <kprocess.h>

kfoldingProcess::kfoldingProcess( QObject* parent, const char* name, kfoldingConfigData* config ) : DCOPObject( "kfoldingProcessIFace" ),
									QObject( parent, name ),
									_cfg( config ),
									_process( new KProcess() ),
									_currentProgress( 0 ),
									_doRestart( false ),
									_processState( StateStopped ),
									_watcher( new KDirWatch() )
{
	_process->setUseShell( true );
	
	createActions();
	slotStop();
	DCOPObject::setObjId( "kfolding" );

//	connect( _process, SIGNAL( receivedStdout( KProcess*, char*, int ) ), 
//		 this, SLOT( slotReadStdout( KProcess*, char*, int ) ) );
	connect( _process, SIGNAL( processExited( KProcess* ) ),
		 this, SLOT( slotProcessExit( KProcess * ) ) );
		 
	connect( _watcher, SIGNAL( created( const QString& ) ), this, SLOT( slotUnitInfoCreated( const QString& ) ) );
	connect( _watcher, SIGNAL( dirty( const QString& ) ), this, SLOT( slotUnitInfoUpdated( const QString& ) ) ); 
	return;
} // kfoldingProcess ctor



kfoldingProcess::~kfoldingProcess()
{
	slotStop();
	delete _watcher;
	delete _process;
	delete _actionCollection;
	return;
} // kfoldingProcess dtor



kfoldingProcess::State kfoldingProcess::processState()
{
	return _processState;
} // processState


void kfoldingProcess::startFolding()
{
	if ( _processState == kfoldingProcess::StateStopped ) {
		slotStart();
	} // if
	slotUnitInfoUpdated( _cfg->workingDir() + "unitinfo.txt");
	return;
} // startFolding


void kfoldingProcess::stopFolding()
{
	if ( !_processState == kfoldingProcess::StateStopped ) {
		slotStop();
	} // if
	slotUnitInfoUpdated( _cfg->workingDir() + "unitinfo.txt" );
	return;
} // stopFolding



void kfoldingProcess::restart()
{
	_doRestart = true;
	slotStop();
	return;
} // restart


void kfoldingProcess::createActions()
{
	_actionCollection = new KActionCollection( this );

	_startAction = new KAction( i18n( "&Start" ), "player_play", 0,
				    this, SLOT( slotStart() ), _actionCollection, "start" );
				    
	_stopAction = new KAction( i18n( "S&top" ), "player_stop", 0,
				   this, SLOT( slotStop() ), _actionCollection, "stop" );
				   
	_suspendAction = new KAction( i18n( "Sus&pend" ), 0, 0,
				      this, SLOT( slotSuspend() ), _actionCollection, "suspend" );
				     
	_resumeAction  = new KAction( i18n( "&Resume" ), 0, 0,
				      this, SLOT( slotResume() ), _actionCollection, "resume" );
	return;
} // createActions


bool kfoldingProcess::okayToRun()
{
	if ( _cfg->machineId() <= 0 )
		return false;

	// first, check that an executable has been specified and that it's executable bit is set
	if ( _cfg->executable().isEmpty() ) {
		emit processError( kfoldingProcess::ErrorNoExecutable );
		return false;
	}
	
	QFileInfo fileInfo( _cfg->executable() );
	if ( !fileInfo.isExecutable() ) {
		emit processError( kfoldingProcess::ErrorBadExecutable );
		return false;
	}
	
	fileInfo.setFile( "/tmp/fah/" );
	bool errorFound = false;
	if ( fileInfo.exists() && fileInfo.isDir() && fileInfo.isReadable() ) {
		QRegExp rxFileName( "^f(\\d+)$" );
		QDir lockDir = fileInfo.dir();
		int lockMachineId;
		int lockPid;
		QString lockWorkingDir;
		
		QStringList locks = lockDir.entryList( "f*" );
		for ( QStringList::iterator it = locks.begin(); it != locks.end(); ++it ) {
			if ( rxFileName.search( *it ) != -1 ) {
				
				lockMachineId = rxFileName.cap( 1 ).toInt();
				
				QFile lock( "/tmp/fah/" + ( *it ) );
				if ( lock.open( IO_ReadOnly ) ) {
					QTextStream textStream( &lock );
					
					QString strLockPid = textStream.readLine().stripWhiteSpace();
					if ( strLockPid.isEmpty() ) {
						lock.close();
						continue;
					}
					lockPid = strLockPid.toInt();
					
					lockWorkingDir = textStream.readLine().stripWhiteSpace();
					if ( lockWorkingDir.isEmpty() ) {
						lock.close();
						continue;
					} // if
					if ( !lockWorkingDir.endsWith( "/" ) )
						lockWorkingDir.append( '/' );
					lock.close();
					
				} else {
					continue;
				} // else
				
				// Test  for stale lockfile
				if ( ( ( _cfg->lastPid() == lockPid ) || ( _cfg->lastPid() == 0 ) ) && ( _cfg->machineId() == lockMachineId ) ) {
					kdDebug() << "kfoldingProcess::okayToRun(): deleting stale lockfile." << endl;
					if ( !lock.remove() ) {
						emit processError( kfoldingProcess::ErrorMachineId );
						errorFound = true;
						break;
					}
					continue;
				} else if ( ( _cfg->lastPid() != lockPid ) && ( _cfg->machineId() == lockMachineId ) ) {
					emit processError( kfoldingProcess::ErrorMachineId );
					errorFound = true;
					break;
				} // else	
				
				if ( _cfg->workingDir() == lockWorkingDir ) {
					emit processError( kfoldingProcess::ErrorAlreadyRunning );
					errorFound = true;
					break;
				} // if	
			} // if
		} // for	
	} // if
	return !errorFound;
} // okayToRun


void kfoldingProcess::slotReadStdout( KProcess*, char*, int )
{
	//kdDebug() << buffer << endl;
	return;
} // slotReadStdOut



void kfoldingProcess::slotProcessExit( KProcess* )
{
	slotStop();
	
	if ( _doRestart ) {
		slotStart();
		_doRestart = false;
	}
	return;
} // slotProcessExit



void kfoldingProcess::slotStart()
{
	slotStop();
	
	if ( !okayToRun() )
		return;
	
	_watcher->addFile( _cfg->workingDir() + "unitinfo.txt" );
	_process->clearArguments();
	_process->setWorkingDirectory( _cfg->workingDir() );
	
	*_process << _cfg->executable();
	
	if ( _cfg->forceASM() )
		*_process << " -forceasm";
			
	if ( _cfg->useAdvMethods() )
		*_process << " -advmethods";
	
#if defined(Q_OS_FREEBSD)
	*_process << " -freeBSD";
#endif // defined(Q_OS_FREEBSD) 
#if defined(Q_OS_OPENBSD)
	*_process << " -openBSD";
#endif // defined(Q_OS_OPENBSD)
		
	// should this be configurable?
	*_process << " -verbosity 1";

	if ( _process->start() ) {
		_processState = StateRunning;
		
		_startAction->setEnabled( false );
		_stopAction->setEnabled( true );
		_suspendAction->setEnabled( true );
		_suspendAction->setIcon( "player_pause" );
		_resumeAction->setEnabled( false );
		_resumeAction->setIcon( 0 );
		
		emit stateChanged( kfoldingProcess::StateRunning );
		
		_cfg->setLastPid( _process->pid() );
		_cfg->save();
		
		slotUnitInfoUpdated( _cfg->workingDir() + "unitinfo.txt" );
	} else {
		emit processError( kfoldingProcess::ErrorUnknown );
	} // else
	return;
} // slotStart



void kfoldingProcess::slotStop()
{
	if ( _processState == StateStopped ) {
		_stopAction->setEnabled( false );
		return;
	}
		
	_process->kill();
	
	_processState = StateStopped;
		
	_startAction->setEnabled( true );
	_stopAction->setEnabled( false );
	_suspendAction->setEnabled( false );
	_suspendAction->setIcon( 0 );
	_resumeAction->setEnabled( false );
	_resumeAction->setIcon( 0 );

	emit stateChanged( kfoldingProcess::StateStopped );
	return;
} // slotStop



void kfoldingProcess::slotSuspend()
{
	_startAction->setEnabled( false );
	_stopAction->setEnabled( true );
	_suspendAction->setEnabled( false );
	_suspendAction->setIcon( 0 );
	_resumeAction->setEnabled( true );
	_resumeAction->setIcon( "player_pause" );

	if ( _processState != StateRunning )
		return;
		
	_process->suspend();
	_processState = StateSuspended;
	emit stateChanged( kfoldingProcess::StateSuspended );
	return;
} // slotSuspend



void kfoldingProcess::slotResume()
{
	_startAction->setEnabled( false );
	_stopAction->setEnabled( true );
	_suspendAction->setEnabled( true );
	_suspendAction->setIcon( "player_pause" );
	_resumeAction->setEnabled( false );
	_resumeAction->setIcon( 0 );

	if ( _processState != StateSuspended )
		return;
		
	_process->resume();
	_processState = StateRunning;
	emit stateChanged( kfoldingProcess::StateRunning );
	return;
} // slotResume



void kfoldingProcess::slotUnitInfoCreated( const QString& file )
{
	slotUnitInfoUpdated( file );
	emit progressUpdated();
	return;
} // slotUnitInfoCreated



void kfoldingProcess::slotUnitInfoUpdated( const QString& updatedFile )
{
	kdDebug() << "slotUnitInfoUpdated" << endl;
	if ( updatedFile != _cfg->workingDir() + "unitinfo.txt" )
		return;
		
	QFile file( updatedFile );
	
	QRegExp rxPercent( "^Progress: (\\d{1,3}).*$" );
	QRegExp rxName( "^Name: (.*)$" );

	if ( file.open( IO_ReadOnly ) ) {
		QTextStream stream( &file );
		QString line;
		while ( !stream.atEnd() ) {
			line = stream.readLine();
			if ( rxName.search( line ) != -1 ) {
				_currentUnit = rxName.cap( 1 );
			}
			else if ( rxPercent.search( line ) != -1 ) {
				
				// only notify of an updated if the progress really has changed...
				if ( rxPercent.cap( 1 ).toInt() != _currentProgress ) {
					_currentProgress = rxPercent.cap( 1 ).toInt();
					emit progressUpdated();
				} else {
					_currentProgress = rxPercent.cap( 1 ).toInt();
				}
			} // else
		} // while
		file.close();
	} else {
		kdDebug() << _cfg->workingDir() << "unitinfo.txt does not exist." << endl;
		_currentProgress = 0;
		_currentUnit = "";
		emit progressUpdated();
	} // else
	return;
} // slotUnitInfoUpdated
