/*
 * Copyright (c) 2007-2011 by The Broad Institute, Inc. and the Massachusetts Institute of
 * Technology.  All Rights Reserved.
 *
 * This software is licensed under the terms of the GNU Lesser General Public License (LGPL),
 * Version 2.1 which is available at http://www.opensource.org/licenses/lgpl-2.1.php.
 *
 * THE SOFTWARE IS PROVIDED "AS IS." THE BROAD AND MIT MAKE NO REPRESENTATIONS OR
 * WARRANTES OF ANY KIND CONCERNING THE SOFTWARE, EXPRESS OR IMPLIED, INCLUDING,
 * WITHOUT LIMITATION, WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF LATENT OR OTHER DEFECTS, WHETHER
 * OR NOT DISCOVERABLE.  IN NO EVENT SHALL THE BROAD OR MIT, OR THEIR RESPECTIVE
 * TRUSTEES, DIRECTORS, OFFICERS, EMPLOYEES, AND AFFILIATES BE LIABLE FOR ANY DAMAGES
 * OF ANY KIND, INCLUDING, WITHOUT LIMITATION, INCIDENTAL OR CONSEQUENTIAL DAMAGES,
 * ECONOMIC DAMAGES OR INJURY TO PROPERTY AND LOST PROFITS, REGARDLESS OF WHETHER
 * THE BROAD OR MIT SHALL BE ADVISED, SHALL HAVE OTHER REASON TO KNOW, OR IN FACT
 * SHALL KNOW OF THE POSSIBILITY OF THE FOREGOING.
 */

package org.broad.igv.util.sigma;

import org.broad.igv.Globals;
import org.broad.igv.feature.AbstractFeatureParser;
import org.broad.igv.feature.FeatureUtils;
import org.broad.igv.feature.IGVFeature;
import org.broad.igv.feature.genome.GenomeManager;
import org.broad.igv.tools.PreprocessingException;
import org.broad.igv.track.FeatureCollectionSource;
import org.broad.igv.track.FeatureSource;
import org.broad.igv.ui.IGV;
import org.broad.igv.util.ParsingUtils;
import org.broad.igv.util.ResourceLocator;
import org.broad.tribble.Feature;
import org.broad.tribble.readers.AsciiLineReader;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @author jrobinso
 * @date Feb 10, 2011
 */
public class SigmaAnalysis2 {

    static String refSeqURL = "http://www.broadinstitute.org/igvdata/annotations/hg18/refGene.txt";

    FeatureCollectionSource chr22Genes;

    List<FeatureCollectionSource> bedFeatureLists = new ArrayList();

    static String[] bedFiles = {"chr22_snps_10.aa.bed", "chr22_snps_10.bip.bed",
            "chr22_snps_10.l.bed", "chr22_snps_10.la.bed", "chr22_snps_10.ok.bed"};


    public static void main(String[] args) throws IOException {
        Globals.setHeadless(true);
        GenomeManager.GenomeListItem item = IGV.getInstance().getGenomeManager().loadGenomeFromLocalFile(
                new File("/Users/jrobinso/.igv/genomes/hg18.genome"));
        String genomeId = item.getId();

        IGV.getInstance().getGenomeManager().setGenomeId(genomeId);

        (new SigmaAnalysis2()).run();
        

    }

    public void run() throws IOException {


        for (String f : bedFiles) {
            System.out.println(f);
            bedFeatureLists.add(getFeatures(f));
        }
        chr22Genes = getFeatures(refSeqURL);

        for (Feature g : chr22Genes.getFeatureList("chr22")) {

            int start = g.getStart() - 2000;
            int end = g.getEnd() + 2000;

            int score = 0;
            for (FeatureCollectionSource bedFeatures : bedFeatureLists) {
                List<Feature> flist = bedFeatures.getFeatureList("chr22");
                int idx = FeatureUtils.getIndexBefore(start, flist);
                for (int i = idx; i < flist.size(); i++) {
                    Feature f = flist.get(i);
                    if (f.getStart() > end) {
                        break;
                    }
                    if (f.getStart() > start) {
                        score++;
                        break;
                    }
                }

            }

            if (score >= 3) {
                System.out.println("chr22\t" + start + "\t" + end + "\t" + ((IGVFeature) g).getName() + "\t" + score);
            }

        }


    }


    private FeatureCollectionSource getFeatures(String url) {

        String chr = "chr22";

        AsciiLineReader reader = null;
        try {
            ResourceLocator rl = new ResourceLocator(url);
            reader = ParsingUtils.openAsciiReader(rl);


            List<Feature> tmp = AbstractFeatureParser.getInstanceFor(rl).loadFeatures(reader);
            List<Feature> features = new ArrayList(tmp.size());
            for (Feature f : tmp) {
                if (f.getChr().equals(chr)) {
                    features.add(f);
                }
            }

            return new FeatureCollectionSource(features);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

}
