/////////////////////////////////////////////////////////////////////////////
// Name:        DVD.cpp
// Purpose:     The class to store a DVD Structure (Titles/Menus)
// Author:      Alex Thuering
// Created:	29.01.2003
// RCS-ID:      $Id: DVD.cpp,v 1.95 2010/10/10 18:03:10 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     GPL
/////////////////////////////////////////////////////////////////////////////

#include "DVD.h"
#include "Menu.h"
#include "Cache.h"
#include "Config.h"
#include "SubtitlePropDlg.h"
#include <wx/file.h>
#include <wx/filename.h>
#include <wx/tokenzr.h>
#include <wx/sstream.h>
#include <wxSVGXML/svgxmlhelpr.h>
#include <wxVillaLib/utils.h>
#include <wxSVG/mediadec_ffmpeg.h>

#define DATA_FILE(fname) wxFindDataFile(_T("data") + wxString(wxFILE_SEP_PATH) + fname)
#define TEMPLATES_DIR wxFindDataDirectory(_T("templates"))

//////////////////////////////// Cell ////////////////////////////////////
Cell::Cell() {
	m_chapter = false;
	m_program = false;
	m_pause = 0;
}

wxSvgXmlNode* Cell::GetXML(DVDFileType type) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("cell"));
	if (m_start.length())
		node->AddProperty(wxT("start"), m_start);
	if (m_end.length())
		node->AddProperty(wxT("end"), m_end);
	if (m_chapter)
		node->AddProperty(wxT("chapter"), wxT("1"));
	if (m_program)
		node->AddProperty(wxT("program"), wxT("1"));
	if (m_pause != 0)
		node->AddProperty(wxT("pause"), m_pause == -1 ? wxT("inf") : wxString::Format(wxT("%d"), m_pause));
	if (m_commands.length())
		node->AddChild(new wxSvgXmlNode(wxSVGXML_TEXT_NODE, wxEmptyString, m_commands));
	return node;
}

bool Cell::PutXML(wxSvgXmlNode* node) {
	if (node == NULL || node->GetName() != wxT("cell"))
		return false;
	long lval;
	wxString val;
	if (node->GetPropVal(wxT("start"), &val))
		SetStart(val);
	if (node->GetPropVal(wxT("end"), &val))
		SetEnd(val);
	if (node->GetPropVal(wxT("chapter"), &val))
		SetChapter(val == wxT("1") || val == wxT("on") || val == wxT("yes"));
	if (node->GetPropVal(wxT("program"), &val))
		SetProgram(val == wxT("1") || val == wxT("on") || val == wxT("yes"));
	if (node->GetPropVal(wxT("pause"), &val) && val == wxT("inf"))
		SetPause(-1);
	else if (node->GetPropVal(wxT("pause"), &val) && val.ToLong(&lval))
		SetPause(lval);
	if (node->GetChildren() && (node->GetChildren()->GetType() == wxSVGXML_TEXT_NODE
				|| node->GetChildren()->GetType() == wxSVGXML_CDATA_SECTION_NODE))
		SetCommands(node->GetChildren()->GetContent().Strip(wxString::both));
	return true;
}

//////////////////////////////// TextSub ////////////////////////////////////

TextSub::TextSub(wxString filename) {
	m_filename = filename;
	m_characterSet = wxT("ISO-8859-1");
	m_fontFamily = wxT("Arial");
	m_fontStyle = wxT("Regular");
	m_fontSize = 28;
	m_alignment = (wxAlignment) (wxALIGN_CENTER_HORIZONTAL | wxALIGN_BOTTOM);
	m_leftMargin = 60;
	m_rightMargin = 60;
	m_topMargin = 20;
	m_bottomMargin = 30;
	m_subtitleFps = 25;
	m_movieFps = 25;
	m_movieWidth = 720;
	m_movieHeight = 574;
}

wxString TextSub::GetFontFile() {
	return SubtitlePropDlg::GetFontMap()[m_fontFamily][m_fontStyle];
}

wxSvgXmlNode* TextSub::GetXML(DVDFileType type) {
	wxSvgXmlNode* textsubNode = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, _T("textsub"));
	TextSub* defaults = new TextSub(wxT(""));
	textsubNode->AddProperty(_T("filename"), m_filename);
	if (m_characterSet != defaults->m_characterSet)
		textsubNode->AddProperty(_T("characterset"), m_characterSet);
	if (type == DVDAUTHOR_XML) {
		if (GetFontFile().length() > 0)
			textsubNode->AddProperty(wxT("font"), GetFontFile());
	} else {
		if (m_fontFamily != defaults->m_fontFamily)
			textsubNode->AddProperty(wxT("fontFamily"), m_fontFamily);
		if (m_fontStyle != defaults->m_fontStyle)
			textsubNode->AddProperty(wxT("fontStyle"), m_fontStyle);
	}
	if (m_fontSize != defaults->m_fontSize)
		textsubNode->AddProperty(_T("fontsize"), wxString::Format(wxT("%g"), m_fontSize));
	wxString hAlignment = m_alignment & wxALIGN_CENTER_HORIZONTAL ? wxT("center")
			: m_alignment & wxALIGN_RIGHT ? wxT("right") : wxT("left");
	if ((type == DVDAUTHOR_XML && hAlignment != wxT("left"))
			|| (type == DVDSTYLER_XML && hAlignment != wxT("center")))
		textsubNode->AddProperty(_T("horizontal-alignment"), hAlignment);
	wxString vAlignment = m_alignment & wxALIGN_CENTER_VERTICAL ? wxT("center")
				: m_alignment & wxALIGN_BOTTOM ? wxT("bottom") : wxT("top");
	if (vAlignment != wxT("bottom"))
		textsubNode->AddProperty(_T("vertical-alignment"), vAlignment);
	if (m_leftMargin != defaults->m_leftMargin)
		textsubNode->AddProperty(_T("left-margin"), wxString::Format(wxT("%d"), m_leftMargin));
	if (m_rightMargin != defaults->m_rightMargin)
		textsubNode->AddProperty(_T("right-margin"), wxString::Format(wxT("%d"), m_rightMargin));
	if (m_topMargin != defaults->m_topMargin)
		textsubNode->AddProperty(_T("top-margin"), wxString::Format(wxT("%d"), m_topMargin));
	if (m_bottomMargin != defaults->m_bottomMargin)
		textsubNode->AddProperty(_T("bottom-margin"), wxString::Format(wxT("%d"), m_bottomMargin));
	if (m_subtitleFps != defaults->m_subtitleFps)
		textsubNode->AddProperty(_T("subtitle-fps"), wxString::Format(wxT("%g"), m_subtitleFps));
	if (m_movieFps != defaults->m_movieFps)
		textsubNode->AddProperty(_T("movie-fps"), wxString::Format(wxT("%g"), m_movieFps));
	if (m_movieWidth != defaults->m_movieWidth)
		textsubNode->AddProperty(_T("movie-width"), wxString::Format(wxT("%d"), m_movieWidth));
	if (m_movieHeight != defaults->m_movieHeight)
		textsubNode->AddProperty(_T("movie-height"), wxString::Format(wxT("%d"), m_movieHeight));
	delete defaults;
	return textsubNode;
}

bool TextSub::PutXML(wxSvgXmlNode* node) {
	wxString val;
	long lval;
	double dval;

	if (!node->GetPropVal(wxT("filename"), &m_filename))
		return false;

	node->GetPropVal(wxT("characterset"), &m_characterSet);
	node->GetPropVal(wxT("fontFamily"), &m_fontFamily);
	node->GetPropVal(wxT("fontStyle"), &m_fontStyle);
	if (node->GetPropVal(wxT("fontsize"), &val) && val.ToDouble(&dval))
		m_fontSize = dval;
	m_alignment = wxALIGN_CENTER_HORIZONTAL;
	if (node->GetPropVal(wxT("horizontal-alignment"), &val)) {
		if (val == wxT("left"))
			m_alignment = wxALIGN_LEFT;
		else if (val == wxT("right"))
			m_alignment = wxALIGN_RIGHT;
	}
	if (node->GetPropVal(wxT("vertical-alignment"), &val)) {
		if (val == wxT("center"))
			m_alignment = (wxAlignment) (m_alignment | wxALIGN_CENTER_VERTICAL);
		else if (val == wxT("top"))
			m_alignment = (wxAlignment) (m_alignment | wxALIGN_TOP);
		else
			m_alignment = (wxAlignment) (m_alignment | wxALIGN_BOTTOM);
	} else
		m_alignment = (wxAlignment) (m_alignment | wxALIGN_BOTTOM);
	if (node->GetPropVal(wxT("left-margin"), &val) && val.ToLong(&lval))
		m_leftMargin = lval;
	if (node->GetPropVal(wxT("right-margin"), &val) && val.ToLong(&lval))
		m_rightMargin = lval;
	if (node->GetPropVal(wxT("top-margin"), &val) && val.ToLong(&lval))
		m_topMargin = lval;
	if (node->GetPropVal(wxT("bottom-margin"), &val) && val.ToLong(&lval))
		m_bottomMargin = lval;
	if (node->GetPropVal(wxT("subtitle-fps"), &val) && val.ToDouble(&dval))
		m_subtitleFps = dval;
	if (node->GetPropVal(wxT("movie-fps"), &val) && val.ToDouble(&dval))
		m_movieFps = dval;
	if (node->GetPropVal(wxT("movie-width"), &val) && val.ToLong(&lval))
		m_movieWidth = lval;
	if (node->GetPropVal(wxT("movie-height"), &val) && val.ToLong(&lval))
		m_movieHeight = lval;

	return true;
}

bool TextSub::SaveSpumux(wxString filename) {
	wxSvgXmlDocument xml;
	wxSvgXmlNode* root = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("subpictures"));
	wxSvgXmlNode* streamNode = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("stream"));
	streamNode->AddChild(GetXML(DVDAUTHOR_XML));
	root->AddChild(streamNode);
	xml.SetRoot(root);
	return xml.Save(filename);
}

//////////////////////////////// Stream /////////////////////////////////////

Stream::Stream(StreamType type, wxString codecName) {
	m_type = type;
	m_sourceCodecName = codecName;
	m_destinationFormat = 1; // vfCOPY/afCOPY/sfCOPY
	m_sourceChannelNumber = -1;
	m_sourceAspectRatio = -1;
}

VideoFormat Stream::GetSourceVideoFormat() {
	if (m_sourceVideoSize.GetWidth() == 720 && m_sourceVideoSize.GetHeight() == 576)
		return vfPAL;
	else if (m_sourceVideoSize.GetWidth() == 720 && m_sourceVideoSize.GetHeight() == 480)
		return vfNTSC;
	return vfNONE;
}

AudioFormat Stream::GetSourceAudioFormat() {
	if (m_sourceCodecName == wxT("mp2"))
		return afMP2;
	else if (m_sourceCodecName == wxT("liba52") || m_sourceCodecName == wxT("ac3")
			|| m_sourceCodecName == wxT("ac-3"))
		return afAC3;
	return afNONE;
}

wxString Stream::GetSourceFormat() {
	wxString result = m_sourceCodecName;
	if (result == wxT("liba52") || m_sourceCodecName == wxT("ac-3"))
		result = wxT("ac3");
	else if (result == wxT("mpeg2video"))
		result = wxT("mpeg2");
	switch (m_type) {
	case stVIDEO:
		if (m_sourceVideoSize.IsFullySpecified()) {
			result += wxString::Format(wxT(", %dx%d"),
					m_sourceVideoSize.GetWidth(), m_sourceVideoSize.GetHeight());
			if (GetSourceVideoFormat() == vfPAL)
				result += wxT(" (PAL)");
			else if (GetSourceVideoFormat() == vfNTSC)
				result += wxT(" (NTSC)");
			if (round(GetSourceAspectRatio()*100) == 133)
				result += wxT(", 4:3");
			else if (round(GetSourceAspectRatio()*100) == 155)
				result += wxT(", 16:9");
		}
		break;
	case stAUDIO:
		if (m_sourceChannelNumber > 0) {
			result += wxT(", ");
			if (m_sourceChannelNumber == 1)
				result += _("mono");
			else if (m_sourceChannelNumber == 2)
				result += _("stereo");
			else if (m_sourceChannelNumber == 6)
				result += wxT("5:1");
			else
				result += wxString::Format(_("%d channels"), m_sourceChannelNumber);
		}
		if (m_sourceSampleRate)
			result += wxT(", ") + wxString::Format(_("%d Hz"), m_sourceSampleRate);
		break;
	default:
		break;
	}
	return result;
}

////////////////////////////////// Vob //////////////////////////////////////
Vob::Vob() {
	Init();
}

Vob::Vob(wxString filename) {
	Init();
	SetFilename(filename);
}

Vob::Vob(Menu* menu) {
	Init(menu);
}

Vob::Vob(Slideshow* slideshow) {
	Init(NULL, slideshow);
}

void Vob::Init(Menu* menu, Slideshow* slideshow) {
	m_pause = 0;
	m_menu = menu;
	m_slideshow = slideshow;
	m_doNotTranscode = false;
	m_interlaced = true;
	m_firstField = ffAUTO;
}

Vob::Vob(const Vob& vob): m_pad(vob.m_pad), m_crop(vob.m_crop) {
	m_filename = vob.m_filename;
	m_audioFilenames = vob.m_audioFilenames;
	for (unsigned int i = 0; i<vob.m_subtitles.size(); i++)
		m_subtitles.Add(new TextSub(*vob.m_subtitles[i]));
	m_tmpFilename = vob.m_tmpFilename;
	m_chapters = vob.m_chapters;
	m_pause = vob.m_pause;
	m_menu = vob.m_menu;
	m_slideshow = vob.m_slideshow;
	m_duration = vob.m_duration;
	for (unsigned int i = 0; i<vob.m_streams.size(); i++)
		m_streams.Add(new Stream(*vob.m_streams[i]));
	VECTOR_COPY(vob.m_cells, m_cells, Cell);
	m_interlaced = vob.m_interlaced;
	m_firstField = vob.m_firstField;
	m_doNotTranscode = vob.m_doNotTranscode;
}

Vob::~Vob() {
	if (m_menu)
		delete m_menu;
	if (m_slideshow)
		delete m_slideshow;
	WX_CLEAR_ARRAY(m_subtitles)
	WX_CLEAR_ARRAY(m_streams)
	VECTOR_CLEAR(m_cells, Cell)
}

bool Vob::SetFilename(wxString value) {
	m_filename = value;
	wxFfmpegMediaDecoder ffmpeg;
	bool ok = m_filename.length() && ffmpeg.Load(m_filename);
	m_duration = ok && ffmpeg.GetDuration() > 0 ? ffmpeg.GetDuration() : 0;
	if (m_streams.GetCount() - m_audioFilenames.GetCount() > 0)
		m_streams.RemoveAt(0, m_streams.GetCount() - m_audioFilenames.GetCount());
	for (unsigned int stIdx = 0; stIdx < ffmpeg.GetStreamCount(); stIdx++) {
		Stream* stream = new Stream(ffmpeg.GetStreamType(stIdx), ffmpeg.GetCodecName(stIdx));
		if (stream->GetType() == stAUDIO) {
			stream->SetSourceChannelNumber(ffmpeg.GetChannelNumber(stIdx));
			stream->SetSourceSampleRate(ffmpeg.GetSampleRate(stIdx));
			stream->SetSourceBitrate(ffmpeg.GetBitrate(stIdx));
		} else if (stream->GetType() == stVIDEO) {
			stream->SetSourceVideoSize(ffmpeg.GetVideoSize());
			stream->SetSourceBitrate(ffmpeg.GetBitrate(stIdx));
			stream->SetSourceAspectRatio(ffmpeg.GetFrameAspectRatio());
		} else
			continue;
		m_streams.Insert(stream, stIdx);
	}
	return ok;
}

bool Vob::HasAudio() {
	if (GetAudioFilenames().GetCount() > 0)
		return true;
	for (int i = 0; i < (int)GetStreams().GetCount(); i++) {
		if (GetStreams()[i]->GetType() == stAUDIO)
			return true;
	}
	return false;
}


unsigned int Vob::GetAudioStreamCount() {
	unsigned int cnt = 0;
	for (StreamArray::iterator it = m_streams.begin(); it != m_streams.end(); it++)
		if ((*it)->GetType() == stAUDIO)
			cnt++;
	return cnt;
}

unsigned int Vob::GetSubtitleStreamsCount() {
	return m_menu ? 1 : GetSubtitles().GetCount();
}

bool Vob::AddAudioFile(wxString filename) {
	wxFfmpegMediaDecoder ffmpeg;
	if (!ffmpeg.Load(filename))
		return false;
	m_audioFilenames.Add(filename);
	if (ffmpeg.GetStreamCount() > 0) {
		Stream* stream = new Stream(ffmpeg.GetStreamType(0), ffmpeg.GetCodecName(0));
		stream->SetSourceChannelNumber(ffmpeg.GetChannelNumber(0));
		stream->SetSourceSampleRate(ffmpeg.GetSampleRate(0));
		m_streams.Add(stream);
	} else
		m_streams.Add(new Stream(stAUDIO, wxT("unknown")));
	if (GetFilename().length() == 0) // menu or slideshow
		m_duration = ffmpeg.GetDuration();
	return true;
}

void Vob::RemoveAudioFile(int index) {
	m_streams.RemoveAt(m_streams.GetCount() - m_audioFilenames.GetCount() + index);
	m_audioFilenames.RemoveAt(index);
}

long Vob::GetChapterTime(int chapter) {
	long time = 0;
	if (chapter > 0) {
		wxString timeStr = m_chapters;
		while (timeStr.Find(wxT(',')) != wxNOT_FOUND && --chapter)
			timeStr = timeStr.AfterFirst(wxT(','));
		timeStr = timeStr.BeforeFirst(wxT(','));
		while (1) {
			time *= 60;
			if (timeStr.Find(wxT(':')) == wxNOT_FOUND) {
				time *= 1000;
				double t = 0;
				if (timeStr.ToDouble(&t))
					time += (int) (t*1000);
				break;
			} else {
				long t = 0;
				if (timeStr.BeforeFirst(wxT(':')).ToLong(&t))
					time += t;
				timeStr = timeStr.AfterFirst(wxT(':'));
			}
		}
	}
	return time;
}

wxString Vob::GetVideoFrameUri(int chapter, long position) {
	wxString uri = GetFilename();
	if (position == -1 && chapter > 0)
		position = GetChapterTime(chapter);
	if (position >= 0 && uri.length() > 0)
		uri += wxT('#') + wxString::Format(wxT("%d"), position);
	return uri;
}

wxSvgXmlNode* Vob::GetXML(DVDFileType type, DVD* dvd) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("vob"));
	wxString fname = GetFilename();
	if (type == DVDAUTHOR_XML) {
		if (GetTmpFilename().length())
			fname = GetTmpFilename();
	} else {
		wxFileName filename(fname);
		if (filename.GetPath() == dvd->GetPath(false))
			fname = filename.GetFullName();
		else if (dvd->GetPath(false).length() > 0 && filename.GetPath().StartsWith(dvd->GetPath(false)))
			fname = filename.GetPath().substr(dvd->GetPath(false).length() + 1) + wxFILE_SEP_PATH
					+  filename.GetFullName();
	}
	if (fname.length())
		node->AddProperty(wxT("file"), fname);
	
	if (type == DVDSTYLER_XML) {
		int stIdx = 0;
		for (; stIdx < (int)(GetStreams().GetCount() - GetAudioFilenames().GetCount()); stIdx++) {
			Stream* stream = GetStreams()[stIdx];
			wxString streamName;
			switch (stream->GetType()) {
			case stVIDEO:
				streamName = wxT("video");
				break;
			case stAUDIO:
				streamName = wxT("audio");
				break;
			case stSUBTITLE:
				streamName = wxT("subtitle");
				break;
			default:
				break;
			}
			if (streamName.length() == 0)
				continue;
			wxSvgXmlNode* streamNode = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, streamName);
			streamNode->AddProperty(wxT("format"), wxString::Format(wxT("%d"), stream->GetDestinationFormat()));
			if (stream->GetType() == stVIDEO && GetInterlaced()) {
				streamNode->AddProperty(wxT("interlaced"), wxT("1"));
				streamNode->AddProperty(wxT("firstField"), wxString::Format(wxT("%d"), GetFirstField()));
			}
			node->AddChild(streamNode);
		}
		for (int i = 0; i < (int)GetAudioFilenames().GetCount(); i++) {
			Stream* stream = GetStreams()[stIdx++];
			wxSvgXmlNode* audioNode = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("audio"));
			audioNode->AddChild(new wxSvgXmlNode(wxSVGXML_TEXT_NODE, wxEmptyString,
					GetAudioFilenames()[i]));
			audioNode->AddProperty(wxT("format"),
					wxString::Format(wxT("%d"), stream->GetDestinationFormat()));
			node->AddChild(audioNode);
		}
		for (int i = 0;  i < (int) GetSubtitles().GetCount(); i++) {
			node->AddChild(GetSubtitles()[i]->GetXML(type));
		}
		if (GetDoNotTranscode())
			node->AddProperty(wxT("doNotTranscode"), wxT("1"));
		if (m_pad.size() == 4 && m_pad[0] + m_pad[1] + m_pad[2] + m_pad[3] > 0) {
			wxSvgXmlNode* padNode = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("pad"));
			if (m_pad[0] > 0)
				padNode->AddProperty(wxT("left"), wxString::Format(wxT("%d"), m_pad[0]));
			if (m_pad[1] > 0)
				padNode->AddProperty(wxT("right"), wxString::Format(wxT("%d"), m_pad[1]));
			if (m_pad[2] > 0)
				padNode->AddProperty(wxT("top"), wxString::Format(wxT("%d"), m_pad[2]));
			if (m_pad[3] > 0)
				padNode->AddProperty(wxT("bottom"), wxString::Format(wxT("%d"), m_pad[3]));
			node->AddChild(padNode);
		}
		if (m_crop.size() == 4 && m_crop[0] + m_crop[1] + m_crop[2] + m_crop[3] > 0) {
			wxSvgXmlNode* cropNode = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("crop"));
			if (m_crop[0] > 0)
				cropNode->AddProperty(wxT("left"), wxString::Format(wxT("%d"), m_crop[0]));
			if (m_crop[1] > 0)
				cropNode->AddProperty(wxT("right"), wxString::Format(wxT("%d"), m_crop[1]));
			if (m_crop[2] > 0)
				cropNode->AddProperty(wxT("top"), wxString::Format(wxT("%d"), m_crop[2]));
			if (m_crop[3] > 0)
				cropNode->AddProperty(wxT("bottom"), wxString::Format(wxT("%d"), m_crop[3]));
			node->AddChild(cropNode);
		}
	}
	
	if (GetPause() != 0) {
		wxString pause= wxT("inf");
		if (GetPause()>0)
			pause = wxString::Format(wxT("%d"), GetPause());
		node->AddProperty(wxT("pause"), pause);
	}
	
	if (GetChapters().length() && !GetSlideshow())
		node->AddProperty(wxT("chapters"), GetChapters());
	
	if (m_cells.size() > 0) {
		for (vector<Cell*>::iterator it = m_cells.begin(); it != m_cells.end(); it++)
			node->AddChild((*it)->GetXML(type));
	}
	
	if (GetMenu() && type == DVDSTYLER_XML)
		node->AddChild(GetMenu()->GetXML(type));
	
	if (GetSlideshow()) {
		if (type == DVDSTYLER_XML)
			node->AddChild(GetSlideshow()->GetXML(type));
		else if (type == DVDAUTHOR_XML) {
			wxString chapters;
			int t = 1;
			for (unsigned i=1; i<GetSlideshow()->Count()/5; i++) {
				t += GetSlideshow()->GetDuration()*5;
				int h = t/3600;
				int m = (t%3600)/60;
				int s = t%60;
				if (chapters.length())
					chapters += wxT(",");
				chapters += wxString::Format(wxT("%d:%2d:%2d.1"), h, m, s);
			}
			node->AddProperty(wxT("chapters"), chapters);
		}
	}
	
	return node;
}

bool Vob::PutXML(wxSvgXmlNode* node, DVD* dvd, int tsi, int pgci) {
	wxString val;
	long lval;

	node->GetPropVal(wxT("file"), &val);
	if (val.length() > 0) {
		wxFileName fname(val);
		if (fname.IsRelative())
			val = dvd->GetPath() + fname.GetFullPath();
		else if (!wxFileExists(val) && wxFileExists(dvd->GetPath() + fname.GetFullName()))
			val = dvd->GetPath() + fname.GetFullName();
		SetFilename(val);
	}
	int stIdx = 0;
	wxSvgXmlNode* child = node->GetChildren();
	while (child) {
		if (child->GetName() == wxT("video")) {
			if (child->GetChildren() != NULL && child->GetChildren()->GetContent().length() > 0) {
				val = child->GetChildren()->GetContent();
				wxFileName fname(val);
				if (fname.IsRelative())
					val = dvd->GetPath() + fname.GetFullPath();
				else if (!wxFileExists(val) && wxFileExists(dvd->GetPath() + fname.GetFullName()))
					val = dvd->GetPath() + fname.GetFullName();
				SetFilename(val);
			}
			
			if (child->GetPropVal(wxT("format"), &val) && val.length() > 0 && val.ToLong(&lval)) {
				if ((int)m_streams.GetCount() <= stIdx || m_streams[stIdx]->GetType() != stVIDEO)
					stIdx = 0;
				while ((int)m_streams.GetCount() > stIdx) {
					if (m_streams[stIdx]->GetType() == stVIDEO) {
						m_streams[stIdx++]->SetDestinationFormat(lval);
						break;
					}
					stIdx++;
				}
			}
			if (child->GetPropVal(wxT("interlaced"), &val) && val == wxT("1")) {
				m_interlaced = true;
				if (child->GetPropVal(wxT("firstField"), &val) && val.length() > 0 && val.ToLong(&lval))
					m_firstField = (FirstField) lval;
			}
		} else if (child->GetName() == wxT("audio")) {
			if (child->GetChildren() != NULL && child->GetChildren()->GetContent().length() > 0) {
				val = child->GetChildren()->GetContent();
				wxFileName fname(val);
				if (fname.IsRelative())
					val = dvd->GetPath() + fname.GetFullPath();
				else if (!wxFileExists(val) && wxFileExists(dvd->GetPath() + fname.GetFullName()))
					val = dvd->GetPath() + fname.GetFullName();
				if (AddAudioFile(val) && child->GetPropVal(wxT("format"), &val) && val.length() > 0 && val.ToLong(&lval)) {
					stIdx = m_streams.GetCount() - 1;
					m_streams[stIdx++]->SetDestinationFormat(lval);
				}
			} else if (child->GetPropVal(wxT("format"), &val) && val.length() > 0 && val.ToLong(&lval)) {
				if ((int)m_streams.GetCount() <= stIdx || m_streams[stIdx]->GetType() != stAUDIO)
					stIdx = 0;
				while ((int)m_streams.GetCount() > stIdx) {
					if (m_streams[stIdx]->GetType() == stAUDIO) {
						m_streams[stIdx++]->SetDestinationFormat(lval);
						break;
					}
					stIdx++;
				}
			}
		} else if (child->GetName() == wxT("cell")) {
			Cell* cell = new Cell;
			cell->PutXML(child);
			m_cells.push_back(cell);
		} else if (child->GetName() == wxT("pad")) {
			m_pad.push_back(child->GetPropVal(wxT("left"), &val) && val.ToLong(&lval) ? lval : 0);
			m_pad.push_back(child->GetPropVal(wxT("right"), &val) && val.ToLong(&lval) ? lval : 0);
			m_pad.push_back(child->GetPropVal(wxT("top"), &val) && val.ToLong(&lval) ? lval : 0);
			m_pad.push_back(child->GetPropVal(wxT("bottom"), &val) && val.ToLong(&lval) ? lval : 0);
		} else if (child->GetName() == wxT("crop")) {
			m_crop.push_back(child->GetPropVal(wxT("left"), &val) && val.ToLong(&lval) ? lval : 0);
			m_crop.push_back(child->GetPropVal(wxT("right"), &val) && val.ToLong(&lval) ? lval : 0);
			m_crop.push_back(child->GetPropVal(wxT("top"), &val) && val.ToLong(&lval) ? lval : 0);
			m_crop.push_back(child->GetPropVal(wxT("bottom"), &val) && val.ToLong(&lval) ? lval : 0);
		} else if (child->GetName() == wxT("textsub")) {
			TextSub* textsub = new TextSub;
			textsub->PutXML(child);
			GetSubtitles().Add(textsub);
		} else if (child->GetName() == wxT("menu")) {
			m_menu = new Menu(dvd, tsi, pgci);
			if (!m_menu->PutXML(child))
				return false;
		} else if (child->GetName() == wxT("slideshow"))
			m_slideshow = new Slideshow(child);
		child = child->GetNext();
	}
	if (node->GetPropVal(wxT("pause"), &val)) {
		if (val == wxT("inf"))
			m_pause = -1;
		else if (val.ToLong(&lval))
			m_pause = int(lval);
	}
	if (node->GetPropVal(wxT("doNotTranscode"), &val) && val == wxT("1"))
		SetDoNotTranscode(true);
	node->GetPropVal(wxT("chapters"), &m_chapters);
	return true;
}

unsigned int Vob::GetFileSize(wxString filename) {
	return wxFile(filename).Length()/1024;
}

int Vob::GetSize(DVD* dvd, bool generated) {
	if (generated) {
		if (GetTmpFilename().length())
			return GetFileSize(GetTmpFilename());
		else if (GetFilename().length())
			return GetFileSize(GetFilename());
		return 0;
	}

	long size = 0;
	int stIdx = 0;
	if (GetMenu()) {
		double duration = m_duration;
		if (duration < 1)
			duration = s_config.GetMenuFrameCount()/(GetMenu()->GetVideoFormat() == vfPAL ? 25 : 30);
		size += (long)(duration*s_config.GetMenuVideoBitrate()/8/3);
	} else if (GetSlideshow()) {
		size += GetSlideshow()->GetDuration()*GetSlideshow()->Count()*s_config.GetSlideshowVideoBitrate()/8/3;
	} else {
		if (m_doNotTranscode) {
			size += GetFileSize(GetFilename());
		} else {
			if (GetFilename().length()) {
				int streamsCount = (int)m_streams.GetCount() - m_audioFilenames.GetCount();
				bool copyVideo = false;
				long audioSize = 0;
				for (; stIdx < streamsCount; stIdx++) {
					Stream* stream = m_streams[stIdx];
					switch (stream->GetType()) {
					case stVIDEO:
						if (stream->GetVideoFormat() == vfCOPY)
							copyVideo = true;
						else
							size += (long) (GetDuration()*dvd->GetVideoBitrate()/8);
						break;
					case stAUDIO: {
						int srcBitrate = stream->GetSourceBitrate() > 0 ? stream->GetSourceBitrate() : 64000;
						audioSize += (long) (GetDuration()*srcBitrate/8/1024);
						if (stream->GetAudioFormat() == afCOPY)
							size += (long) (GetDuration()*srcBitrate/8/1024);
						else if (stream->GetAudioFormat() != afNONE)
							size += (long) (GetDuration()*s_config.GetAudioBitrate()/8);
						break;
					}
					case stSUBTITLE:
					case stUNKNOWN:
						break;
					}
				}
				if (copyVideo)
					size += GetFileSize(GetFilename()) - audioSize;
			}
		}
	}
	for (unsigned int i=0; i<GetAudioFilenames().Count(); i++) {
		Stream* stream = m_streams[stIdx++];
		size += GetFileSize(GetAudioFilenames()[i]);
		if (stream->GetAudioFormat() == afCOPY)
			size += GetFileSize(GetAudioFilenames()[i]);
		else
			size += (long) (GetDuration()*s_config.GetAudioBitrate()/8);
	}
	for (unsigned int i=0; i<GetSubtitles().Count(); i++)
		size += GetFileSize(GetSubtitles()[i]->GetFilename());
	return size;
}

int Vob::GetRequiredSize(DVD* dvd, Cache* cache) {
	int size = GetSize(dvd);
	if (GetMenu() || GetSlideshow() || GetSubtitles().Count() > 0
			 || ((!GetDoNotTranscode() || GetAudioFilenames().Count() > 0)
					 && cache->Find(this, dvd, false).length() == 0))
		size = size * 2;
	return size;
}

Vob* Vob::CreateEmptyVob(VideoFormat videoFormat, AudioFormat audioFormat) {
	wxString filename= wxT("empty_pal_mp2.mpg");
	if (videoFormat == vfNTSC) {
		filename = audioFormat == afMP2 ? wxT("empty_ntsc_mp2.mpg") : wxT("empty_ntsc_ac3.mpg");
	} else if (audioFormat != afMP2) {
		filename = wxT("empty_pal_ac3.mpg");
	}
	return new Vob(DATA_FILE(filename));
}

////////////////////////////////// Pgc //////////////////////////////////////

wxString Pgc::GetEntriesAsString() {
	wxString result;
	for (StringSet::const_iterator entry = m_entries.begin(); entry != m_entries.end(); entry++) {
		if (result.length())
			result += wxT(",");
		result += *entry;
	}
	return result;
}

void Pgc::RemoveEntries(const StringSet& entries) {
	for (StringSet::const_iterator entry = entries.begin(); entry != entries.end(); entry++) {
		m_entries.erase(*entry);
	}
}

Menu* Pgc::GetMenu() {
	for (int i = 0; i < (int) GetVobs().GetCount(); i++)
		if (GetVobs()[i]->GetMenu())
			return GetVobs()[i]->GetMenu();
	return NULL;
}

Slideshow* Pgc::GetSlideshow() {
	for (int i = 0; i < (int) GetVobs().GetCount(); i++)
		if (GetVobs()[i]->GetSlideshow())
			return GetVobs()[i]->GetSlideshow();
	return NULL;
}

unsigned int Pgc::GetChapterCount() {
	int result = 1;
	for (int vi = 0; vi < (int) GetVobs().Count(); vi++) {
		Vob& vob = *GetVobs()[vi];
		if (vob.GetSlideshow()) {
			result += vob.GetSlideshow()->Count() / 5;
		} else {
			wxStringTokenizer tkz(vob.GetChapters(), _T(","));
			while (tkz.HasMoreTokens()) {
				wxString token = tkz.GetNextToken();
				if (token != _T("0") || vi > 0)
					result++;
			}
		}
	}
	return result;
}

wxSvgXmlNode* Pgc::GetXML(DVDFileType type, DVD* dvd, wxString nextTitle) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("pgc"));
	if (m_entries.size() > 0)
		node->AddProperty(wxT("entry"), GetEntriesAsString());
	
	if (m_palette.length())
		node->AddProperty(wxT("palette"), m_palette);
	
	if (GetMenu() && type == DVDAUTHOR_XML)
		GetMenu()->GetXML(type, node);
	
	if (GetSlideshow() && type == DVDAUTHOR_XML)
		GetSlideshow()->GetXML(type, node);
	
	for (int i = 0; i < (int) GetVobs().GetCount(); i++)
		node->AddChild(GetVobs()[i]->GetXML(type, dvd));
	
	wxString preCommands = GetPreCommands();
	if (type == DVDAUTHOR_XML && dvd->GetRememberLastButtonRegister() != -1) {
		wxString g = wxT("g") + wxString::Format(wxT("%d"), dvd->GetRememberLastButtonRegister());
		preCommands = wxT("if (") + g + wxT(">0) button = ") + g + wxT(";") + preCommands;
	}
	if (preCommands.length())
		XmlWriteValue(node, wxT("pre"), preCommands);
	
	wxString postCommands = GetPostCommands();
	if (type == DVDAUTHOR_XML && dvd->GetPlayAllRegister() != -1 && nextTitle.length()) {
		wxString g = wxT("g") + wxString::Format(wxT("%d"), dvd->GetPlayAllRegister());
		postCommands = wxT("if (") + g + wxT("==1) jump ") + nextTitle + wxT(";") + postCommands;
	}
	if (postCommands.length())
		XmlWriteValue(node, wxT("post"), postCommands);
	
	return node;
}

bool Pgc::PutXML(wxSvgXmlNode* node, DVD* dvd, int tsi, int pgci) {
	WX_CLEAR_ARRAY(GetVobs());

	wxString val;
	m_entries.clear();
	if (node->GetPropVal(wxT("entry"), &val)) {
		wxArrayString entryArray = wxStringTokenize(val, wxT(","));
		for (unsigned int entryIdx = 0; entryIdx < entryArray.Count(); entryIdx++)
			m_entries.insert(entryArray[entryIdx]);
	}

	node->GetPropVal(wxT("palette"), &m_palette);

	wxSvgXmlNode* child = node->GetChildren();
	Menu* menu = NULL; // old

	while (child) {
		if (child->GetName() == wxT("spumux")) {
			// get spunode (old)
			menu = new Menu(dvd, tsi, pgci);
			menu->PutXML(child);
		} else if (child->GetName() == wxT("vob")) {
			// get vob
			Vob* vob;
			if (menu)
				vob = new Vob(menu);
			else
				vob = new Vob(wxT(""));
			menu = NULL;
			if (vob->PutXML(child, dvd, tsi, pgci))
				GetVobs().Add(vob);
			else {
				delete vob;
				wxLogError(_("Can't load vob"));
				return false;
			}
		} else if (child->GetName() == wxT("pre")) // get pre commands
			SetPreCommands(child->GetChildren()->GetContent());
		else if (child->GetName() == wxT("post")) // get post commands
			SetPostCommands(child->GetChildren()->GetContent());
		child = child->GetNext();
	}
	return true;
}

////////////////////////////// Video ///////////////////////////////////
wxArrayString Video::GetFormatStrings(wxString def) {
	wxArrayString ret;
	ret.Add(def);
	ret.Add(wxT("pal"));
	ret.Add(wxT("ntsc"));
	return ret;
}

wxArrayString Video::GetAspectStrings(wxString def) {
	wxArrayString ret;
	ret.Add(def);
	ret.Add(wxT("4:3"));
	ret.Add(wxT("16:9"));
	return ret;
}

wxArrayString Video::GetWidescreenStrings(wxString def) {
	wxArrayString ret;
	ret.Add(def);
	ret.Add(wxT("nopanscan"));
	ret.Add(wxT("noletterbox"));
	return ret;
}

wxSvgXmlNode* Video::GetXML(DVDFileType type, bool menu) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("video"));
	
	if (!menu || type == DVDAUTHOR_XML) {
		if (m_format.length())
			node->AddProperty(wxT("format"), m_format);
		if (m_aspect != arAUTO)
			node->AddProperty(wxT("aspect"), GetAspectStrings()[m_aspect]);
		if (m_resolution.length())
			node->AddProperty(wxT("resolution"), m_resolution);
	}
	if (m_caption.length())
		node->AddProperty(wxT("caption"), m_caption);
	if (m_widescreen != wtAUTO && (m_aspect == ar16_9 || type == DVDSTYLER_XML))
		node->AddProperty(wxT("widescreen"), GetWidescreenStrings()[m_widescreen]);

	if (node->GetProperties() == NULL) {
		delete node;
		node = NULL;
	}
	return node;
}

bool Video::PutXML(wxSvgXmlNode* node, bool menu) {
	if (node == NULL || node->GetName() != wxT("video"))
		return false;

	wxString val;
	if (!menu) {
		node->GetPropVal(wxT("format"), &m_format);
		if (node->GetPropVal(wxT("aspect"), &val) && GetAspectStrings().Index(val) > 0)
			m_aspect = (AspectRatio) GetAspectStrings().Index(val);
		node->GetPropVal(wxT("resolution"), &m_resolution);
	}
	node->GetPropVal(wxT("caption"), &m_caption);
	if (node->GetPropVal(wxT("widescreen"), &val) && GetWidescreenStrings().Index(val) > 0)
		m_widescreen = (WidescreenType) GetWidescreenStrings().Index(val);

	return true;
}

////////////////////////////////// SubStream //////////////////////////////////////

wxString s_subModeArray[4] = { wxT("normal"), wxT("widescreen"), wxT("letterbox"), wxT("panscan") };
wxArrayString s_subModes(4, s_subModeArray);

wxString s_subConentArray[10] = { wxT("normal"), wxT("large"), wxT("children"), wxT("normal_cc"), wxT("large_cc"),
		wxT("children_cc"), wxT("forced"), wxT("director"), wxT("large_director"), wxT("children_director") };
wxArrayString s_subConents(10, s_subConentArray);

wxSvgXmlNode* SubStream::GetXML(DVDFileType type) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("stream"));
	if (m_id.length())
		node->AddProperty(wxT("id"), m_id);
	if (m_mode != ssmNORMAL)
		node->AddProperty(wxT("mode"), s_subModes[m_mode]);
	if (m_content != sscNORMAL)
		node->AddProperty(wxT("content"), s_subConents[m_content]);
	return node;
}

bool SubStream::PutXML(wxSvgXmlNode* node) {
	if (node == NULL || node->GetName() != wxT("stream"))
		return false;

	wxString val;
	m_id = node->GetPropVal(wxT("id"), wxT(""));
	int ival = node->GetPropVal(wxT("mode"), &val) && s_subModes.Index(val) > 0 ? s_subModes.Index(val) : 0;
	m_mode = (SubStreamMode) ival;
	ival = node->GetPropVal(wxT("content"), &val) && s_subConents.Index(val) > 0 ? s_subConents.Index(val) : 0;
	m_content = (SubStreamContent) ival;

	return true;
}

////////////////////////////////// SubPicture //////////////////////////////////////

wxSvgXmlNode* SubPicture::GetXML(DVDFileType type, Video& video, bool menu) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("subpicture"));
	
	node->AddProperty(wxT("lang"), m_langCode);
	for (vector<SubStream*>::iterator it = m_streams.begin(); it != m_streams.end(); it++)
		node->AddChild((*it)->GetXML(type));
	if (type == DVDAUTHOR_XML && menu && m_streams.size() == 0 && video.GetAspect() == ar16_9) {
		node->AddChild(SubStream(wxT("0"), ssmWIDESCREEN).GetXML(type));
		if (video.GetWidescreen() == wtNOLETTERBOX)
			node->AddChild(SubStream(wxT("1"), ssmPANSCAN).GetXML(type));
		else
			node->AddChild(SubStream(wxT("1"), ssmLETTERBOX).GetXML(type));
		if (video.GetWidescreen() == wtAUTO)
			node->AddChild(SubStream(wxT("2"), ssmPANSCAN).GetXML(type));
	}
	
	return node;
}

bool SubPicture::PutXML(wxSvgXmlNode* node) {
	if (node == NULL || node->GetName() != wxT("subpicture"))
		return false;
	
	wxString val;
	if (node->GetPropVal(wxT("lang"), &val))
		m_langCode = val;
	
	for (wxSvgXmlNode* child = node->GetChildren(); child != NULL; child = child->GetNext()) {
		SubStream* subStream = new SubStream();
		if (subStream->PutXML(child))
			m_streams.push_back(subStream);
		else
			delete subStream;
	}
	
	return true;
}

////////////////////////////////// PgcArray //////////////////////////////////////

unsigned int PgcArray::GetAudioStreamCount() {
	unsigned int count = 1; // at least one audio stream
	for (iterator it = begin(); it != end(); it++)
		for (VobArray::iterator it2 = (*it)->GetVobs().begin(); it2 != (*it)->GetVobs().end(); it2++)
			if (count < (*it2)->GetAudioStreamCount())
				count = (*it2)->GetAudioStreamCount();
	return count;
}

unsigned int PgcArray::GetSubtitleStreamsCount() {
	unsigned int count = 0;
	for (iterator it = begin(); it != end(); it++)
		for (VobArray::iterator it2 = (*it)->GetVobs().begin(); it2 != (*it)->GetVobs().end(); it2++)
			if (count < (*it2)->GetSubtitleStreamsCount())
				count = (*it2)->GetSubtitleStreamsCount();
	return count;
}

void PgcArray::UpdateAudioLangCodes() {
	unsigned int audioCnt = GetAudioStreamCount();
	while (m_audioLangCodes.size() < audioCnt)
		m_audioLangCodes.push_back(s_config.GetDefAudioLanguage());
	while (m_audioLangCodes.size() > audioCnt)
			m_audioLangCodes.pop_back();
}

void PgcArray::UpdateSubtitleLangCodes() {
	unsigned int subtitleCnt = GetSubtitleStreamsCount();
	while (m_subPictures.size() < subtitleCnt)
		m_subPictures.push_back(new SubPicture(s_config.GetDefSubtitleLanguage()));
	while (m_subPictures.size() > subtitleCnt)
		m_subPictures.pop_back();
}

/** Sets video format */
void PgcArray::SetVideoFormat(VideoFormat videoFormat) {
	for (unsigned int pgci=0; pgci<GetCount(); pgci++) {
		Menu* menu = (*this)[pgci]->GetMenu();
		if (menu != NULL)
			menu->SetVideoFormat(videoFormat);
	}
}

/** Sets aspect ratio */
void PgcArray::SetAspectRatio(AspectRatio aspectRatio) {
	m_video.SetAspect(aspectRatio);
	for (unsigned int pgci=0; pgci<GetCount(); pgci++) {
		Menu* menu = (*this)[pgci]->GetMenu();
		if (menu != NULL)
			menu->SetAspectRatio(aspectRatio);
	}
}

////////////////////////////////// Menus //////////////////////////////////////
StringSet Menus::GetEntries() {
	StringSet entries;
	for (unsigned int pgci = 0; pgci < Count(); pgci++) {
		const StringSet& pgcEntries = Item(pgci)->GetEntries();
		for (StringSet::const_iterator entry = pgcEntries.begin(); entry != pgcEntries.end(); entry++) {
			entries.insert(*entry);
		}
	}
	return entries;
}

int Menus::GetPgciByEntry(wxString entry) {
	for (unsigned int pgci = 0; pgci < Count(); pgci++) {
		const StringSet& pgcEntries = Item(pgci)->GetEntries();
		for (StringSet::const_iterator e = pgcEntries.begin(); e != pgcEntries.end(); e++) {
			if (*e == entry)
				return pgci;
		}
	}
	return -1;
}

wxSvgXmlNode* Menus::GetXML(DVDFileType type, DVD* dvd) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("menus"));
	wxSvgXmlNode* videoNode = GetVideo().GetXML(type, true);
	if (videoNode)
		node->AddChild(videoNode);
	UpdateAudioLangCodes();
	for (wxArrayString::iterator it = GetAudioLangCodes().begin(); it != GetAudioLangCodes().end(); it++) {
		wxSvgXmlNode* audio = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("audio"));
		audio->AddProperty(wxT("lang"), *it);
		node->AddChild(audio);
	}
	UpdateSubtitleLangCodes();
	for (vector<SubPicture*>::iterator it = GetSubPictures().begin(); it != GetSubPictures().end(); it++) {
		node->AddChild((*it)->GetXML(type, m_video, true));
	}
	for (int i = 0; i < (int) GetCount(); i++)
		node->AddChild((*this)[i]->GetXML(type, dvd, wxT("")));
	return node;
}

bool Menus::PutXML(wxSvgXmlNode* node, DVD* dvd, int tsi) {
	if (node == NULL || node->GetName() != wxT("menus"))
		return false;
	WX_CLEAR_ARRAY(*this);
	wxSvgXmlNode* child = node->GetChildren();
	while (child) {
		if (child->GetName() == wxT("audio") && child->HasAttribute(wxT("lang")))
			GetAudioLangCodes().Add(child->GetAttribute(wxT("lang")));
		else if (child->GetName() == wxT("video"))
			m_video.PutXML(child, true);
		else if (child->GetName() == wxT("pgc")) {
			Pgc* pgc = new Pgc;
			if (pgc->PutXML(child, dvd, tsi, GetCount()))
				Add(pgc);
			else {
				delete pgc;
				wxLogError(_("Can't load pgc"));
				return false;
			}
		}
		child = child->GetNext();
	}
	if (m_video.GetAspect() == arAUTO && GetCount() > 0) {
		for (int i = 0; i < (int) GetCount(); i++) {
			Menu* menu = (*this)[i]->GetMenu();
			if (menu != NULL) {
				m_video.SetAspect(menu->GetAspectRatio());
				break;
			}
		}
	}
	return true;
}

////////////////////////////////// Titles ////////////////////////////////////

wxSvgXmlNode* Titles::GetXML(DVDFileType type, DVD* dvd, int nextTitleset) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("titles"));
	wxSvgXmlNode* videoNode = GetVideo().GetXML(type, false);
	if (videoNode)
		node->AddChild(videoNode);
	UpdateAudioLangCodes();
	for (wxArrayString::iterator it = GetAudioLangCodes().begin(); it != GetAudioLangCodes().end(); it++) {
		wxSvgXmlNode* audio = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("audio"));
		audio->AddProperty(wxT("lang"), *it);
		node->AddChild(audio);
	}
	UpdateSubtitleLangCodes();
	for (vector<SubPicture*>::iterator it = GetSubPictures().begin(); it != GetSubPictures().end(); it++) {
		node->AddChild((*it)->GetXML(type, m_video, false));
	}
	for (int i = 0; i < (int) GetCount(); i++) {
		wxString nextTitle;
		if (i != (int) GetCount() - 1)
			nextTitle = wxString::Format(wxT("title %d"), i + 2);
		node->AddChild((*this)[i]->GetXML(type, dvd, nextTitle));
	}
	return node;
}

bool Titles::PutXML(wxSvgXmlNode* node, DVD* dvd, int tsi) {
	if (node == NULL || node->GetName() != wxT("titles"))
		return false;
	wxSvgXmlNode* child = node->GetChildren();
	while (child) {
		if (child->GetName() == wxT("audio") && child->HasAttribute(wxT("lang"))) {
			GetAudioLangCodes().Add(child->GetAttribute(wxT("lang")));
		} else if (child->GetName() == wxT("subpicture")) {
			SubPicture* subPicture = new SubPicture();
			if (subPicture->PutXML(child))
				GetSubPictures().push_back(subPicture);
		} else if (child->GetName() == wxT("video")) {
			GetVideo().PutXML(child, false);
		} else if (child->GetName() == wxT("pgc")) {
			Pgc* pgc = new Pgc;
			if (pgc->PutXML(child, dvd, tsi, GetCount()))
				Add(pgc);
			else {
				delete pgc;
				wxLogError(_("Can't load pgc"));
				return false;
			}
		}
		child = child->GetNext();
	}
	return true;
}

////////////////////////////////// Titleset //////////////////////////////////

wxSvgXmlNode* Titleset::GetXML(DVDFileType type, DVD* dvd, int nextTitleset) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("titleset"));
	if (m_menus.Count() || type != DVDAUTHOR_XML)
		node->AddChild(m_menus.GetXML(type, dvd));
	else if (dvd->IsEmptyMenuEnabled() && dvd->GetVmgm().Count()) {
		// create empty menu with post command
		Menus menus;
		Pgc* pgc = new Pgc;
		pgc->SetPreCommands(wxT("jump vmgm menu;"));
		pgc->GetVobs().Add(Vob::CreateEmptyVob(dvd->GetVideoFormat(), dvd->GetAudioFormat()));
		menus.Add(pgc);
		node->AddChild(menus.GetXML(type, dvd));
	}
	node->AddChild(m_titles.GetXML(type, dvd, nextTitleset));
	return node;
}

bool Titleset::PutXML(wxSvgXmlNode* node, DVD* dvd, int tsi) {
	if (node == NULL || node->GetName() != wxT("titleset"))
		return false;
	wxSvgXmlNode* child = node->GetChildren();
	while (child) {
		if (child->GetName() == wxT("menus")) {
			if (!GetMenus().PutXML(child, dvd, tsi)) {
				wxLogError(_("Can't load menus"));
				return false;
			}
		} else if (child->GetName() == wxT("titles")) {
			if (!GetTitles().PutXML(child, dvd, tsi)) {
				wxLogError(_("Can't load titles"));
				return false;
			}
		}
		child = child->GetNext();
	}
	return true;
}

////////////////////////////////// VMGM //////////////////////////////////////

wxSvgXmlNode* Vmgm::GetXML(DVDFileType type, DVD* dvd) {
	wxSvgXmlNode* node = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("vmgm"));
	// add fgc
	if (GetFpc().length())
		XmlWriteValue(node, wxT("fpc"), GetFpc());

	// add menus
	if (type == DVDSTYLER_XML || Count())
		node->AddChild(Menus::GetXML(type, dvd));
	else if (dvd->IsEmptyMenuEnabled() && dvd->GetTitlesets().Count()
			&& dvd->GetTitlesets()[0]->GetMenus().Count()) {
		// create empty vmgm with post command
		Menus menus;
		Pgc* pgc = new Pgc;
		pgc->SetPreCommands(wxT("jump titleset 1 menu;"));
		pgc->GetVobs().Add(Vob::CreateEmptyVob(dvd->GetVideoFormat(), dvd->GetAudioFormat()));
		menus.Add(pgc);
		node->AddChild(menus.GetXML(DVDAUTHOR_XML, dvd));
	}
	return node;
}

bool Vmgm::PutXML(wxSvgXmlNode* node, DVD* dvd) {
	wxSvgXmlNode* child = node->GetChildren();
	while (child) {
		if (child->GetName() == wxT("menus"))
			Menus::PutXML(child, dvd, -1);
		else if (child->GetName() == wxT("fpc"))
			SetFpc(child->GetChildren()->GetContent());
		child = child->GetNext();
	}
	return true;
}

////////////////////////////////// DVD //////////////////////////////////////

DVD::DVD() {
	m_label = wxT("DVD");
	m_capacity = dcDVD47;
	m_videoBitrateAuto = true;
	m_videoBitrate = 4500;
	m_jumppad = false;
	m_emptyMenu = true;
	m_playAllRegister = -1;
	m_rememberLastButtonRegister = -1;
	m_videoFormat = vfPAL;
	m_audioFormat = afMP2;
	m_aspectRatio = ar4_3;
}

DVD::~DVD() {
	WX_CLEAR_ARRAY(m_vmgm);
	WX_CLEAR_ARRAY(m_titlesets);
}

/** Returns disc capacity in KB */
long DVD::GetCapacityValue() {
	if (m_capacity == dcDVD47)
		return 4590208;
	else if (m_capacity == dcDVD85)
		return 8343424;
	return 0;
}

/** Calculates video bitrate if it set to auto */
wxArrayInt DVD::CalculateVideoBitrate() {
	long size = 0;
	long fixSize = s_config.GetDvdReservedSpace();
	long fixDuration = 0;
	long duration = 0;
	if (m_videoBitrateAuto)
		SetVideoBitrate(500);
	// menus
	for (int tsi = -1; tsi < (int) GetTitlesets().Count(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, true);
		for (int pgci = 0; pgci < (int) pgcs.Count(); pgci++) {
			Pgc* pgc = pgcs[pgci];
			for (int vobi = 0; vobi < (int) pgc->GetVobs().Count(); vobi++) {
				Vob* vob = pgc->GetVobs()[vobi];
				fixSize += vob->GetSize(this);
			}
		}
	}
	// titles
	for (int tsi = 0; tsi < (int) GetTitlesets().Count(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, false);
		for (int pgci = 0; pgci < (int) pgcs.Count(); pgci++) {
			Pgc* pgc = pgcs[pgci];
			for (int vobi = 0; vobi < (int) pgc->GetVobs().Count(); vobi++) {
				Vob* vob = pgc->GetVobs()[vobi];
				VideoFormat videoFormat = vfCOPY;
				if (!vob->GetSlideshow() && !vob->GetDoNotTranscode()) {
					for (int stIdx = 0; stIdx < (int) vob->GetStreams().GetCount(); stIdx++) {
						Stream* stream = vob->GetStreams()[stIdx];
						if (stream->GetType() == stVIDEO) {
							videoFormat = stream->GetVideoFormat();
							break;
						}
					}
				}
				if (videoFormat != vfCOPY) {
					duration += (long) vob->GetDuration();
					size += vob->GetSize(this);
				} else { 
					fixDuration += (long) vob->GetDuration();
					fixSize += vob->GetSize(this);
				}
			}
		}
	}
	long bitrate = GetVideoBitrate();
	if (m_videoBitrateAuto) {
		if (GetCapacity() != dcUNLIMITED) {
			long s = GetCapacityValue() - size - fixSize;
			bitrate = 500 + (s > 0 && duration > 0 ? s/duration*8 : (duration > 0 ? 0 : 7500));
			if (bitrate > 8000)
				bitrate = 8000;
		} else
			bitrate = 4500;
		SetVideoBitrate(bitrate);
	}
	wxArrayInt res;
	res.Add(bitrate);
	res.Add(fixSize);
	res.Add(fixDuration);
	return res;
}

unsigned int DVD::GetAudioStreamCount() {
	unsigned int count = 1; // at least one audio stream
	for (TitlesetArray::iterator it = GetTitlesets().begin(); it != GetTitlesets().end(); it++)
		if (count < (*it)->GetTitles().GetAudioStreamCount())
			count = (*it)->GetTitles().GetAudioStreamCount();
	return count;
}

unsigned int DVD::GetSubtitleStreamsCount() {
	unsigned int count = 0;
	for (TitlesetArray::iterator it = GetTitlesets().begin(); it != GetTitlesets().end(); it++)
		if (count < (*it)->GetTitles().GetSubtitleStreamsCount())
			count = (*it)->GetTitles().GetSubtitleStreamsCount();
	return count;
}

int DVD::AddTitleset() {
	Titleset* titleset = new Titleset;
	GetTitlesets().Add(titleset);
	return GetTitlesets().Count();
}

int DVD::AddMenu(int tsi) {
	return AddMenu(new Menu(this, tsi, GetPgcArray(tsi, true).GetCount(), m_videoFormat, m_aspectRatio), tsi);
}

int DVD::AddMenu(Menu* menu, int tsi) {
	Vob* vob = new Vob(menu);
	vob->SetPause(-1);
	Pgc* pgc = new Pgc;
	pgc->GetVobs().Add(vob);
	PgcArray& pgcs = GetPgcArray(tsi, true);
	pgcs.Add(pgc);
	return pgcs.Count()-1;
}

int DVD::AddPgc(int tsi, bool menu, Pgc* pgc)
{
  if (!pgc)
	pgc = new Pgc;
  PgcArray& pgcs = GetPgcArray(tsi, menu);
  pgcs.Add(pgc);
  return pgcs.Count()-1;
}

PgcArray& DVD::GetPgcArray(int tsi, bool menus) {
	if (tsi >= 0 && tsi < (int) GetTitlesets().Count()) {
		Titleset* ts = GetTitlesets()[tsi];
		if (menus)
			return ts->GetMenus();
		else
			return ts->GetTitles();
	}
	return GetVmgm();
}

Vob* DVD::GetVob(int tsi, bool isMenu, int pgci, int vobi) {
	PgcArray& pgcs = GetPgcArray(tsi, isMenu);
	if (pgci < 0 || pgci >= (int) pgcs.Count())
		return NULL;
	if (vobi < 0 || vobi >= (int) pgcs[pgci]->GetVobs().Count())
		return NULL;
	return pgcs[pgci]->GetVobs()[vobi];
}

Vob* DVD::GetMenuVob(int tsi, int pgci) {
	PgcArray& pgcs = GetPgcArray(tsi, true);
	if (pgci < 0 || pgci >= (int) pgcs.Count())
		return NULL;
	// find vob with menu
	for (int vobi = 0; vobi < (int) pgcs[pgci]->GetVobs().GetCount(); vobi++) {
		if (pgcs[pgci]->GetVobs()[vobi]->GetMenu() != NULL)
			return pgcs[pgci]->GetVobs()[vobi];
	}
	return NULL;
}

Menu* DVD::GetMenu(int tsi, int pgci) {
	Vob* vob = GetMenuVob(tsi, pgci);
	if (vob)
		return vob->GetMenu();
	return NULL;
}

/** Returns true if all menus are ok */
bool DVD::CheckMenus() {
	for (int tsi = -1; tsi < (int) GetTitlesets().Count(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, true);
		for (unsigned int pgci = 0; pgci < pgcs.GetCount(); pgci++) {
			Menu* menu = pgcs[pgci]->GetMenu();
			if (menu) {
				if (menu->GetAspectRatio() == ar16_9) {
					if (pgcs.GetVideo().GetWidescreen() == wtAUTO && menu->GetButtonsCount() > 12) {
						wxLogError(_("Wide screen DVD menu can contain maximal 12 buttons"));
						return false;
					} else if (menu->GetButtonsCount() > 18) {
						wxLogError(_("Wide screen DVD menu (nopanscan/noletterbox) can contain maximal 18 buttons"));
						return false;
					}
				} else if (menu->GetButtonsCount() > 34) {
					wxLogError(_("DVD menu can contain maximal 34 buttons"));
					return false;
				}
			}
		}
	}
	return true;
}

/** Returns true if all actions are valid */
bool DVD::CheckActions(bool skipInvalidTarget) {
	bool valid = true;
	for (int tsi = -1; tsi < (int) GetTitlesets().Count(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, true);
		for (unsigned int pgci = 0; pgci < pgcs.GetCount(); pgci++) {
			Menu* menu = pgcs[pgci]->GetMenu();
			if (menu) {
				for (unsigned int obji = 0; obji < menu->GetObjectsCount(); obji++) {
					MenuObject* obj = menu->GetObject(obji);
					if (obj->IsButton() && !obj->GetAction().IsValid(
							this, tsi, pgci, true, obj->GetId(true), true, true, skipInvalidTarget))
						valid = false;
				}
			}
			// check commands
			if (pgcs[pgci]->GetPreCommands().length()) {
				DVDAction action(false, BUTTON_ACTION);
				action.SetCustom(pgcs[pgci]->GetPreCommands());
				if (!action.IsValid(this, tsi, pgci, true, wxT(""), true))
					valid = false;
			}
			if (pgcs[pgci]->GetPostCommands().length()) {
				DVDAction action(false, BUTTON_ACTION);
				action.SetCustom(pgcs[pgci]->GetPostCommands());
				if (!action.IsValid(this, tsi, pgci, true, wxT(""), true))
					valid = false;
			}
		}
		if (tsi >= 0) {
			PgcArray& pgcs = GetPgcArray(tsi, false);
			for (unsigned int pgci = 0; pgci < pgcs.GetCount(); pgci++) {
				// check commands
				if (pgcs[pgci]->GetPreCommands().length()) {
					DVDAction action(false, BUTTON_ACTION);
					action.SetCustom(pgcs[pgci]->GetPreCommands());
					if (!action.IsValid(this, tsi, pgci, false, wxT(""), true))
						valid = false;
				}
				if (pgcs[pgci]->GetPostCommands().length()) {
					DVDAction action(false, BUTTON_ACTION);
					action.SetCustom(pgcs[pgci]->GetPostCommands());
					if (!action.IsValid(this, tsi, pgci, false, wxT(""), true))
						valid = false;
				}
			}
		}
	}
	if (!valid) {
		wxLogError(_("Some actions are invalid. Please click 'Details' for more information."));
	}
	return valid;
}

/** Updates image in buttons with given jump action */
bool DVD::UpdateButtonImageFor(int actionTsi, int actionPgci) {
	bool result = false;
	for (int tsi = -1; tsi < (int) GetTitlesets().GetCount(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, true);
		for (unsigned int pgci = 0; pgci < pgcs.GetCount(); pgci++) {
			Menu* menu = pgcs[pgci]->GetMenu();
			if (menu)
				result |= menu->UpdateButtonImageFor(actionTsi != tsi ? actionTsi : -2, actionPgci, this);
		}
	}
	return result;
}


/** Fix coordinates of buttons if they are out of range */
void DVD::FixButtonCoordinates() {
	for (int tsi = -1; tsi < (int) GetTitlesets().GetCount(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, true);
		for (unsigned int pgci = 0; pgci < pgcs.GetCount(); pgci++) {
			Menu* menu = pgcs[pgci]->GetMenu();
			if (menu)
				menu->FixButtonCoordinates();
		}
	}
}


/** Sets video format of DVD */
void DVD::SetVideoFormat(VideoFormat videoFormat, bool updateMenus) {
	m_videoFormat = videoFormat;
	if (updateMenus) {
		for (int tsi=-1; tsi< (int) GetTitlesets().GetCount(); tsi++)
			GetPgcArray(tsi, true).SetVideoFormat(videoFormat);
	}
}

/** Sets audio format of DVD */
void DVD::SetAudioFormat(AudioFormat format) {
	m_audioFormat = format;
}

/** Sets aspect ratio of DVD */
void DVD::SetAspectRatio(AspectRatio aspectRatio, bool updateMenus) {
	m_aspectRatio = aspectRatio;
	if (updateMenus) {
		for (int tsi=-1; tsi< (int) GetTitlesets().GetCount(); tsi++)
			GetPgcArray(tsi, true).SetAspectRatio(aspectRatio);
	}
}

/** Loads a project file */
bool DVD::Open(wxString fname) {
	wxSvgXmlDocument xml;
	if (!xml.Load(fname)) {
		wxLogError(_("Cannot open file '%s'."), fname.c_str());
		return false;
	}

	wxSvgXmlNode* root = xml.GetRoot();
	if (root == NULL || root->GetName() != wxT("dvdstyler")) {
		wxLogError(_("'%s' is not a DVDStyler project file"), fname.c_str());
		return false;
	}

	if (fname.EndsWith(wxT(".dvdt")))
		m_templateFile = fname;
	else
		m_filename = fname;
	return PutXML(xml);
}

/** Initializes object with XML data */
bool DVD::PutXML(const wxSvgXmlDocument& xml) {
	WX_CLEAR_ARRAY(GetTitlesets());

	wxSvgXmlNode* root = xml.GetRoot();
	if (root == NULL || root->GetName() != wxT("dvdstyler")) {
		wxLogError(wxT("Invalid project XML data"));
		return false;
	}
	root->GetPropVal(wxT("name"), &m_label);

	wxString val;
	long lval;

	int format = 0;
	if (root->GetPropVal(wxT("format"), &val) && val.ToLong(&lval))
		format = int(lval);
	if (root->GetPropVal(wxT("template"), &val)) {
		if (wxFileName(val).IsAbsolute())
			m_templateFile = val;
		else
			m_templateFile = TEMPLATES_DIR + val;
	}
	if (root->GetPropVal(wxT("capacity"), &val) && val.ToLong(&lval))
		m_capacity = DiscCapacity(lval);
	if (root->GetPropVal(wxT("jumppad"), &val) && val.ToLong(&lval))
		m_jumppad = int(lval);
	if (root->GetPropVal(wxT("emptyMenu"), &val) && val.ToLong(&lval))
		m_emptyMenu = int(lval);
	if (root->GetPropVal(wxT("videoFormat"), &val) && val.ToLong(&lval))
		m_videoFormat = VideoFormat(lval + (format == 2 ? 2 : (format == 3 ? 1 : 0)));
	if (root->GetPropVal(wxT("audioFormat"), &val) && val.ToLong(&lval))
		m_audioFormat = AudioFormat(lval + (format == 2 ? 2 : (format == 3 ? 1 : 0)));
	if (root->GetPropVal(wxT("aspectRatio"), &val) && val.ToLong(&lval))
		m_aspectRatio = AspectRatio(lval);
	if (root->GetPropVal(wxT("videoBitrate"), &val) && val.ToLong(&lval)) {
		m_videoBitrateAuto = lval == -1;
		m_videoBitrate = lval == -1 ? 4500 : lval;
	}

	wxSvgXmlNode* child = root->GetChildren();
	while (child) {
		if (child->GetName() == wxT("vmgm")) {
			if (!GetVmgm().PutXML(child, this)) {
				wxLogError(_("Can't load vmgm menus"));
				return false;
			}
		} else if (child->GetName() == wxT("titleset")) {
			Titleset* titleset = new Titleset;
			if (titleset->PutXML(child, this, GetTitlesets().GetCount()))
				GetTitlesets().Add(titleset);
			else {
				delete titleset;
				wxLogError(_("Can't load titleset"));
				return false;
			}
		}
		child = child->GetNext();
	}
	return true;
}


/** Returns XML Document */
wxSvgXmlDocument* DVD::GetXml() {
	m_playAllRegister = -1;
	m_rememberLastButtonRegister = -1;
	wxSvgXmlDocument* xml = new wxSvgXmlDocument;
	wxSvgXmlNode* root = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("dvdstyler"));
	root->AddProperty(wxT("format"), wxT("4"));
	if (m_templateFile.length() > 0) {
		wxString tplName = m_templateFile;
		wxString dir = TEMPLATES_DIR;
		if (tplName.StartsWith(dir))
			tplName = tplName.substr(dir.length());
		root->AddProperty(wxT("template"), tplName);
	}
	root->AddProperty(wxT("name"), m_label);
	root->AddProperty(wxT("capacity"), wxString::Format(wxT("%d"), m_capacity));
	root->AddProperty(wxT("jumppad"), m_jumppad ? wxT("1") : wxT("0"));
	root->AddProperty(wxT("emptyMenu"), m_emptyMenu ? wxT("1") : wxT("0"));
	root->AddProperty(wxT("videoFormat"), wxString::Format(wxT("%d"), m_videoFormat));
	root->AddProperty(wxT("audioFormat"), wxString::Format(wxT("%d"), m_audioFormat));
	root->AddProperty(wxT("aspectRatio"), wxString::Format(wxT("%d"), m_aspectRatio));
	root->AddProperty(wxT("videoBitrate"), wxString::Format(wxT("%d"), m_videoBitrateAuto ? -1 : m_videoBitrate));
	root->AddChild(GetVmgm().GetXML(DVDSTYLER_XML, this));
	for (int i=0; i<(int)m_titlesets.GetCount(); i++)
		if (m_titlesets[i]->GetMenus().Count() || m_titlesets[i]->GetTitles().Count())
			root->AddChild(m_titlesets[i]->GetXML(DVDSTYLER_XML, this));
	xml->SetRoot(root);
	return xml;
}

bool DVD::Save(wxString fname) {
	if (fname.length() > 0)
		m_filename = fname;
	wxSvgXmlDocument* xml = GetXml();
	bool result = xml->Save(m_filename);
	delete xml;
	return result;
}

/** Stores object data to string */
wxString DVD::Serialize() {
	wxSvgXmlDocument* xml = GetXml();
	wxStringOutputStream stream;
	xml->Save(stream);
	delete xml;
	return stream.GetString();
}

/** Restores object from data */
void DVD::Deserialize(const wxString& data) {
	wxStringInputStream stream(data);
	wxSvgXmlDocument xml;
	xml.Load(stream);
	PutXML(xml);
}

bool DVD::SaveDVDAuthor(wxString fname) {
	// check if we need "play all"
	m_playAllRegister = -1;
	m_playAllRegister = HasPlayAllButton() ? FindFreeRegister() : -1;
	m_rememberLastButtonRegister = HasRememberLastButton() ? FindFreeRegister() : -1;
	// save config for dvdauthor
	wxSvgXmlDocument xml;
	wxSvgXmlNode* root = new wxSvgXmlNode(wxSVGXML_ELEMENT_NODE, wxT("dvdauthor"));
	root->AddProperty(wxT("jumppad"), m_jumppad ? wxT("1") : wxT("0"));
	root->AddChild(GetVmgm().GetXML(DVDAUTHOR_XML, this));
	for (int i = 0; i < (int) m_titlesets.GetCount(); i++) {
		int nextTitleset =
				i < (int) m_titlesets.GetCount() - 1 && !m_titlesets[i + 1]->IsEmpty() ? i + 2 : -1;
		if (!m_titlesets[i]->IsEmpty())
			root->AddChild(m_titlesets[i]->GetXML(DVDAUTHOR_XML, this, nextTitleset));
	}
	xml.SetRoot(root);
	return xml.Save(fname);
}

wxString DVD::GetPath(bool withSep) {
	wxString path = wxPathOnly(m_filename);
	return withSep && path.length() > 0 ? path + wxFILE_SEP_PATH : path;
}

long DVD::GetSize(bool generated) {
	int size = s_config.GetDvdReservedSpace();
	for (int tsi = -1; tsi<(int)GetTitlesets().Count(); tsi++) {
		for (int m=0; m<=1; m++) {
			bool menu = m == 0;
			if (tsi == -1 && !menu) // "titleset -1" contains only vmMenus
				break;
			PgcArray& pgcs = GetPgcArray(tsi, menu);
			for (int pgci = 0; pgci<(int)pgcs.Count(); pgci++) {
				Pgc* pgc = pgcs[pgci];
				for (int vobi = 0; vobi<(int)pgc->GetVobs().Count(); vobi++)
					size += pgc->GetVobs()[vobi]->GetSize(this, generated);
			}
		}
	}
	return size;
}

long DVD::GetRequiredSize(Cache* cache) {
	int size = 0;
	for (int tsi = -1; tsi<(int)GetTitlesets().Count(); tsi++) {
		for (int m=0; m<=1; m++) {
			bool menu = m == 0;
			if (tsi == -1 && !menu) // "titleset -1" contains only vmMenus
				break;
			PgcArray& pgcs = GetPgcArray(tsi, menu);
			for (int pgci = 0; pgci<(int)pgcs.Count(); pgci++) {
				Pgc* pgc = pgcs[pgci];
				for (int vobi = 0; vobi<(int)pgc->GetVobs().Count(); vobi++)
					size += pgc->GetVobs()[vobi]->GetRequiredSize(this, cache);
			}
		}
	}
	return size;
}

/** Returns true if some of buttons have flag "Play All" checked */
bool DVD::HasPlayAllButton() {
	for (int tsi = -1; tsi<(int)GetTitlesets().Count(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, true);
		for (int pgci = 0; pgci<(int)pgcs.Count(); pgci++) {
			Pgc* pgc = pgcs[pgci];
			for (int vobi = 0; vobi<(int)pgc->GetVobs().Count(); vobi++) {
				Vob* vob = pgc->GetVobs()[vobi];
				if (vob->GetMenu()) {
					Menu* menu = vob->GetMenu();
					for (unsigned int obji = 0; obji<menu->GetObjectsCount(); obji++)
						if (menu->GetObject(obji)->IsButton()
								&& menu->GetObject(obji)->GetAction().IsPlayAll())
							return true;
				}
			}
		}
	}
	return false;
}

/** Returns true if some of menus have flag "Remember last selected button" checked */
bool DVD::HasRememberLastButton() {
	for (int tsi = -1; tsi<(int)GetTitlesets().Count(); tsi++) {
		PgcArray& pgcs = GetPgcArray(tsi, true);
		for (int pgci = 0; pgci<(int)pgcs.Count(); pgci++) {
			Pgc* pgc = pgcs[pgci];
			for (int vobi = 0; vobi<(int)pgc->GetVobs().Count(); vobi++) {
				Vob* vob = pgc->GetVobs()[vobi];
				if (vob->GetMenu() && vob->GetMenu()->GetRememberLastButton())
					return true;
			}
		}
	}
	return false;
}

int DVD::FindFreeRegister() {
	for (int g=0; g<=12; g++) {
		if (!IsRegisterUsed(g))
			return g;
	}
	return -1;
}

bool DVD::IsRegisterUsed(int g) {
	if (m_playAllRegister == g)
		return true;
	wxString gstr = wxT("g") + wxString::Format(wxT("%d"),g);
	for (int tsi = -1; tsi<(int)GetTitlesets().Count(); tsi++) {
		for (int m=0; m<=1; m++) {
			bool menu = m == 0;
			if (tsi == -1 && !menu) // titleset -1 is Video-Manager and contains only vmMenus
				continue;
			PgcArray& pgcs = GetPgcArray(tsi, menu);
			for (int pgci = 0; pgci<(int)pgcs.Count(); pgci++) {
				Pgc* pgc = pgcs[pgci];
				if (pgc->GetPostCommands().Find(gstr) != -1
						|| pgc->GetPreCommands().Find(gstr) != -1)
					return true;
				for (int vobi = 0; vobi<(int)pgc->GetVobs().Count(); vobi++) {
					Vob* vob = pgc->GetVobs()[vobi];
					if (vob->GetMenu()) {
						Menu* menu = vob->GetMenu();
						for (unsigned int obji = 0; obji<menu->GetObjectsCount(); obji++) {
							MenuObject* obj = menu->GetObject(obji);
							if (obj->IsButton() && obj->GetAction().IsCustom()
									&& obj->GetAction().GetCustom().Find(gstr) != -1)
								return true;
						}
						for (unsigned int aIdx = 0; aIdx<menu->GetActionsCount(); aIdx++) {
							DVDAction* action = menu->GetAction(aIdx);
							if (action->IsCustom() && action->GetCustom().Find(gstr) != -1)
								return true;
						}
					}
				}
			}
		}
	}
	return false;
}

wxArrayString DVD::GetVideoFormatLabels(bool copy, bool none, bool menu) {
	wxArrayString formats;
	if (none)
		formats.Add(_("None"));
	if (copy)
		formats.Add(_("Copy"));
	formats.Add(wxT("PAL 720x576"));
	formats.Add(wxT("NTSC 720x480"));
	if (!menu) {
		formats.Add(wxT("PAL 704x576"));
		formats.Add(wxT("NTSC 704x480"));
		formats.Add(wxT("PAL 352x576"));
		formats.Add(wxT("NTSC 352x480"));
		formats.Add(wxT("PAL 352x288"));
		formats.Add(wxT("NTSC 352x240"));
	}
	return formats;
}

wxArrayString DVD::GetAudioFormatLabels(bool copy, bool none) {
	wxArrayString formats;
	if (none)
		formats.Add(_("None"));
	if (copy)
		formats.Add(_("Copy"));
	formats.Add(_("MP2 48 kHz"));
	formats.Add(_("AC3 48 kHz"));
	return formats;
}

wxArrayString DVD::GetAspectRatioLabels(bool autom) {
	wxArrayString formats;
	if (autom)
		formats.Add(_("Auto"));
	formats.Add(wxT("4:3"));
	formats.Add(wxT("16:9"));
	return formats;
}

wxArrayString DVD::GetCapacityLabels() {
	wxArrayString labels;
	labels.Add(_("4.7 GB"));
	labels.Add(_("8.5 GB"));
	labels.Add(_("Unlimited"));
	return labels;
}

wxArrayString DVD::GetVideoBitrateLabels() {
	wxArrayString labels;
	labels.Add(_("Auto"));
	for (int i=8; i>=2; i--)
		labels.Add(wxString::Format(_("%d MBit/s"), i));
	labels.Add(_("Custom"));
	return labels;
}

wxArrayString DVD::GetVideoFormatNames() {
	wxArrayString formats;
	formats.Add(wxT("none"));
	formats.Add(wxT("copy"));
	formats.Add(wxT("pal"));
	formats.Add(wxT("ntsc"));
	return formats;
}

wxArrayString DVD::GetAudioFormatNames() {
	wxArrayString formats;
	formats.Add(wxT("none"));
	formats.Add(wxT("copy"));
	formats.Add(wxT("mp2"));
	formats.Add(wxT("ac3"));
	return formats;
}

VideoFormat DVD::GetVideoFormatByName(wxString name) {
	int idx = GetVideoFormatNames().Index(name, false);
	return idx > 0 ? (VideoFormat) idx : vfNONE;
}

AudioFormat DVD::GetAudioFormatByName(wxString name) {
	int idx = GetAudioFormatNames().Index(name, false);
	return idx > 0 ? (AudioFormat) idx : afNONE;
}

wxString DVD::GetVideoFormatName(VideoFormat format) {
	return GetVideoFormatNames()[(int)format];
}

wxString DVD::GetAudioFormatName(AudioFormat format) {
	return GetAudioFormatNames()[(int)format];
}

const wxString audioLangCodesArray[] = { wxT("AA"), wxT("AB"), wxT("AF"), wxT("AM"), wxT("AR"), wxT("AS"),
		wxT("AY"), wxT("AZ"), wxT("BA"), wxT("BE"), wxT("BG"), wxT("BH"), wxT("BI"), wxT("BN"), wxT("BO"),
		wxT("BR"), wxT("CA"), wxT("CO"), wxT("CS"), wxT("CY"), wxT("DA"), wxT("DE"), wxT("DZ"), wxT("EL"),
		wxT("EN"), wxT("EO"), wxT("ES"), wxT("ET"), wxT("EU"), wxT("FA"), wxT("FI"), wxT("FJ"), wxT("FO"),
		wxT("FR"), wxT("FY"), wxT("GA"), wxT("GD"), wxT("GL"), wxT("GN"), wxT("GU"), wxT("HA"), wxT("HI"),
		wxT("HR"), wxT("HU"), wxT("HY"), wxT("IA"), wxT("IE"), wxT("IK"), wxT("IN"), wxT("IS"), wxT("IT"),
		wxT("IW"), wxT("JA"), wxT("JI"), wxT("JV"), wxT("KA"), wxT("KK"), wxT("KL"), wxT("KM"), wxT("KN"),
		wxT("KO"), wxT("KS"), wxT("KU"), wxT("KY"), wxT("LA"), wxT("LN"), wxT("LO"), wxT("LT"), wxT("LV"),
		wxT("MG"), wxT("MI"), wxT("MK"), wxT("ML"), wxT("MN"), wxT("MO"), wxT("MR"), wxT("MS"), wxT("MT"),
		wxT("MY"), wxT("NA"), wxT("NE"), wxT("NL"), wxT("NO"), wxT("OC"), wxT("OM"), wxT("OR"), wxT("PA"),
		wxT("PL"), wxT("PS"), wxT("PT"), wxT("QU"), wxT("RM"), wxT("RN"), wxT("RO"), wxT("RU"), wxT("RW"),
		wxT("SA"), wxT("SD"), wxT("SG"), wxT("SH"), wxT("SI"), wxT("SK"), wxT("SL"), wxT("SM"), wxT("SN"),
		wxT("SO"), wxT("SQ"), wxT("SR"), wxT("SS"), wxT("ST"), wxT("SU"), wxT("SV"), wxT("SW"), wxT("TA"),
		wxT("TE"), wxT("TG"), wxT("TH"), wxT("TI"), wxT("TK"), wxT("TL"), wxT("TN"), wxT("TO"), wxT("TR"),
		wxT("TS"), wxT("TT"), wxT("TW"), wxT("UK"), wxT("UR"), wxT("UZ"), wxT("VI"), wxT("VO"), wxT("WO"),
		wxT("XH"), wxT("YO"), wxT("ZH"), wxT("ZU") };
static wxArrayString s_audioLanguageCodes(136, audioLangCodesArray);

wxArrayString DVD::GetAudioLanguageCodes() {
	return s_audioLanguageCodes;
}

static map<wxString, wxString> s_languageMap;

map<wxString, wxString>& DVD::GetLanguageMap() {
	if (s_languageMap.size() == 0) {
		s_languageMap[wxT("Abkhazian")] = wxT("AB");
		s_languageMap[wxT("Afar")] = wxT("AA");
		s_languageMap[wxT("Afrikaans")] = wxT("AF");
		s_languageMap[wxT("Akan")] = wxT("AK");
		s_languageMap[wxT("Albanian")] = wxT("SQ");
		s_languageMap[wxT("Amharic")] = wxT("AM");
		s_languageMap[wxT("Arabic")] = wxT("AR");
		s_languageMap[wxT("Aragonese")] = wxT("AN");
		s_languageMap[wxT("Assamese")] = wxT("AS");
		s_languageMap[wxT("Armenian")] = wxT("HY");
		s_languageMap[wxT("Avaric")] = wxT("AV");
		s_languageMap[wxT("Avestan")] = wxT("AE");
		s_languageMap[wxT("Aymara")] = wxT("AY");
		s_languageMap[wxT("Azerbaijani")] = wxT("AZ");
		s_languageMap[wxT("Bashkir")] = wxT("BA");
		s_languageMap[wxT("Bambara")] = wxT("BM");
		s_languageMap[wxT("Basque")] = wxT("EU");
		s_languageMap[wxT("Belarusian")] = wxT("BE");
		s_languageMap[wxT("Bengali")] = wxT("BN");
		s_languageMap[wxT("Bihari")] = wxT("BH");
		s_languageMap[wxT("Bislama")] = wxT("BI");
		s_languageMap[wxT("Bosnian")] = wxT("BS");
		s_languageMap[wxT("Breton")] = wxT("BR");
		s_languageMap[wxT("Bulgarian")] = wxT("BG");
		s_languageMap[wxT("Burmese")] = wxT("MY");
		s_languageMap[wxT("Catalan; Valencian")] = wxT("CA");
		s_languageMap[wxT("Chamorro")] = wxT("CH");
		s_languageMap[wxT("Chechen")] = wxT("CE");
		s_languageMap[wxT("Chichewa; Chewa; Nyanja")] = wxT("NY");
		s_languageMap[wxT("Chinese")] = wxT("ZH");
		s_languageMap[wxT("Chuvash")] = wxT("CV");
		s_languageMap[wxT("Cornish")] = wxT("KW");
		s_languageMap[wxT("Corsican")] = wxT("CO");
		s_languageMap[wxT("Cree")] = wxT("CR");
		s_languageMap[wxT("Croatian")] = wxT("HR");
		s_languageMap[wxT("Czech")] = wxT("CS");
		s_languageMap[wxT("Danish")] = wxT("DA");
		s_languageMap[wxT("Divehi; Dhivehi; Maldivian;")] = wxT("DV");
		s_languageMap[wxT("Dzongkha")] = wxT("DZ");
		s_languageMap[wxT("English")] = wxT("EN");
		s_languageMap[wxT("Esperanto")] = wxT("EO");
		s_languageMap[wxT("Estonian")] = wxT("ET");
		s_languageMap[wxT("Ewe")] = wxT("EE");
		s_languageMap[wxT("Faroese")] = wxT("FO");
		s_languageMap[wxT("Fijian")] = wxT("FJ");
		s_languageMap[wxT("Finnish")] = wxT("FI");
		s_languageMap[wxT("French")] = wxT("FR");
		s_languageMap[wxT("Fula; Fulah; Pulaar; Pular")] = wxT("FF");
		s_languageMap[wxT("Galician")] = wxT("GL");
		s_languageMap[wxT("German")] = wxT("DE");
		s_languageMap[wxT("Greek, Modern")] = wxT("EL");
		s_languageMap[wxT("Guarani")] = wxT("GN");
		s_languageMap[wxT("Gujarati")] = wxT("GU");
		s_languageMap[wxT("Haitian; Haitian Creole")] = wxT("HT");
		s_languageMap[wxT("Hausa")] = wxT("HA");
		s_languageMap[wxT("Hebrew, Modern")] = wxT("HE");
		s_languageMap[wxT("Herero")] = wxT("HZ");
		s_languageMap[wxT("Hindi")] = wxT("HI");
		s_languageMap[wxT("Hiri Motu")] = wxT("HO");
		s_languageMap[wxT("Hungarian")] = wxT("HU");
		s_languageMap[wxT("Interlingua")] = wxT("IA");
		s_languageMap[wxT("Indonesian")] = wxT("ID");
		s_languageMap[wxT("Interlingue, Occidental")] = wxT("IE");
		s_languageMap[wxT("Irish")] = wxT("GA");
		s_languageMap[wxT("Igbo")] = wxT("IG");
		s_languageMap[wxT("Sichuan Yi, Nuosu")] = wxT("II");
		s_languageMap[wxT("Inupiaq")] = wxT("IK");
		s_languageMap[wxT("Ido")] = wxT("IO");
		s_languageMap[wxT("Icelandic")] = wxT("IS");
		s_languageMap[wxT("Italian")] = wxT("IT");
		s_languageMap[wxT("Inuktitut")] = wxT("IU");
		s_languageMap[wxT("Japanese")] = wxT("JA");
		s_languageMap[wxT("Javanese")] = wxT("JV");
		s_languageMap[wxT("Georgian")] = wxT("KA");
		s_languageMap[wxT("Kongo")] = wxT("KG");
		s_languageMap[wxT("Kikuyu, Gikuyu")] = wxT("KI");
		s_languageMap[wxT("Kwanyama, Kuanyama")] = wxT("KJ");
		s_languageMap[wxT("Kazakh")] = wxT("KK");
		s_languageMap[wxT("Kalaallisut, Greenlandic")] = wxT("KL");
		s_languageMap[wxT("Central Khmer")] = wxT("KM");
		s_languageMap[wxT("Kannada")] = wxT("KN");
		s_languageMap[wxT("Korean")] = wxT("KO");
		s_languageMap[wxT("Kanuri")] = wxT("KR");
		s_languageMap[wxT("Kashmiri")] = wxT("KS");
		s_languageMap[wxT("Kurdish")] = wxT("KU");
		s_languageMap[wxT("Komi")] = wxT("KV");
		s_languageMap[wxT("Kirghiz, Kyrgyz")] = wxT("KY");
		s_languageMap[wxT("Latin")] = wxT("LA");
		s_languageMap[wxT("Luxembourgish")] = wxT("LB");
		s_languageMap[wxT("Luganda")] = wxT("LG");
		s_languageMap[wxT("Limburgish")] = wxT("LI");
		s_languageMap[wxT("Lingala")] = wxT("LN");
		s_languageMap[wxT("Lao")] = wxT("LO");
		s_languageMap[wxT("Lithuanian")] = wxT("LT");
		s_languageMap[wxT("Luba-Katanga")] = wxT("LU");
		s_languageMap[wxT("Latvian")] = wxT("LV");
		s_languageMap[wxT("Malagasy")] = wxT("MG");
		s_languageMap[wxT("Marshallese")] = wxT("MH");
		s_languageMap[wxT("Manx")] = wxT("GV");
		s_languageMap[wxT("Maori")] = wxT("MI");
		s_languageMap[wxT("Macedonian")] = wxT("MK");
		s_languageMap[wxT("Malayalam")] = wxT("ML");
		s_languageMap[wxT("Mongolian")] = wxT("MN");
		s_languageMap[wxT("Marathi")] = wxT("MR");
		s_languageMap[wxT("Malay")] = wxT("MS");
		s_languageMap[wxT("Maltese")] = wxT("MT");
		s_languageMap[wxT("Nauru")] = wxT("NA");
		s_languageMap[wxT("Norwegian Bokmal")] = wxT("NB");
		s_languageMap[wxT("North Ndebele")] = wxT("ND");
		s_languageMap[wxT("Nepali")] = wxT("NE");
		s_languageMap[wxT("Ndonga")] = wxT("NG");
		s_languageMap[wxT("Dutch, Flemish")] = wxT("NL");
		s_languageMap[wxT("Norwegian Nynorsk")] = wxT("NN");
		s_languageMap[wxT("Norwegian")] = wxT("NO");
		s_languageMap[wxT("South Ndebele")] = wxT("NR");
		s_languageMap[wxT("Navajo, Navaho")] = wxT("NV");
		s_languageMap[wxT("Occitan (after 1500)")] = wxT("OC");
		s_languageMap[wxT("Ojibwa")] = wxT("OJ");
		s_languageMap[wxT("Church Slavic")] = wxT("CU");
		s_languageMap[wxT("Oromo")] = wxT("OM");
		s_languageMap[wxT("Oriya")] = wxT("OR");
		s_languageMap[wxT("Ossetian, Ossetic")] = wxT("OS");
		s_languageMap[wxT("Panjabi, Punjabi")] = wxT("PA");
		s_languageMap[wxT("Pali")] = wxT("PI");
		s_languageMap[wxT("Persian")] = wxT("FA");
		s_languageMap[wxT("Polish")] = wxT("PL");
		s_languageMap[wxT("Pashto, Pushto")] = wxT("PS");
		s_languageMap[wxT("Portuguese")] = wxT("PT");
		s_languageMap[wxT("Quechua")] = wxT("QU");
		s_languageMap[wxT("Romansh")] = wxT("RM");
		s_languageMap[wxT("Kirundi")] = wxT("RN");
		s_languageMap[wxT("Romanian, Moldavian")] = wxT("RO");
		s_languageMap[wxT("Russian")] = wxT("RU");
		s_languageMap[wxT("Kinyarwanda")] = wxT("RW");
		s_languageMap[wxT("Sanskrit")] = wxT("SA");
		s_languageMap[wxT("Sardinian")] = wxT("SC");
		s_languageMap[wxT("Sindhi")] = wxT("SD");
		s_languageMap[wxT("Northern Sami")] = wxT("SE");
		s_languageMap[wxT("Samoan")] = wxT("SM");
		s_languageMap[wxT("Sango")] = wxT("SG");
		s_languageMap[wxT("Serbian")] = wxT("SR");
		s_languageMap[wxT("Scottish Gaelic; Gaelic")] = wxT("GD");
		s_languageMap[wxT("Shona")] = wxT("SN");
		s_languageMap[wxT("Sinhala, Sinhalese")] = wxT("SI");
		s_languageMap[wxT("Slovak")] = wxT("SK");
		s_languageMap[wxT("Slovene")] = wxT("SL");
		s_languageMap[wxT("Somali")] = wxT("SO");
		s_languageMap[wxT("Southern Sotho")] = wxT("ST");
		s_languageMap[wxT("Spanish; Castilian")] = wxT("ES");
		s_languageMap[wxT("Sundanese")] = wxT("SU");
		s_languageMap[wxT("Swahili")] = wxT("SW");
		s_languageMap[wxT("Swati")] = wxT("SS");
		s_languageMap[wxT("Swedish")] = wxT("SV");
		s_languageMap[wxT("Tamil")] = wxT("TA");
		s_languageMap[wxT("Telugu")] = wxT("TE");
		s_languageMap[wxT("Tajik")] = wxT("TG");
		s_languageMap[wxT("Thai")] = wxT("TH");
		s_languageMap[wxT("Tigrinya")] = wxT("TI");
		s_languageMap[wxT("Tibetan")] = wxT("BO");
		s_languageMap[wxT("Turkmen")] = wxT("TK");
		s_languageMap[wxT("Tagalog")] = wxT("TL");
		s_languageMap[wxT("Tswana")] = wxT("TN");
		s_languageMap[wxT("Tonga (Tonga Islands)")] = wxT("TO");
		s_languageMap[wxT("Turkish")] = wxT("TR");
		s_languageMap[wxT("Tsonga")] = wxT("TS");
		s_languageMap[wxT("Tatar")] = wxT("TT");
		s_languageMap[wxT("Twi")] = wxT("TW");
		s_languageMap[wxT("Tahitian")] = wxT("TY");
		s_languageMap[wxT("Uighur, Uyghur")] = wxT("UG");
		s_languageMap[wxT("Ukrainian")] = wxT("UK");
		s_languageMap[wxT("Urdu")] = wxT("UR");
		s_languageMap[wxT("Uzbek")] = wxT("UZ");
		s_languageMap[wxT("Venda")] = wxT("VE");
		s_languageMap[wxT("Vietnamese")] = wxT("VI");
		s_languageMap[wxT("Volapuk")] = wxT("VO");
		s_languageMap[wxT("Walloon")] = wxT("WA");
		s_languageMap[wxT("Welsh")] = wxT("CY");
		s_languageMap[wxT("Wolof")] = wxT("WO");
		s_languageMap[wxT("Western Frisian")] = wxT("FY");
		s_languageMap[wxT("Xhosa")] = wxT("XH");
		s_languageMap[wxT("Yiddish")] = wxT("YI");
		s_languageMap[wxT("Yoruba")] = wxT("YO");
		s_languageMap[wxT("Zhuang, Chuang")] = wxT("ZA");
		s_languageMap[wxT("Zulu")] = wxT("ZU");
	}
	return s_languageMap;
}

static wxArrayString s_languageNames;

wxArrayString& DVD::GetLanguageNames() {
	if (s_languageNames.size() == 0) {
		map<wxString, wxString>& langMap = DVD::GetLanguageMap();
		for (map<wxString, wxString>::iterator it = langMap.begin(); it != langMap.end(); it++) {
			s_languageNames.Add(it->first);
		}
	}
	return s_languageNames;
}

const wxString charsetsArray[] = {
	wxT("ASCII"), wxT("BIG5"), wxT("BIG5-HKSCS"), wxT("BIG5-HKSCS:2001"), wxT("BIG5-HKSCS:1999"),
	wxT("CP850"), wxT("CP862"), wxT("CP866"), wxT("CP874"), wxT("CP932"), wxT("CP936"), wxT("CP949"), wxT("CP950"),
	wxT("CP1131"), wxT("CP1133"), wxT("CP1250"), wxT("CP1251"), wxT("CP1252"),wxT("CP1133"),
	wxT("CP1253"), wxT("CP1254"), wxT("CP1255"), wxT("CP1256"), wxT("CP1257"), wxT("CP1258"),
	wxT("EUC-CN"), wxT("EUC-JP"), wxT("EUC-KR"), wxT("EUC-TW"),
	wxT("GB18030"), wxT("GBK"), wxT("Georgian-Academy"), wxT("Georgian-PS"), wxT("HZ"),
	wxT("ISO-2022-CN"), wxT("ISO-2022-CN-EXT"), wxT("ISO-2022-JP"), wxT("ISO-2022-JP-2"), wxT("ISO-2022-JP-1"),
	wxT("ISO-2022-KR"), wxT("ISO-8859-1"), wxT("ISO-8859-2"), wxT("ISO-8859-3"), wxT("ISO-8859-4"), wxT("ISO-8859-5"),
	wxT("ISO-8859-6"), wxT("ISO-8859-7"), wxT("ISO-8859-8"), wxT("ISO-8859-9"), wxT("ISO-8859-10"), wxT("ISO-8859-11"),
	wxT("ISO-8859-13"), wxT("ISO-8859-14"), wxT("ISO-8859-15"), wxT("ISO-8859-16"),
	wxT("JOHAB"), wxT("KOI8-R"), wxT("KOI8-RU"), wxT("KOI8-T"), wxT("KOI8-U"),
	wxT("Macintosh"), wxT("MacArabic"), wxT("MacCentralEurope"), wxT("MacCroatian"), wxT("MacCyrillic"),
	wxT("MacGreek"), wxT("MacHebrew"), wxT("MacIceland"), wxT("MacRoman"), wxT("MacRomania"), wxT("MacThai"),
	wxT("MacTurkish"), wxT("MacUkraine"), wxT("MuleLao-1"),
	wxT("PT154"), wxT("RK1048"), wxT("SHIFT_JIS"), wxT("TCVN"), wxT("TIS-620"),
	wxT("UCS-2"), wxT("UCS-2BE"), wxT("UCS-2LE"), wxT("UCS-4"), wxT("UCS-4BE"), wxT("UCS-4LE"),
	wxT("UTF-7"), wxT("UTF-8"), wxT("UTF-16"), wxT("UTF-16BE"), wxT("UTF-16LE"),
	wxT("UTF-32"), wxT("UTF-32BE"), wxT("UTF-32LE"), wxT("VISCII")
};
static wxArrayString s_charsets(80, charsetsArray);

wxArrayString& DVD::GetCharsets() {
	return s_charsets;
}
