/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * OCL Compiler                                                      *
 * Copyright (C) 1999, 2000 Frank Finger (frank@finger.org).         *
 * All rights reserved.                                              *
 *                                                                   *
 * This work was done as a diploma project at the Chair for Software *
 * Technology, Dresden University Of Technology, Germany             *
 * (http://www-st.inf.tu-dresden.de).  It is understood that any     *
 * modification not identified as such is not covered by the         *
 * preceding statement.                                              *
 *                                                                   *
 * This work is free software; you can redistribute it and/or        *
 * modify it under the terms of the GNU Library General Public       *
 * License as published by the Free Software Foundation; either      *
 * version 2 of the License, or (at your option) any later version.  *
 *                                                                   *
 * This work is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU *
 * Library General Public License for more details.                  *
 *                                                                   *
 * You should have received a copy of the GNU Library General Public *
 * License along with this library; if not, write to the             *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,      *
 * Boston, MA  02111-1307, USA.                                      *
 *                                                                   *
 * To submit a bug report, send a comment, or get the latest news on *
 * this project and other projects, please visit the web site:       *
 * http://www-st.inf.tu-dresden.de/ (Chair home page) or             *
 * http://www-st.inf.tu-dresden.de/ocl/ (project home page)          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

package tudresden.ocl.parser;

import tudresden.ocl.parser.node.*;
import tudresden.ocl.parser.analysis.*;
import tudresden.ocl.parser.parser.*;
import tudresden.ocl.parser.lexer.*;
import tudresden.ocl.OnlyNameFinder;

import java.util.*;

/** This class adapts the Parser class generated by SableCC to produce
 *  abstract syntax trees (instead of concrete syntax trees). It is necessary
 *  because the OCL grammar is not LALR(1)-parsable.
 */
public class OclParser extends Parser {

  protected AstFix fix=new AstFix();
  protected Node oclNode;

  public OclParser(Lexer lexer) {
    super(lexer);
  }

  protected void filter() {
    oclNode=node;
    oclNode.apply(fix);
    node=oclNode;
  }

  class AstFix extends AnalysisAdapter {

    public void caseAEmptyFeatureCallParameters(AEmptyFeatureCallParameters efcp) {
      oclNode = new AFeatureCallParameters(efcp.getLPar(), null, null, efcp.getRPar());
    }

    public void caseAConcreteFeatureCallParameters(AConcreteFeatureCallParameters cfcp) {
      boolean isDeclarator=false;
      boolean isIterateDeclarator=false;

      LinkedList tail=cfcp.getFcpHelper();
      PFcpHelper[] fcpHelpers=new PFcpHelper[tail.size()];
      Iterator iter=tail.iterator();
      int i=0;
      while (iter.hasNext()) {
        PFcpHelper fh=(PFcpHelper) iter.next();
        fcpHelpers[i]=fh;
        i++;
        if (fh instanceof ACommaFcpHelper) {
        } else if (fh instanceof AColonFcpHelper) {
        } else if (fh instanceof AIterateFcpHelper) {
          isIterateDeclarator=true;
        } else if (fh instanceof ABarFcpHelper) {
          isDeclarator=true;
        } else {
          throw new RuntimeException("unknown subclass of PFcpHelper");
        }
      }

      if (isIterateDeclarator && ! isDeclarator) {
        throw new OclParserException(
          "parser error: illegal feature call parameters format in \""+cfcp+"\"; "+
          "must contain \";\" only if it contains \"|\""
        );
      }
      AFeatureCallParameters fcp;
      if (isIterateDeclarator) {
        fcp=getFcpWithIterateDeclarator(cfcp, (AExpression)cfcp.getExpression(), fcpHelpers);
      } else if (isDeclarator) {
        fcp=getFcpWithStandardDeclarator(cfcp, (AExpression)cfcp.getExpression(), fcpHelpers);
      } else {
        fcp=getFcpWithoutDeclarator(cfcp, (AExpression)cfcp.getExpression(), fcpHelpers);
      }
      oclNode=fcp;
    }

    protected AFeatureCallParameters getFcpWithIterateDeclarator(
            AConcreteFeatureCallParameters cfcp, AExpression expr, PFcpHelper[] fcpHelpers ) {
      AIterateDeclarator iDecl=new AIterateDeclarator();
      if (fcpHelpers.length != 3 ||
          ! (fcpHelpers[0] instanceof AColonFcpHelper) ||
          ! (fcpHelpers[1] instanceof AIterateFcpHelper) ||
          ! (fcpHelpers[2] instanceof ABarFcpHelper) ) {
        throw new OclParserException(
          "parser error: feature call parameters with iterate declarator must have the format "+
          "\"( name: type; name: type = expr | expr )\""
        );
      }
      AColonFcpHelper fcpHelper0=(AColonFcpHelper)fcpHelpers[0];
      AIterateFcpHelper fcpHelper1=(AIterateFcpHelper)fcpHelpers[1];
      ABarFcpHelper fcpHelper2=(ABarFcpHelper)fcpHelpers[2];
      iDecl.setIterator( getOnlyNameIn(expr) );
      iDecl.setIterType(
        new ADeclaratorTypeDeclaration(
          fcpHelper0.getColon(),
          fcpHelper0.getSimpleTypeSpecifier()
        )
      );
      iDecl.setSemicolon( fcpHelper1.getSemicolon() );
      iDecl.setAccumulator( fcpHelper1.getName() );
      iDecl.setAccType(
        new ADeclaratorTypeDeclaration(
          fcpHelper1.getColon(),
          fcpHelper1.getSimpleTypeSpecifier()
        )
      );
      iDecl.setEqual(fcpHelper1.getEqual());
      iDecl.setExpression(fcpHelper1.getExpression());
      iDecl.setBar(
        fcpHelper2.getBar()
      );
      AActualParameterList params=new AActualParameterList(
        fcpHelper2.getExpression(),
        new LinkedList() // empty actual_parameter_list_tail
      );
      AFeatureCallParameters result=new AFeatureCallParameters(
        cfcp.getLPar(),
        iDecl,
        params,
        cfcp.getRPar()
      );
      return result;
    }

    protected AFeatureCallParameters getFcpWithStandardDeclarator(
            AConcreteFeatureCallParameters cfcp, AExpression expr, PFcpHelper[] fcpHelpers ) {
      int numHelpers=fcpHelpers.length;
      boolean ok=true;
      boolean hasTypeDeclarator;
      if (numHelpers<1) {
        ok=false;
      } else if (! (fcpHelpers[numHelpers-1] instanceof ABarFcpHelper)) {
        ok=false;
      }
      for (int i=0; i<numHelpers-2; i++) {
        if (! (fcpHelpers[i] instanceof ACommaFcpHelper)) {
          ok=false;
        }
      }
      if (numHelpers-2 >= 0) {
        if (! (fcpHelpers[numHelpers-2] instanceof ACommaFcpHelper ||
               fcpHelpers[numHelpers-2] instanceof AColonFcpHelper) ) {
          ok=false;
        }
      }
      if (!ok) {
        throw new OclParserException(
          "parser error: feature call parameters with standard declarator must have the format "+
          "\"( name, ... , name: type | expr )\""
        );
      }
      LinkedList declaratorTail=new LinkedList();
      ADeclaratorTypeDeclaration typeDecl;
      for (int i=0; i<numHelpers-2; i++) {
        ACommaFcpHelper fcpHelperI=(ACommaFcpHelper)fcpHelpers[i];
        ADeclaratorTail dt=new ADeclaratorTail(
          fcpHelperI.getComma(),
          getOnlyNameIn(fcpHelperI.getExpression())
        );
        declaratorTail.addLast(dt);
      }
      if (numHelpers>1) {
        if (fcpHelpers[numHelpers-2] instanceof ACommaFcpHelper) {
          ACommaFcpHelper fcpHelperNm2=(ACommaFcpHelper)fcpHelpers[numHelpers-2];
          ADeclaratorTail dt=new ADeclaratorTail(
            fcpHelperNm2.getComma(),
            getOnlyNameIn(fcpHelperNm2.getExpression())
          );
          declaratorTail.addLast(dt);
          typeDecl=null;
        } else {
          AColonFcpHelper fcpHelperNm2=(AColonFcpHelper)fcpHelpers[numHelpers-2];
          typeDecl=new ADeclaratorTypeDeclaration(
            fcpHelperNm2.getColon(),
            fcpHelperNm2.getSimpleTypeSpecifier()
          );
        }
      } else {
        typeDecl=null;
      }
      ABarFcpHelper barHelper=(ABarFcpHelper)fcpHelpers[numHelpers-1];
      AStandardDeclarator sDecl=new AStandardDeclarator(
        getOnlyNameIn(expr),
        declaratorTail,
        typeDecl,
        barHelper.getBar()
      );
      AActualParameterList params=new AActualParameterList(
        barHelper.getExpression(),
        new LinkedList() // empty actual_parameter_list_tail
      );
      AFeatureCallParameters result=new AFeatureCallParameters(
        cfcp.getLPar(),
        sDecl,
        params,
        cfcp.getRPar()
      );
      return result;
    }

    protected AFeatureCallParameters getFcpWithoutDeclarator(
            AConcreteFeatureCallParameters cfcp, AExpression expr, PFcpHelper[] fcpHelpers ) {
      LinkedList paramList=new LinkedList();
      for (int i=0; i<fcpHelpers.length; i++) {
        if (! (fcpHelpers[i] instanceof ACommaFcpHelper)) {
          throw new OclParserException(
            "parser error: declarator-less feature call paramaters must have the format "+
            "\"( expr, ..., expr )\""
          );
        } else {
          ACommaFcpHelper cfh=(ACommaFcpHelper)fcpHelpers[i];
          AActualParameterListTail aplt=new AActualParameterListTail(
            cfh.getComma(),
            cfh.getExpression()
          );
          paramList.add(aplt);
        }
      }
      AActualParameterList params=new AActualParameterList(expr, paramList);
      AFeatureCallParameters result=new AFeatureCallParameters(
        cfcp.getLPar(),
        null,
        params,
        cfcp.getRPar()
      );
      return result;
    }

    protected TName getOnlyNameIn(PExpression expr) {
      OnlyNameFinder onf=new OnlyNameFinder(false);
      expr.apply(onf);
      return onf.getName();
    }

    /*protected PSimpleTypeSpecifier getOnlySimpleTypeSpecifierIn(PExpression expr) {
      return null;
    }*/
  }

}



