(** Reading PDF files *)

exception PDFReadError of string
exception PDFSemanticError of string
(** There can be many things malformed with a PDF file, in addition to the
usual difficulties concerned with file I/O. CamlPDF wraps all these up in these
two exceptions. *)

(** Read a PDF from an [Io.input], with an optional user password which, if
absent, is assumed to be the empty string. *)
val pdf_of_input : string option -> Pdfio.input -> Pdf.pdfdoc

(** Same, but delay loading of streams and parsing of objects. *)
val pdf_of_input_lazy : string option-> Pdfio.input -> Pdf.pdfdoc

(** Read a PDF from an OCaml channel. *)
val pdf_of_channel : string option -> in_channel -> Pdf.pdfdoc

(** Same, but delay loading of streams and parsing of objects. *)
val pdf_of_channel_lazy : string option -> in_channel -> Pdf.pdfdoc

(** Read a PDF from the given filename. *)
val pdf_of_file : string option -> string -> Pdf.pdfdoc

(**/**)
(* For internal use by other parts of the library *)

val read_header : (Pdfio.input -> int * int)

type lexeme =
  | LexNull
  | LexBool of bool
  | LexInt of int
  | LexReal of float
  | LexString of string
  | LexName of string
  | LexLeftSquare
  | LexRightSquare
  | LexLeftDict
  | LexRightDict
  | LexStream of Pdf.stream
  | LexEndStream
  | LexObj
  | LexEndObj
  | LexR
  | LexComment
  | StopLexing
  | LexNone

val lex_stream_data : Pdfio.input -> int -> bool -> lexeme

val getuntil_white_or_delimiter : (Pdfio.input -> char list)

val lex_name : Pdfio.input -> lexeme

val lex_number : Pdfio.input -> lexeme

val lex_string : Pdfio.input -> lexeme

val lex_hexstring : Pdfio.input -> lexeme

val lex_comment : Pdfio.input -> lexeme

val lex_dictionary : Pdfio.input -> lexeme list

val parse : lexeme list -> int * Pdf.pdfobject

val dropwhite : Pdfio.input -> unit

val print_lexeme : lexeme -> unit

val string_of_lexeme : lexeme -> string

(** Return encryption method in use *)
val what_encryption : Pdf.pdfdoc -> Pdfwrite.encryption_method option

(** Return list of permissions *)
val permissions : Pdf.pdfdoc -> Pdfcrypt.permission list

