/* whoopsie
 * 
 * Copyright © 2011-2012 Canonical Ltd.
 * Author: Evan Dandrea <evan.dandrea@canonical.com>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _XOPEN_SOURCE

#include <stdlib.h>
#include <gio/gio.h>
#include <glib.h>
#include <nm-client.h>
#include <nm-device-wifi.h>
#include <nm-device-ethernet.h>
#include "connectivity.h"

static gboolean route_available = FALSE;
static gboolean network_available = FALSE;

struct _connectivity_data {
    ConnectionAvailableCallback callback;
    const char* url;
};

static struct _connectivity_data connectivity_data;
static NMClient* nm_client = NULL;
static int subscription_id = 0;

void
route_changed (GNetworkMonitor *nm, gboolean available, gpointer user_data)
{
    GError* err = NULL;
    GSocketConnectable *addr = NULL;

    g_return_if_fail (nm);

    if (!available) {
        route_available = FALSE;
        return;
    }

    addr = g_network_address_parse_uri (connectivity_data.url, 80, &err);
    if (!addr) {
        g_print ("Could not parse crash database URL: %s\n", err->message);
        g_error_free (err);
        return;
    }

    route_available = g_network_monitor_can_reach (nm, addr, NULL, NULL);

    if (network_available && route_available)
        connectivity_data.callback (TRUE);

    g_object_unref (addr);
}

void
setup_network_route_monitor (void)
{
    GNetworkMonitor* nm = NULL;
    GSocketConnectable *addr = NULL;
    GError* err = NULL;

    /* Using GNetworkMonitor brings in GSettings, which brings in DConf, which
     * brings in a DBus session bus, which brings in pain. */
    if (putenv ("GSETTINGS_BACKEND=memory") != 0)
        g_print ("Could not set the GSettings backend to memory.\n");

    addr = g_network_address_parse_uri (connectivity_data.url, 80, &err);
    if (!addr) {
        g_print ("Could not parse crash database URL: %s\n", err->message);
        g_error_free (err);
        return;
    }

    nm = g_network_monitor_get_default ();
    if (!nm) {
        g_print ("Could not get the network monitor.\n");
        goto out;
    }

    route_available = (g_network_monitor_get_network_available (nm) &&
                       g_network_monitor_can_reach (nm, addr, NULL, NULL));

    g_signal_connect (nm, "network-changed", G_CALLBACK (route_changed), NULL);

out:
    if (addr)
        g_object_unref (addr);
}

gboolean
is_potentially_paid_connection (NMActiveConnection* connection)
{
    int i = 0;
    const GPtrArray* devices = NULL;
    NMDevice* device = NULL;
    devices = nm_active_connection_get_devices (connection);
    for (i=0; i < devices->len; i++) {
        device = g_ptr_array_index (devices, i);
        if (!(NM_IS_DEVICE_ETHERNET (device) || NM_IS_DEVICE_WIFI (device))) {
            return TRUE;
        }
    }
    return FALSE;
}

void
nm_state_change (NMClient *client, const GParamSpec *pspec, gpointer user_data)
{
    int i = 0;
    const GPtrArray* active_connections = NULL;
    NMActiveConnection* active_connection = NULL;
    gboolean paid = FALSE;

    ConnectionAvailableCallback callback = user_data;

    if (nm_client_get_state (client) != NM_STATE_CONNECTED_GLOBAL) {
        network_available = FALSE;
        callback (network_available);
        return;
    }

    active_connections = nm_client_get_active_connections (client);
    for (i=0; i < active_connections->len; i++) {
        active_connection = g_ptr_array_index (active_connections, i);
        if (nm_active_connection_get_default (active_connection)) {
            if (is_potentially_paid_connection (active_connection)) {
                paid = TRUE;
            }
            break;
        }
    }

    if (paid) {
        network_available = FALSE;
        return;
    }

    for (i=0; i < active_connections->len; i++) {
        active_connection = g_ptr_array_index (active_connections, i);
        if (nm_active_connection_get_default6 (active_connection)) {
            if (is_potentially_paid_connection (active_connection)) {
                paid = TRUE;
            }
            break;
        }
    }

    if (paid) {
        network_available = FALSE;
        return;
    }

    network_available = TRUE;
    callback (network_available && route_available);

}

gboolean
monitor_connectivity (const char* crash_url, ConnectionAvailableCallback callback)
{
    g_return_val_if_fail (crash_url, FALSE);
    g_return_val_if_fail (!nm_client, FALSE);
    g_return_val_if_fail (!subscription_id, FALSE);

    connectivity_data.url = crash_url;
    connectivity_data.callback = callback;
    /* Checking whether a NetworkManager connection is not enough.
     * NetworkManager will report CONNECTED_GLOBAL when a route is not present
     * when the connectivity option is not set. We'll use GNetworkMonitor here
     * to fill in the gap. */
    setup_network_route_monitor ();

    nm_client = nm_client_new ();
    if (!nm_client)
        return FALSE;
    subscription_id = g_signal_connect (G_OBJECT (nm_client),
                                        "notify::"
                                        NM_CLIENT_STATE,
                                        G_CALLBACK (nm_state_change),
                                        callback);

    /* We don't need to now call nm_state_change manually, as connecting the
     * signal will force a call of it, apparently. */

    return TRUE;
}

void
unmonitor_connectivity (void)
{
    if (nm_client) {
        g_signal_handler_disconnect (G_OBJECT (nm_client), subscription_id);
        g_object_unref (nm_client);
    }
}
