/*****************************************************************************
 *
 * grail - Gesture Recognition And Instantiation Library
 *
 * Copyright (C) 2010-2012 Canonical Ltd.
 *
 * This library is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License version 3
 * as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "grail-impl.h"
#include <stdlib.h>
#include <string.h>
#include <math.h>

static void set_center_velocity_and_radius(struct grail_impl *impl,
					   struct grail_element *slot)
{
	const struct utouch_contact **tc = slot->touches;
	double x, y, vx, vy, r2, dx, dy;
	int i;

	switch (slot->num_touches) {
	case 1:
		x = tc[0]->x;
		y = tc[0]->y;
		vx = tc[0]->vx;
		vy = tc[0]->vy;
		r2 = 0;
		break;
	case 2:
		dx = 0.5 * (tc[1]->x - tc[0]->x);
		dy = 0.5 * (tc[1]->y - tc[0]->y);
		x = tc[0]->x + dx;
		y = tc[0]->y + dy;
		vx = 0.5 * (tc[0]->vx + tc[1]->vx);
		vy = 0.5 * (tc[0]->vy + tc[1]->vy);
		r2 = dx * dx + dy * dy;
		break;
	default:
		x = y = vx = vy = r2 = 0;
		for (i = 0; i < slot->num_touches; i++) {
			x += tc[i]->x;
			y += tc[i]->y;
			vx += tc[i]->vx;
			vy += tc[i]->vy;
		}
		x /= slot->num_touches;
		y /= slot->num_touches;
		vx /= slot->num_touches;
		vy /= slot->num_touches;
		for (i = 0; i < slot->num_touches; i++) {
			dx = tc[i]->x - x;
			dy = tc[i]->y - y;
			r2 += dx * dx + dy * dy;
		}
		r2 /= slot->num_touches;
		break;
	}

	slot->center.x = x;
	slot->center.y = y;
	slot->velocity.x = 1000 * vx;
	slot->velocity.y = 1000 * vy;
	slot->radius2 = r2;
}

static void set_transform(struct grail_impl *impl, struct grail_element *slot,
			  double ds, double dc)
{
	const struct grail_element *pslot = slot->prev;
	double dx = slot->center.x - pslot->center.x;
	double dy = slot->center.y - pslot->center.y;
	float *T = slot->transform;

	T[0] = dc;
	T[1] = ds;
	T[2] = dx;
	T[3] = -ds;
	T[4] = dc;
	T[5] = dy;

	slot->drag.x = pslot->drag.x + dx;
	slot->drag.y = pslot->drag.y + dy;
}

static void start_slot(struct grail_impl *impl,
		       struct grail_element *slot,
		       const struct utouch_frame *touch)
{
	float *T = slot->transform;

	slot->id = impl->seqid++ & GRAIL_ID_MAX;
	slot->expect_mask = GRAIL_EXPECT_MASK;
	slot->active_mask = 0;
	slot->start_time = touch->time;
	set_center_velocity_and_radius(impl, slot);
	T[0] = T[4] = 1;
	T[1] = T[2] = T[3] = T[5] = 0;
	slot->drag.x = 0;
	slot->drag.y = 0;
	slot->scale2 = 1;
	slot->angle = 0;
}

static void update_slot(struct grail_impl *impl,
			struct grail_element *slot,
			double ds, double dc)
{
	const struct grail_element *pslot = slot->prev;

	slot->id = pslot->id;
	slot->start_time = pslot->start_time;
	slot->expect_mask = pslot->expect_mask;
	slot->active_mask = pslot->active_mask;

	set_center_velocity_and_radius(impl, slot);
	set_transform(impl, slot, ds, dc);

	slot->scale2 = pslot->scale2 * (ds * ds + dc * dc);
	slot->angle = pslot->angle + ds / dc; /* atan2(ds, dc) */
}

static void stop_slot(struct grail_impl *impl,
		      struct grail_element *slot)
{
	const struct grail_element *pslot = slot->prev;
	float *T = slot->transform;

	slot->id = -1;
	slot->num_touches = 0;
	slot->start_time = pslot->start_time;
	slot->expect_mask = 0;
	slot->active_mask = pslot->active_mask;
	slot->center = pslot->center;
	slot->velocity = pslot->velocity;
	slot->radius2 = pslot->radius2;
	T[0] = T[4] = 1;
	T[1] = T[2] = T[3] = T[5] = 0;
	slot->drag = pslot->drag;
	slot->scale2 = pslot->scale2;
	slot->angle = pslot->angle;
}

static void set_slot_one(struct grail_impl *impl,
			 struct grail_element *slot,
			 const struct utouch_frame *touch,
			 const struct utouch_contact *t1)
{
	const struct grail_element *pslot = slot->prev;
	const struct utouch_contact *p1 = pslot->touches[0];

	if (!t1->active) {
		stop_slot(impl, slot);
		return;
	}

	slot->touches[0] = t1;
	slot->num_touches = 1;

	if (pslot->num_touches != slot->num_touches || t1->id != p1->id) {
		start_slot(impl, slot, touch);
		return;
	}

	update_slot(impl, slot, 0, 1);
}

static void set_slot_two(struct grail_impl *impl,
			 struct grail_element *slot,
			 const struct utouch_frame *touch,
			 const struct utouch_contact *t1,
			 const struct utouch_contact *t2)
{
	const struct grail_element *pslot = slot->prev;
	const struct utouch_contact *p1 = pslot->touches[0];
	const struct utouch_contact *p2 = pslot->touches[1];
	double tx, ty, px, py, d2;

	if (!t1->active || !t2->active) {
		stop_slot(impl, slot);
		return;
	}

	slot->touches[0] = t1;
	slot->touches[1] = t2;
	slot->num_touches = 2;

	if (pslot->num_touches != slot->num_touches ||
	    t1->id != p1->id || t2->id != p2->id) {
		start_slot(impl, slot, touch);
		return;
	}

	tx = t2->x - t1->x;
	ty = t2->y - t1->y;
	px = p2->x - p1->x;
	py = p2->y - p1->y;

	d2 = px * px + py * py;
	if (d2 > 0) {
		px /= d2;
		py /= d2;
	}

	update_slot(impl, slot, tx * py - ty * px, tx * px + ty * py);
}

static void set_slot_multi(struct grail_impl *impl,
			   struct grail_element *slot,
			   struct grail_frame *frame,
			   const struct utouch_frame *touch)
{
	const struct grail_element *pslot = slot->prev;
	struct grail_element **slots = frame->slots;
	int i, j, n = impl->num_touches;
	struct grail_element *best = 0;

	if (touch->num_active < 3) {
		stop_slot(impl, slot);
		return;
	}

	memcpy(slot->touches, touch->active,
	       touch->num_active * sizeof(slot->touches[0]));
	slot->num_touches = touch->num_active;

	if (pslot->num_touches != slot->num_touches) {
		start_slot(impl, slot, touch);
		return;
	}

	for (i = 0; i < slot->num_touches; i++) {
		if (slot->touches[i]->id != pslot->touches[i]->id) {
			start_slot(impl, slot, touch);
			return;
		}
	}

	for (i = 0; i < impl->num_touches; i++) {
		for (j = i + 1; j < impl->num_touches; j++) {
			struct grail_element *s = slots[n++];
			if (!s->num_touches)
				continue;
			if (!best || s->radius2 > best->radius2)
				best = s;
		}
	}

	update_slot(impl, slot, best->transform[1], best->transform[0]);
}

/**
 * Determine the center of rotation point.
 *
 * For any given point q that is transformed by a 2D affine transformation
 * matrix T about anchor point P the new point q' may be determined by the
 * following equation:
 *
 * q' = T * (q - P) + P
 *
 * T and P are dependent, so we can modify one and find a new value for the
 * other. We will label the original T and P as T0 and P0, and the new values
 * will be labeled T1 and P1. We can find new values by solving the following
 * equation:
 *
 * q' = T0 * (q - P0) + P0 = T1 * (q - P1) + P1
 *
 * In the calculations below, we use variables for the scalar values
 * that make up T0, P0, T1, and P1:
 *
 * T0: [ a -b c ]  P0: [ x0 ]  T1: [ a -b 0 ]  P1: [ x1 ]
 *     [ b  a d ]      [ y0 ]      [ b  a 0 ]      [ y1 ]
 *     [ 0  0 1 ]      [  0 ]      [ 0  0 1 ]      [  0 ]
 *
 * Note that rotation and scaling are independent of the anchor point, so a and
 * b are equivalent between the transformation matrices.
 *
 * Since we know all the values of T0, P0, and T1, we can calculate the values
 * x1 and y1 in P1.
 */
static inline int set_center_of_rotation(struct grail_element *e)
{
	float a = e->transform[0];
	float b = e->transform[3];
	float c = e->transform[2];
	float d = e->transform[5];
	float x0 = e->center.x;
	float y0 = e->center.y;
	float x1;
	float y1;

	float div = a*a - 2*a + b*b + 1;

	if (fabsf(div) < 1e-5)
		return 0;

	x1 = (a*a*x0 - a*(2*x0+c) + b*b*x0 - b*d + c + x0) / div;
	y1 = (a*a*y0 - a*(2*y0+d) + b*b*y0 + b*c + d + y0) / div;

	e->rotation_center.x = x1;
	e->rotation_center.y = y1;

	return 1;
}

static void set_slots(struct grail_impl *impl,
		      struct grail_frame *frame,
		      const struct utouch_frame *touch)
{
	struct grail_element **slots = frame->slots;
	struct utouch_contact *const *tc = touch->slots;
	int i, j, n = 0;

	for (i = 0; i < impl->num_touches; i++)
		set_slot_one(impl, slots[n++], touch, tc[i]);

	for (i = 0; i < impl->num_touches; i++)
		for (j = i + 1; j < impl->num_touches; j++)
			set_slot_two(impl, slots[n++], touch, tc[i], tc[j]);

	set_slot_multi(impl, slots[n++], frame, touch);
}

static void collect_transforms(struct grail_impl *impl,
			       struct grail_frame *frame,
			       const struct utouch_frame *touch)
{
	const struct utouch_surface *s = utouch_frame_get_surface(impl->fh);
	const struct grail_control *ctl = impl->ctl;
	float d_x = ctl->bar_drag_x * (s->mapped_max_x - s->mapped_min_x);
	float d_y = ctl->bar_drag_y * (s->mapped_max_y - s->mapped_min_y);
	float ds2 = ctl->bar_scale * ctl->bar_scale;
	float dt;
	int i;

	for (i = 0; i < impl->num_slots; i++) {
		struct grail_element *s = frame->slots[i];

		if (!s->num_touches)
			continue;

		set_center_of_rotation(s);

		dt = touch->time - s->start_time;
		if (dt > ctl->glue_ms) {
			unsigned int mask = s->active_mask;

			if (fabs(s->drag.x) > d_x)
				mask |= GRAIL_EXPECT_DRAG_X;
			if (fabs(s->drag.y) > d_y)
				mask |= GRAIL_EXPECT_DRAG_Y;
			if (fabs(s->scale2 - 1) > ds2)
				mask |= GRAIL_EXPECT_SCALE;
			if (fabs(s->angle) > ctl->bar_angle)
				mask |= GRAIL_EXPECT_ANGLE;

			s->active_mask = mask;

			if (dt < ctl->drop_x_ms)
				mask |= GRAIL_EXPECT_DRAG_X;
			if (dt < ctl->drop_y_ms)
				mask |= GRAIL_EXPECT_DRAG_Y;
			if (dt < ctl->drop_scale_ms)
				mask |= GRAIL_EXPECT_SCALE;
			if (dt < ctl->drop_angle_ms)
				mask |= GRAIL_EXPECT_ANGLE;

			s->expect_mask &= mask;
		}

		frame->ongoing[frame->num_ongoing++] = s;
	}
}

const struct grail_frame GRAIL_PUBLIC *
grail_pump_frame(grail_handle ge, const struct utouch_frame *touch)
{
	struct grail_impl *impl = ge->impl;
	struct grail_frame *frame = impl->frames[impl->nextframe];
	const struct grail_frame *prev = frame->prev;
	int i;

	if (touch->slot_revision == touch->prev->slot_revision &&
	    !prev->num_ongoing)
		return 0;

	frame->touch = touch;
	frame->num_ongoing = 0;
	for (i = 0; i < impl->num_slots; i++)
		frame->slots[i]->prev = prev->slots[i];

	set_slots(impl, frame, touch);
	collect_transforms(impl, frame, touch);

	impl->nextframe = (impl->nextframe + 1) % impl->num_frames;

	return frame;
}
