/**
 * @file geis_dbus_server.c
 * @brief internal GEIS DBus Server 
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include "geis_backend_protected.h"
#include "geis_logging.h"
#include "geis_private.h"
#include "geis_dbus.h"
#include <stdlib.h>
#include <stdio.h>
#include <dbus/dbus.h>

#include "geis_dbus_server.h"


static GeisDBus s_geis_dbus =
{
  NULL,
  -1,
  "",
  ""
};


void
geis_server_finalize(GeisDBus *gdbus)
{
  dbus_connection_unref(gdbus->conn); 
  geis_debug("DBus object finalized");
}


GeisDBus *
geis_dbus_request_name()
{
  if (!s_geis_dbus.conn)
  {
    DBusError error;
    dbus_error_init(&error);

    s_geis_dbus.conn = dbus_bus_get(DBUS_BUS_SESSION, &error);
    if (!s_geis_dbus.conn || dbus_error_is_set(&error))
    {
      geis_debug("error connecting to system bus: %s (%s)\n",
                 error.name, error.message);
      dbus_error_free(&error);
      return NULL;
    }

    dbus_connection_set_exit_on_disconnect(s_geis_dbus.conn, FALSE);

    if (!dbus_connection_get_unix_fd(s_geis_dbus.conn,
                                     &s_geis_dbus.fd))
    {
      geis_debug("could not get descriptor for system bus.\n");
      dbus_error_free(&error);
      dbus_connection_unref(s_geis_dbus.conn);
      s_geis_dbus.conn = NULL;
      return NULL;
    }

    snprintf(s_geis_dbus.busname, sizeof(s_geis_dbus.busname),
						GEIS_DBUS_BUS_NAME);
    dbus_bus_request_name(s_geis_dbus.conn, s_geis_dbus.busname, 
			DBUS_NAME_FLAG_REPLACE_EXISTING , &error);
    if (dbus_error_is_set(&error))
    {
            geis_debug("error requesting %s: %s (%s)\n", s_geis_dbus.busname, error.name, error.message);
            dbus_error_free(&error);
            dbus_connection_unref(s_geis_dbus.conn);
            s_geis_dbus.conn = NULL;
            return NULL;
    }
    dbus_error_free(&error);
  }
  return &s_geis_dbus;
}


void
geis_dbus_reply_method_call(DBusMessage* msg, DBusConnection* conn)
{
  DBusMessage* reply;
  DBusMessageIter args;
  int stat = 1;
  dbus_uint32_t level = 21614;
  dbus_uint32_t serial = 0;
  char* param = "";

  if (!dbus_message_iter_init(msg, &args))
    ;
  else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
    ;
  else 
      dbus_message_iter_get_basic(&args, &param);

   reply = dbus_message_new_method_return(msg);

   dbus_message_iter_init_append(reply, &args);
   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_BOOLEAN, &stat)) { 
     geis_error("Failed to append argument");
   }
   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT32, &level)) { 
     geis_error("Failed to append argument");
   }

   if (!dbus_connection_send(conn, reply, &serial)) {
     geis_error("Failed to send reply");
   }
   dbus_connection_flush(conn);

   dbus_message_unref(reply);
}


void
geis_dbus_dispatch(GeisDBus *gdbus)
{
  DBusMessage *msg;

  if (gdbus->conn)
  {
    do
    {
      if (dbus_connection_read_write_dispatch(gdbus->conn, 0))
      {
        msg = dbus_connection_pop_message(gdbus->conn);
        if (dbus_message_is_method_call(msg, GEIS_DBUS_BUS_NAME, GEIS_REMOTE_FUNCTION)) 
               geis_dbus_reply_method_call(msg, gdbus->conn);
            dbus_message_unref(msg);
      }        
    } while (gdbus->conn
         && dbus_connection_get_dispatch_status(gdbus->conn)
             == DBUS_DISPATCH_DATA_REMAINS);
  }
}


static void
_fd_callback(int                          fd __attribute__((unused)),
             GeisBackendMultiplexorEvent  ev __attribute__((unused)),
             void                         *ctx)
{
  geis_debug("callback invoked");
  GeisDBus *gdbus = (GeisDBus*)ctx;
  geis_dbus_dispatch(gdbus);
}


GeisDBus *
geis_server_dbus_new(Geis geis)
{

  GeisDBus *gdbus = &s_geis_dbus;
  geis_dbus_request_name();

  geis_multiplex_fd(geis, s_geis_dbus.fd, _fd_callback, (void*)gdbus);

  return gdbus;
}

