/**
 * @file libutouch-geis/geis.c
 * @brief implementation of the uTouch GEIS v2.0 API instance
 *
 * Copyright 2010, 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_config.h"
#include "geis_private.h"

#include <errno.h>
#include "geis_atomic.h"
#include "geis_attr.h"
#include "geis_backend.h"
#include "geis_class.h"
#include "geis_device.h"
#include "geis_backend_multiplexor.h"
#include "geis_error.h"
#include "geis_event.h"
#include "geis_event_queue.h"
#include "geis_logging.h"
#include <stdarg.h>
#include <string.h>
#include "test_fixture/geis_backend_test_fixture.h"
#include <unistd.h>
#include "xcb/geis_xcb_backend.h"


struct _Geis
{
  GeisRefCount            refcount;
  GeisErrorStack          error_stack;
  GeisSubBag              subscription_bag;
  GeisBackendMultiplexor  backend_multiplexor;
  GeisBackend             backend;
  GeisEventQueue          input_event_queue;
  int                     input_event_signal_pipe[2];
  GeisEventQueue          output_event_queue;
  GeisEventCallback       output_event_callback;
  void                   *output_event_callback_context;
  GeisDeviceBag           devices;
  GeisEventCallback       device_event_callback;
  void                   *device_event_callback_context;
  GeisGestureClassBag     gesture_classes;
  GeisEventCallback       class_event_callback;
  void                   *class_event_callback_context;
};


/*
 * The default event callback -- just pushes events on the internal queue
 */
static void
_default_output_event_callback(Geis       geis,
                               GeisEvent  event,
                               void      *context __attribute__((unused)))
{
  geis_debug("posting output event");
  geis_event_queue_enqueue(geis->output_event_queue, event);
}


/*
 * Handles device events coming in from the back end.
 */
static GeisBoolean
_device_event_handler(Geis geis, GeisEvent event)
{
  GeisBoolean handled = GEIS_FALSE;
  GeisAttr attr = geis_event_attr_by_name(event, GEIS_EVENT_ATTRIBUTE_DEVICE);
  if (!attr)
  {
    geis_warning("invalid device event received from back end.");
    handled = GEIS_TRUE;
    goto final_exit;
  }
  
  GeisDevice device = geis_attr_value_to_pointer(attr);

  GeisEventType event_type = geis_event_type(event);
  if (event_type == GEIS_EVENT_DEVICE_AVAILABLE)
  {
    geis_device_bag_insert(geis->devices, device);
  }
  else if (event_type == GEIS_EVENT_DEVICE_UNAVAILABLE)
  {
    geis_device_bag_remove(geis->devices, device);
  }

  if (geis->device_event_callback != GEIS_DEFAULT_EVENT_CALLBACK)
  {
    geis->device_event_callback(geis, event, geis->device_event_callback_context);
    handled = GEIS_TRUE;
  }

final_exit:
  return handled;
}


/*
 * Handles class events coming in from the back end.
 */
static GeisBoolean
_class_event_handler(Geis geis, GeisEvent event)
{
  GeisBoolean handled = GEIS_FALSE;

  GeisAttr attr = geis_event_attr_by_name(event, GEIS_EVENT_ATTRIBUTE_CLASS);
  if (!attr)
  {
    geis_warning("invalid class event received from back end.");
    handled = GEIS_TRUE;
    goto final_exit;
  }
  
  GeisGestureClass gesture_class = geis_attr_value_to_pointer(attr);

  GeisEventType event_type = geis_event_type(event);
  if (event_type == GEIS_EVENT_CLASS_AVAILABLE)
  {
    geis_gesture_class_bag_insert(geis->gesture_classes, gesture_class);
  }
  else if (event_type == GEIS_EVENT_CLASS_CHANGED)
  {
    /** @todo implement GEIS_EVENT_CLASS_CHANGED */
  }
  else if (event_type == GEIS_EVENT_CLASS_UNAVAILABLE)
  {
    geis_gesture_class_bag_remove(geis->gesture_classes, gesture_class);
  }

  if (geis->class_event_callback != GEIS_DEFAULT_EVENT_CALLBACK)
  {
    geis->class_event_callback(geis, event, geis->class_event_callback_context);
    handled = GEIS_TRUE;
  }

final_exit:
  return handled;
}


/*
 * Filters and transforms raw gesture events into cooked gesture events.
 *
 * For now, does nothing except copy from the input queueu to the output queue.
 */
static void
_input_event_handler(int fd, GeisBackendMultiplexorEvent mux_ev, void *context)
{
  Geis geis = (Geis)context;

  if (mux_ev == GEIS_BE_MX_READ_AVAILABLE)
  {
    GeisEvent event;

    /* clear the input event signal */
    char buf[2];
    if (read(fd, buf, 1) != 1)
    {
      geis_warning("unexpected number of bytes read from signal pipe");
    }

    geis_debug("input event available");
    event = geis_event_queue_dequeue(geis->input_event_queue);
    if (event)
    {
      GeisBoolean handled = 0;
      switch (geis_event_type(event))
      {
	case GEIS_EVENT_DEVICE_AVAILABLE:
	case GEIS_EVENT_DEVICE_UNAVAILABLE:
	  handled = _device_event_handler(geis, event);
	  break;

	case GEIS_EVENT_CLASS_AVAILABLE:
	case GEIS_EVENT_CLASS_CHANGED:
	case GEIS_EVENT_CLASS_UNAVAILABLE:
	  handled = _class_event_handler(geis, event);
	  break;

	default:
	  break;
      }

      if (!handled)
      {
	geis->output_event_callback(geis,
	                            event,
	                            geis->output_event_callback_context);
      }
    }
  }
}

/**
 * Creates a new empty Geis API instance.
 */
static Geis
geis_new_empty()
{
  geis_error_clear(NULL);
  Geis geis = calloc(1, sizeof(struct _Geis));
  if (!geis)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("calloc failed");
    goto final_exit;
  }

  geis->subscription_bag = geis_subscription_bag_new(1);
  if (!geis->subscription_bag)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("creation of subscroption bag failed");
    free(geis);
    geis = NULL;
    goto unwind_geis;
  }

  geis->backend_multiplexor = geis_backend_multiplexor_new();
  if (!geis->backend_multiplexor)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("creation of back end multiplexor failed");
    goto unwind_subscription_bag;
  }

  geis->input_event_queue = geis_event_queue_new();
  if (!geis->input_event_queue)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("creation of input event queue failed");
    goto unwind_backend_mux;
  }
  if (pipe(geis->input_event_signal_pipe) < 0)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("error %d creating input event signal pipe: %s",
               errno, strerror(errno));
    goto unwind_input_queue;
  }
  geis_backend_multiplexor_add_fd(geis->backend_multiplexor,
                                  geis->input_event_signal_pipe[0],
                                  _input_event_handler,
                                  geis);

  geis->output_event_queue = geis_event_queue_new();
  if (!geis->output_event_queue)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("creation of output event queue failed");
    goto unwind_input_signal_pipe;
  }
  geis->output_event_callback = _default_output_event_callback;

  geis->devices = geis_device_bag_new();
  if (!geis->devices)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("creation of geis device bag failed");
    goto unwind_output_queue;
  }
  geis->device_event_callback = _default_output_event_callback;

  geis->gesture_classes = geis_gesture_class_bag_new();
  if (!geis->gesture_classes)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("creation of geis gesture class bag failed");
    goto unwind_device_bag;
  }
  geis->class_event_callback = _default_output_event_callback;

  goto final_exit;

unwind_device_bag:
  geis_device_bag_delete(geis->devices);
unwind_output_queue:
  geis_event_queue_delete(geis->output_event_queue);
unwind_input_signal_pipe:
  close(geis->input_event_signal_pipe[0]);
  close(geis->input_event_signal_pipe[1]);
unwind_input_queue:
  geis_event_queue_delete(geis->input_event_queue);
unwind_backend_mux:
  geis_backend_multiplexor_delete(geis->backend_multiplexor);
unwind_subscription_bag:
  geis_subscription_bag_delete(geis->subscription_bag);
unwind_geis:
  free(geis);
  geis = NULL;

final_exit:
  return geis;
}


typedef enum _BackendType
{
  BACK_END_TYPE_NONE,
  BACK_END_TYPE_MOCK_ENGINE,
  BACK_END_TYPE_XCB
} BackendType;

/**
 * Sets optional parts of a Geis API instance from a variable argument list.
 */
static GeisBoolean
_set_valist(Geis geis, GeisString init_arg_name, va_list varargs)
{
  GeisBoolean status = GEIS_TRUE;
  BackendType back_end_type = BACK_END_TYPE_NONE;
  GeisBoolean track_devices = GEIS_FALSE;
  GeisBoolean track_classes = GEIS_FALSE;

  while (init_arg_name)
  {
    if (0 == strcmp(init_arg_name, GEIS_INIT_SERVICE_PROVIDER))
    {
      geis_debug("initializing GEIS server");
    }
    else if (0 == strcmp(init_arg_name, GEIS_INIT_TRACK_DEVICES))
    {
      track_devices = GEIS_TRUE;
    }
    else if (0 == strcmp(init_arg_name, GEIS_INIT_TRACK_GESTURE_CLASSES))
    {
      track_classes = GEIS_TRUE;
    }
    else if (0 == strcmp(init_arg_name, GEIS_INIT_UTOUCH_MOCK_ENGINE))
    {
      if (back_end_type != BACK_END_TYPE_NONE)
      {
	geis_error("multiple back ends requested, only using last request");
      }
      back_end_type = BACK_END_TYPE_MOCK_ENGINE;
    }
    else if (0 == strcmp(init_arg_name, GEIS_INIT_UTOUCH_XCB))
    {
      if (back_end_type != BACK_END_TYPE_NONE)
      {
	geis_error("multiple back ends requested, only using last request");
      }
      back_end_type = BACK_END_TYPE_XCB;
    }

    init_arg_name = va_arg(varargs, GeisString);
  }

  if (back_end_type ==BACK_END_TYPE_MOCK_ENGINE)
  {
    geis->backend = geis_backend_new_test_fixture(geis,
                                                  track_devices,
                                                  track_classes);
  }
  else if (back_end_type ==BACK_END_TYPE_XCB)
  {
    geis->backend = geis_xcb_backend_new(geis, track_devices, track_classes);
  }
  else
  {
    geis_warning("back end not specified, defaulting to XCB");
    geis->backend = geis_xcb_backend_new(geis, track_devices, track_classes);
  }
  if (!geis->backend)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("can not create back end");
    status = GEIS_FALSE;
  }

  return status;
}


/**
 * Creates an initialized Geis API instance.
 */
Geis
geis_new(GeisString init_arg_name, ...)
{
  GeisBoolean success = GEIS_FALSE;
  Geis geis = geis_new_empty();
  if (geis)
  {
    geis_ref(geis);

    va_list varargs;
    va_start(varargs, init_arg_name);
    success = _set_valist(geis, init_arg_name, varargs);
    va_end(varargs);
  }

  if (!success)
  {
    geis_error_push(NULL, GEIS_STATUS_UNKNOWN_ERROR);
    geis_error("can not initialize GEIS API");
    geis_delete(geis);
    geis = NULL;
  }
  return geis;
}


static void _geis_destroy(Geis geis)
{
  geis_device_bag_delete(geis->devices);
  geis_event_queue_delete(geis->output_event_queue);
  close(geis->input_event_signal_pipe[0]);
  close(geis->input_event_signal_pipe[1]);
  geis_event_queue_delete(geis->input_event_queue);
  if (geis->backend)
    geis_backend_delete(geis->backend);
  geis_backend_multiplexor_delete(geis->backend_multiplexor);
  geis_subscription_bag_delete(geis->subscription_bag);
  free(geis);
}

/*
 * Disposes of a Geis API instance.
 */
GeisStatus
geis_delete(Geis geis)
{
  if (geis == NULL)
  {
    return GEIS_STATUS_BAD_ARGUMENT;
  }

  geis_subscription_bag_invalidate(geis->subscription_bag);
  geis_unref(geis);
  return GEIS_STATUS_SUCCESS;
}

/*
 * Increases the reference count of an API instance object.
 */
Geis
geis_ref(Geis geis)
{
  geis_atomic_ref(&geis->refcount);
  return geis;
}


/*
 * Decremenets the reference count of an API instance object.
 */
void
geis_unref(Geis geis)
{
  if (0 == geis_atomic_unref(&geis->refcount))
  {
    _geis_destroy(geis);
  }
}


/**
 * Gets a named configuration item.
 */
GeisStatus
geis_get_configuration(Geis        geis, 
                       GeisString  configuration_item_name,
                       void       *configuration_item_value)
{
  GeisStatus status = GEIS_STATUS_NOT_SUPPORTED;

  if (0 == strcmp(configuration_item_name, GEIS_CONFIGURATION_FD))
  {
    *(int*)configuration_item_value
        = geis_backend_multiplexor_fd(geis->backend_multiplexor);
    status = GEIS_STATUS_SUCCESS; 
  }
  else if (0 == strcmp(configuration_item_name, GEIS_CONFIG_UTOUCH_MAX_EVENTS))
  {
    *(int*)configuration_item_value
        = geis_backend_multiplexor_max_events_per_pump(geis->backend_multiplexor);
    status = GEIS_STATUS_SUCCESS; 
  }

  return status;
}


/*
 * Sets a named configuration item.
 */
GeisStatus
geis_set_configuration(Geis        geis,
                       GeisString  configuration_item_name,
                       void       *configuration_item_value)
{
  GeisStatus status = GEIS_STATUS_NOT_SUPPORTED;

  if (0 == strcmp(configuration_item_name, GEIS_CONFIG_UTOUCH_MAX_EVENTS))
  {
    int max_events = *(int *)configuration_item_value;
    geis_backend_multiplexor_set_max_events_per_pump(geis->backend_multiplexor,
                                                     max_events);
    status = GEIS_STATUS_SUCCESS; 
  }
  return status;
}


/*
 * Registers a callback to receive device change notifications.
 */
void
geis_register_device_callback(Geis               geis,
                              GeisEventCallback  event_callback,
                              void              *context)
{
  geis->device_event_callback = event_callback;
  geis->device_event_callback_context = context;
}


/*
 * Registers a callback to receive gesture class change notifications.
 */
void
geis_register_class_callback(Geis               geis,
                             GeisEventCallback  event_callback,
                             void              *context)
{
  geis->class_event_callback = event_callback;
  geis->class_event_callback_context = context;
}


/*
 * Registers an application-supplied event callback.
 */
void 
geis_register_event_callback(Geis               geis,
                             GeisEventCallback  output_event_callback,
                             void              *context)
{
  if (output_event_callback == GEIS_DEFAULT_EVENT_CALLBACK)
  {
    geis->output_event_callback = _default_output_event_callback;
  }
  else
  {
    geis->output_event_callback = output_event_callback;
  }
  geis->output_event_callback_context = context;
}


/*
 * Pumps the GEIS v2 event loop.
 */
GeisStatus
geis_dispatch_events(Geis geis)
{
  GeisStatus status = geis_backend_multiplexor_pump(geis->backend_multiplexor);
  return status;
}


/*
 * Posts an event through the API.
 *
 * Pushes the new event onto the input event queue and signals that a new event
 * has arrived.
 */
void
geis_post_event(Geis geis, GeisEvent event)
{
  geis_event_queue_enqueue(geis->input_event_queue, event);
  if (write(geis->input_event_signal_pipe[1], "1", 1) != 1)
  {
    geis_error("error %d writing input event signal: %s", errno, strerror(errno));
  }
}


/*
 * Pulls the next event off the queue.
 */
GeisStatus
geis_next_event(Geis geis, GeisEvent *event)
{
  GeisStatus status = GEIS_STATUS_EMPTY;
  *event = geis_event_queue_dequeue(geis->output_event_queue);
  if (*event)
  {
    status = geis_event_queue_is_empty(geis->output_event_queue)
           ? GEIS_STATUS_SUCCESS
           : GEIS_STATUS_CONTINUE;
  }

  return status;
}


/*
 * Adds a back end file descriptor to multiplex.
 */
void
geis_multiplex_fd(Geis                        geis,
                  int                         fd,
                  GeisBackendFdEventCallback  callback,
                  void                       *context)
{
  geis_backend_multiplexor_add_fd(geis->backend_multiplexor,
                                  fd, callback, context);
}


/*
 * Removes a back end file descriptor from the multiplex.
 */
void
geis_demultiplex_fd(Geis geis, int fd)
{
  geis_backend_multiplexor_remove_fd(geis->backend_multiplexor, fd);
}


GeisErrorStack *
geis_error_stack(Geis geis)
{
  return &geis->error_stack;
}


GeisSubBag
geis_subscription_bag(Geis geis)
{
  return geis->subscription_bag;
}


/*
 * Activates a subscription.
 */
GeisStatus
geis_activate_subscription(Geis geis, GeisSubscription sub)
{
  return geis_backend_subscribe(geis->backend, sub);
}


/*
 * Deactivates a subscription.
 */
GeisStatus
geis_deactivate_subscription(Geis geis, GeisSubscription sub)
{
  return geis_backend_unsubscribe(geis->backend, sub);
}


GeisAttrType
geis_get_device_attr_type(Geis geis, GeisString attr_name)
{
  GeisSize i = 0;
  for (; i < geis_device_bag_count(geis->devices); ++i)
  {
    GeisDevice device = geis_device_bag_device(geis->devices, i);
    GeisSize j = 0;
    for (; j < geis_device_attr_count(device); ++j)
    {
      GeisAttr attr = geis_device_attr(device, j);
      if (0 == strcmp(geis_attr_name(attr), attr_name))
      {
	return geis_attr_type(attr);
      }
    }
  }
  return GEIS_ATTR_TYPE_UNKNOWN;
}


GeisAttrType
geis_get_class_attr_type(Geis geis, GeisString attr_name)
{
  GeisSize i = 0;
  for (; i < geis_gesture_class_bag_count(geis->gesture_classes); ++i)
  {
    GeisGestureClass gesture_class = geis_gesture_class_bag_gesture_class(geis->gesture_classes, i);
    GeisSize j = 0;
    for (; j < geis_gesture_class_attr_count(gesture_class); ++j)
    {
      GeisAttr attr = geis_gesture_class_attr(gesture_class, j);
      if (0 == strcmp(geis_attr_name(attr), attr_name))
      {
	return geis_attr_type(attr);
      }
    }
  }
  return GEIS_ATTR_TYPE_UNKNOWN;
}


GeisAttrType
geis_get_region_attr_type(Geis geis, GeisString attr_name)
{
  if (0 == strcmp(attr_name, GEIS_REGION_ATTRIBUTE_WINDOWID))
  {
    return GEIS_ATTR_TYPE_INTEGER;
  }
  return GEIS_ATTR_TYPE_UNKNOWN;
}


