/**
 * @file xcb/geis_xcb_backend.c
 * @brief Implements the GEIS XCB back end.
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.	 See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_xcb_backend.h"

#include "geis_attr.h"
#include "geis_backend_protected.h"
#include "geis_class.h"
#include "geis_device.h"
#include "geis_event.h"
#include "geis_frame.h"
#include "geis_filter.h"
#include "geis_group.h"
#include "geis_logging.h"
#include "geis_private.h"
#include "geis_touch.h"
#include "grail_gestures.h"
#include <grail.h>
#include <grail-types.h>
#include <stdio.h>
#include <string.h>
#include "xcb_gesture.h"
#include <X11/extensions/XInput2.h>
#include <X11/X.h>
#include <X11/Xlib-xcb.h>
#include <xcb/xcb.h>

#define MIN_TOUCHES 1
#define MAX_TOUCHES 5
#define MAX_NUM_DEVICES 10
#define MAX_NUM_WINDOWS 10

#define GRAIL_XCB_BITMASK_LEN 2


static inline GeisSize
_min(GeisSize a, GeisSize b)
{
  return (a < b) ? a : b;
}


static inline GeisSize
_max(GeisSize a, GeisSize b)
{
  return (a > b) ? a : b;
}


typedef struct _GeisXcbBackend
{
  struct _GeisBackend  base;
  Geis                 geis;
  Display             *x11_display;
  xcb_connection_t    *xcb_connection;
  int                  xcb_fd;
} *GeisXcbBackend;


/*
 * Information required for each xcb-gesture subscription.
 *
 * @todo: unhardcode sizes and lengths (and malloc them instead).
 */
typedef struct _XcbGestureSub
{
  int           num_device_filters;
  uint16_t      devices[MAX_NUM_DEVICES]; 
  uint16_t      device_count;
  int           num_class_filters;
  uint32_t      mask[GRAIL_XCB_BITMASK_LEN];
  uint32_t      mask_len;
  int           num_region_filters;
  xcb_window_t  windows[MAX_NUM_WINDOWS];
  int           window_count;
} *XcbGestureSub;

static void _finalize(GeisBackend g);
static void _fd_callback(int fd, GeisBackendMultiplexorEvent ev, void *ctx);
static GeisStatus _subscribe(GeisBackend be, GeisSubscription sub);
static GeisStatus _unsubscribe(GeisBackend be, GeisSubscription sub);


static struct _GeisBackendVtable be_vtbl = {
  _finalize,
  _subscribe,
  _unsubscribe
};


static void
_map_xi2_mode_to_geis_device_attrs(int xi2_mode, GeisDevice geis_device)
{
  GeisAttr device_attr;
  GeisBoolean is_direct = GEIS_FALSE;
  GeisBoolean is_independent = GEIS_FALSE;

  if (xi2_mode == XIDirectTouch)
    is_direct = GEIS_TRUE;

  device_attr = geis_attr_new(GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH,
                              GEIS_ATTR_TYPE_BOOLEAN,
                              &is_direct);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
  }
  else
  {
    geis_device_add_attr(geis_device, device_attr);
  }

  if (xi2_mode == XIIndependentPointer)
    is_independent = GEIS_TRUE;

  device_attr = geis_attr_new(GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH,
                              GEIS_ATTR_TYPE_BOOLEAN,
                              &is_independent);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
  }
  else
  {
    geis_device_add_attr(geis_device, device_attr);
  }
}


static int
_verify_xcb_version(xcb_connection_t *xcb_connection)
{
  int                                 is_valid_version = 0;
  xcb_gesture_query_version_cookie_t  version_cookie;
  xcb_gesture_query_version_reply_t  *version_reply = NULL;
  xcb_generic_error_t                *error = NULL;

  version_cookie = xcb_gesture_query_version(xcb_connection,
                                             XCB_GESTURE_MAJOR_VERSION,
                                             XCB_GESTURE_MINOR_VERSION);
  version_reply = xcb_gesture_query_version_reply(xcb_connection,
                                                  version_cookie,
                                                  &error);
  if (!version_reply)
  {
    geis_error("failed to receive XCB gesture version reply.");
    goto final_exit;
  }

  if (version_reply->major_version != XCB_GESTURE_MAJOR_VERSION
   && version_reply->minor_version != XCB_GESTURE_MINOR_VERSION)
  {
    geis_error("server supports unrecognized version: %d.%d",
               version_reply->major_version, version_reply->minor_version);
  }
  else
  {
    is_valid_version = 1;
  }

 free(version_reply);
final_exit:
 return is_valid_version;
}


void 
_finalize(GeisBackend g)
{
  GeisXcbBackend be = (GeisXcbBackend)g;

  /* XCB has a bug that causes XCloseDisplay to fire a fatal IO error if the xcb
   * connection has an error. The best we can do right now is to leave the
   * connection dangling. */
  if (!be->xcb_connection || !xcb_connection_has_error(be->xcb_connection))
    XCloseDisplay(be->x11_display);

  free(be);
  geis_debug("XCB back end finalized");
}


static void
_dispatch_gesture(GeisXcbBackend be, xcb_gesture_notify_event_t *grail_event)
{
  GeisEvent     geis_event = NULL;
  GeisGroupSet  groupset = geis_groupset_new();
  GeisAttr      group_attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_GROUPSET,
                                           GEIS_ATTR_TYPE_POINTER,
                                           groupset);
  GeisTouchSet  touchset = geis_touchset_new();
  GeisAttr      touch_attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_TOUCHSET,
                                           GEIS_ATTR_TYPE_POINTER,
                                           touchset);

  /* Play games with the grail event to get the grail gesture properties. */
  float        *properties = (float *)(grail_event + 1);
  GeisSize      num_properties = grail_event->num_props;
  GeisSize      num_unmapped_properties = num_properties;
  GeisSize      touch_count;

  geis_attr_set_destructor(group_attr, (GeisAttrDestructor)geis_groupset_delete);
  geis_attr_set_destructor(touch_attr, (GeisAttrDestructor)geis_touchset_delete);

  switch (grail_event->status)
  {
    case GRAIL_STATUS_BEGIN:
      geis_event = geis_event_new(GEIS_EVENT_GESTURE_BEGIN);
      break;
    case GRAIL_STATUS_UPDATE:
      geis_event = geis_event_new(GEIS_EVENT_GESTURE_UPDATE);
      break;
    case GRAIL_STATUS_END:
      geis_event = geis_event_new(GEIS_EVENT_GESTURE_END);
      break;
  }

  GeisGroup group = geis_group_new(1);
  geis_groupset_insert(groupset, group);

  GeisFrame frame = geis_frame_new(grail_event->gesture_id);
  geis_group_insert_frame(group, frame);

  GeisAttr attr = NULL;

  GeisInteger ival = grail_event->device_id;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_DEVICE_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->time;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TIMESTAMP,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->root;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_ROOT_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->event;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_EVENT_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->child;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_CHILD_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_FOCUS_X,
                       GEIS_ATTR_TYPE_FLOAT,
                       &grail_event->focus_x);
  geis_frame_add_attr(frame, attr);

  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_FOCUS_Y,
                       GEIS_ATTR_TYPE_FLOAT,
                       &grail_event->focus_y);
  geis_frame_add_attr(frame, attr);

  num_unmapped_properties = geis_xcb_backend_map_grail_attrs(
                                   grail_event->gesture_type,
                                   num_properties, properties, frame);

  /* any remaining properties are touch properties */
  touch_count = num_unmapped_properties / 3;
  properties += num_properties - num_unmapped_properties;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TOUCHES,
                       GEIS_ATTR_TYPE_INTEGER,
                       &touch_count);
  geis_frame_add_attr(frame, attr);

  GeisSize i;
  for (i = 0; i < touch_count; ++i)
  {
    GeisTouch  touch = geis_touch_new(i);

    if (num_unmapped_properties > 0)
    {
      GeisInteger touch_slot = *properties;
      attr = geis_attr_new(GEIS_TOUCH_ATTRIBUTE_ID,
                           GEIS_ATTR_TYPE_INTEGER,
                           &touch_slot);
      geis_touch_add_attr(touch, attr);
      ++properties;
      --num_unmapped_properties;
    }
    if (num_unmapped_properties > 0)
    {
      attr = geis_attr_new(GEIS_TOUCH_ATTRIBUTE_X,
                           GEIS_ATTR_TYPE_FLOAT,
                           properties);
      geis_touch_add_attr(touch, attr);
      ++properties;
      --num_unmapped_properties;
    }
    if (num_unmapped_properties > 0)
    {
      attr = geis_attr_new(GEIS_TOUCH_ATTRIBUTE_Y,
                           GEIS_ATTR_TYPE_FLOAT,
                           properties);
      geis_touch_add_attr(touch, attr);
      ++properties;
      --num_unmapped_properties;
    }
    geis_touchset_insert(touchset, touch);
    geis_frame_add_touchid(frame, geis_touch_id(touch));
  }

  geis_event_add_attr(geis_event, group_attr);
  geis_event_add_attr(geis_event, touch_attr);
  geis_post_event(be->geis, geis_event);
}


/**
 * Dispatches events coming from XCB.
 */
static void
_xcb_dispatch(GeisXcbBackend be)
{
  if (be->xcb_connection)
  {
    const xcb_query_extension_reply_t *extension_info;
    extension_info = xcb_get_extension_data(be->xcb_connection, &xcb_gesture_id);

    xcb_generic_event_t *event = NULL;
    while ((event = xcb_poll_for_event(be->xcb_connection)))
    {
      xcb_gesture_notify_event_t *gesture_event = NULL;
      if (event->response_type != GenericEvent) {
	geis_warning("received non-generic event type: %d",
	             event->response_type);
	goto next_event;
      }

      gesture_event = (xcb_gesture_notify_event_t*)event;
      if (gesture_event->extension != extension_info->major_opcode)
      {
	geis_warning("received non-gesture extension event: %d",
	             gesture_event->extension);
	goto next_event;
      }

      if (gesture_event->event_type != XCB_GESTURE_NOTIFY)
      {
	geis_warning("received unrecognized gesture event type: %d",
	             gesture_event->event_type);
	goto next_event;
      }

      _dispatch_gesture(be, gesture_event);
next_event:
      free(event);
    }
  }
}

/** @todo implement this */
void
_fd_callback(int                          fd __attribute__((unused)),
             GeisBackendMultiplexorEvent  ev __attribute__((unused)),
             void                        *ctx)
{
  GeisXcbBackend be = (GeisXcbBackend)ctx;
  _xcb_dispatch(be);
}


/*
 * Translates a GEIS v2 device filter term into something XCB understands.
 */
static void
_translate_device_term_to_xcb(GeisFilterTerm  term,
                              uint16_t       *devices,
                              uint16_t       *device_count)
{
  GeisAttr            attr = geis_filter_term_attr(term);
  GeisString          name = geis_attr_name(attr);
  GeisFilterOperation op   = geis_filter_term_operation(term);

  if (0 == strcmp(name, GEIS_DEVICE_ATTRIBUTE_ID)
       && op == GEIS_FILTER_OP_EQ)
  {
    if (*device_count >= MAX_NUM_DEVICES)
    {
      geis_warning("too many device terms, maximum is %d", MAX_NUM_DEVICES);
    }
    else
    {
      uint16_t dev =  geis_attr_value_to_integer(attr);
      uint16_t i;

      for (i = 0; i < *device_count; ++i)
	if (devices[i] == dev)
	  return;

      devices[*device_count] = dev;
      (*device_count)++;
    }
  }
  else
  {
    geis_warning("invalid filter term");
  }
}


/*
 * Translates a GEIS v2 class filter term into something XCB understands.
 */
static void
_translate_class_term_to_xcb(GeisFilterTerm  term,
                             GeisString     *class_name,
                             GeisSize       *min_touches,
                             GeisSize       *max_touches)
{
  GeisAttr   attr = geis_filter_term_attr(term);
  GeisString name = geis_attr_name(attr);
  GeisFilterOperation op = geis_filter_term_operation(term);

  if (0 == strcmp(name, GEIS_CLASS_ATTRIBUTE_NAME)
      && op == GEIS_FILTER_OP_EQ)
  {
    *class_name = geis_attr_value_to_string(attr);
  }
  else if (0 == strcmp(name, GEIS_GESTURE_ATTRIBUTE_TOUCHES))
  {
    GeisSize touches = geis_attr_value_to_integer(attr);
    switch (op)
    {
      case GEIS_FILTER_OP_GT:
	*min_touches = _max(*min_touches, touches+1);
	break;
      case GEIS_FILTER_OP_GE:
	*min_touches = _max(*min_touches, touches);
	break;
      case GEIS_FILTER_OP_LT:
	*max_touches = _min(touches-1, *max_touches);
	break;
      case GEIS_FILTER_OP_LE:
	*max_touches = _min(touches, *max_touches);
	break;
      case GEIS_FILTER_OP_EQ:
	*min_touches = _max(*min_touches, touches);
	*max_touches = _min(touches, *max_touches);
	break;
      case GEIS_FILTER_OP_NE:
	break;
    }
  }
  else
  {
    geis_warning("invalid filter term");
  }

}


/*
 * Translates a GEIS v2 region filter term into something XCB understands.
 */
static void
_translate_region_term_to_xcb(GeisFilterTerm   term,
                              xcb_window_t    *windows,
                              int             *window_count)
{
  GeisAttr            attr;
  GeisString          name;
  GeisFilterOperation op;

  attr = geis_filter_term_attr(term);
  if (!attr)
  {
    geis_error("can not get attr for region term");
    return;
  }
  name = geis_attr_name(attr);
  op   = geis_filter_term_operation(term);

  if (0 == strcmp(name, GEIS_REGION_ATTRIBUTE_WINDOWID)
      && op == GEIS_FILTER_OP_EQ)
  {
    if (*window_count >= MAX_NUM_WINDOWS)
    {
      geis_warning("too many region terms, maximum is %d", MAX_NUM_WINDOWS);
    }
    else
    {
      int i;
      uint32_t window_id = (uint32_t)geis_attr_value_to_integer(attr);

      for (i = 0; i < *window_count; ++i)
	if (windows[i] == window_id)
	  return;

      windows[*window_count] = window_id;
      (*window_count)++;
    }
  }
  else
  {
    geis_warning("invalid filter term");
  }
}


/*
 * Translates the filters for a subscription into the appropriate XCB-specific
 * objects.
 *
 * Geis filter terms are always ANDed together, and geis subscription filters
 * are always ORed together.
 */
static void
_translate_filter_to_xcb(GeisSubscription sub,
                         GeisFilter       filter,
                         XcbGestureSub    xsub)
{
  GeisSize term_count = geis_filter_term_count(filter);
  GeisSize i;

  GeisString class_name  = NULL;
  GeisSize   min_touches = MIN_TOUCHES;
  GeisSize   max_touches = MAX_TOUCHES;
  uint32_t   mask_len    = GRAIL_XCB_BITMASK_LEN;
  uint32_t   mask[mask_len];

  geis_xcb_backend_gesture_bitmask_init(geis_subscription_flags(sub),
                                        &mask_len, mask);
  for (i = 0; i < term_count; ++i)
  {
    GeisFilterTerm term = geis_filter_term(filter, i);
    switch (geis_filter_term_facility(term))
    {
      case GEIS_FILTER_DEVICE:
	_translate_device_term_to_xcb(term, xsub->devices, &xsub->device_count);
	++xsub->num_device_filters;
	break;
      case GEIS_FILTER_CLASS:
	_translate_class_term_to_xcb(term, &class_name,
	                             &min_touches, &max_touches);
	++xsub->num_class_filters;
	break;
      case GEIS_FILTER_REGION:
	_translate_region_term_to_xcb(term, xsub->windows, &xsub->window_count);
	++xsub->num_region_filters;
	break;
      default:
	geis_warning("-- unknown term --");
	break;
    }
  }

  if (class_name)
  {
    geis_xcb_backend_gesture_bitmask_filter_class(class_name,
                                                  &mask_len, mask);
  }
  geis_xcb_backend_gesture_bitmask_filter_touches(min_touches, max_touches,
                                                  &mask_len, mask);

  xsub->mask_len = mask_len;
  for (i = 0; i < mask_len; ++i)
  {
    xsub->mask[i] |= mask[i];
  }
}


/*
 * Translates geis subscriptions into XCB-gesture subscriptions.
 */
static GeisStatus 
_subscribe(GeisBackend be, GeisSubscription sub)
{
  GeisStatus status = GEIS_STATUS_SUCCESS;
  int dev;
  int win;
  GeisSize i;
  GeisXcbBackend xcb = (GeisXcbBackend)be;
  GeisSize filter_count = geis_subscription_filter_count(sub);
  XcbGestureSub xsub = calloc(1, sizeof(struct _XcbGestureSub));
  if (!xsub)
  {
    geis_error("error allocating xcb-gesture subscription");
    goto error_exit;
  }
  xsub->mask_len = GRAIL_XCB_BITMASK_LEN;

  /* Translate the GEIS v2 subscription into an XCB subscription. */
  for (i = 0; i < filter_count; ++i)
  {
    GeisFilter filter = geis_subscription_filter(sub, i);
    _translate_filter_to_xcb(sub, filter, xsub);
  }

  /* Adjust for special case of "no device filters" == "all devices". */
  if (xsub->num_device_filters == 0)
  {
    geis_debug("defaulting to ALL devices");
    xsub->devices[0] = 0;
    xsub->device_count = 1;
  }

  /* Adjust for special case of "no class filters" == "all gestures". */
  if (xsub->num_class_filters == 0)
  {
    geis_debug("defaulting to ALL gestures");
    geis_xcb_backend_gesture_bitmask_init(geis_subscription_flags(sub),
                                          &xsub->mask_len, xsub->mask);
  }

  /* Adjust for special case of "no region filters" == "root window". */
  geis_debug("xsub->num_region_filters=%d xsub->window_count=%d", xsub->num_region_filters, xsub->window_count);
  if (xsub->num_region_filters == 0)
  {
    geis_debug("defaulting to ROOT window");
    const xcb_setup_t *setup = xcb_get_setup(xcb->xcb_connection);
    if (!setup)
    {
      geis_error("error getting xcb setup");
      goto error_exit;
    }

    xcb_screen_iterator_t screen = xcb_setup_roots_iterator(setup);
    while (screen.rem)
    {
      xsub->windows[xsub->window_count++] = screen.data->root;
      xcb_screen_next(&screen);
    }
  }

  /* Tell XCB to select gesture events for each (window, device) combo. */
  for (dev = 0; dev < xsub->device_count; ++dev)
  {
    for (win = 0; win < xsub->window_count; ++win)
    {
      xcb_generic_error_t  *error;
      xcb_void_cookie_t select_cookie;

      geis_debug("window_id=0x%08x device_id=%d mask_len=%d mask=0x%08x %08x",
                 xsub->windows[win], xsub->devices[dev],
                 xsub->mask_len, xsub->mask[1], xsub->mask[0]);
      select_cookie = xcb_gesture_select_events_checked(xcb->xcb_connection,
                                                        xsub->windows[win],
                                                        (uint16_t)xsub->devices[dev],
                                                        (uint16_t)xsub->mask_len,
                                                        xsub->mask);
      error = xcb_request_check(xcb->xcb_connection, select_cookie);
      if (error)
      {
	geis_error("failed to select events for window 0x%08x",
	           xsub->windows[win]);
	break;
      }
    }
  }

  /* remember the subscription description */
  geis_subscription_set_pdata(sub, xsub);

error_exit:
  return status;
}


/*
 * Tears down an XCB-specific subscription instance.
 */
static GeisStatus 
_unsubscribe(GeisBackend be __attribute__((unused)), GeisSubscription sub)
{
  GeisStatus status = GEIS_STATUS_SUCCESS;
  XcbGestureSub xsub = geis_subscription_pdata(sub);
  free(xsub);
  return status;
}


/*
 * Report a gesture-capable device.
 */
static void
_report_an_xcb_device(GeisXcbBackend be,  XIDeviceInfo *xcb_device)
{
  int        class_index;
  GeisDevice geis_device;
  GeisAttr   device_attr;
  GeisEvent  device_event;

  device_event = geis_event_new(GEIS_EVENT_DEVICE_AVAILABLE);
  if (!device_event)
  {
    geis_error("failed to create device-available event");
    goto final_exit;
  }

  geis_device = geis_device_new(xcb_device->name, xcb_device->deviceid);
  if (!geis_device)
  {
    geis_error("failed to create device");
    goto unroll_event;
  }

  device_attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_DEVICE,
                              GEIS_ATTR_TYPE_POINTER,
                              geis_device);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
    goto unroll_device;
  }
  geis_event_add_attr(device_event, device_attr);

  for (class_index = 0; class_index < xcb_device->num_classes; ++class_index)
  {
    XIAnyClassInfo *any = xcb_device->classes[class_index];
    if (any->type == XITouchClass)
    {
      XITouchClassInfo *v = (XITouchClassInfo *)any;
      _map_xi2_mode_to_geis_device_attrs(v->mode, geis_device);
      geis_debug("touch class for device %d \"%s\": mode %d num_touches %d",
		 xcb_device->deviceid, xcb_device->name,
		 v->mode, v->num_touches);

      device_attr = geis_attr_new(GEIS_DEVICE_ATTRIBUTE_TOUCHES,
                                  GEIS_ATTR_TYPE_INTEGER,
                                  &v->num_touches);
      if (!device_attr)
      {
	geis_error("failed to create device attr");
      }
      else
      {
	geis_device_add_attr(geis_device, device_attr);
      }
    }
    else if (any->type == XITouchValuatorClass)
    {
      char name[64];
      GeisFloat f;
      GeisInteger i;
      XITouchValuatorClassInfo *v = (XITouchValuatorClassInfo *)any;
      char *label = v->label ? XGetAtomName(be->x11_display, v->label) : "";
      sprintf(name, "%.48s %d min", label, v->number);
      f = v->min;
      geis_debug("touch valuator for device %d \"%s\": label \"%s\" %g",
                 xcb_device->deviceid, xcb_device->name, name, f);
      device_attr = geis_attr_new(name, GEIS_ATTR_TYPE_FLOAT, &f);
      if (!device_attr)
      {
	geis_error("failed to create device attr");
      }
      else
      {
	geis_device_add_attr(geis_device, device_attr);
      }

      sprintf(name, "%.48s %d max", label, v->number);
      f = v->max;
      geis_debug("touch valuator for device %d \"%s\": label \"%s\" %g",
                 xcb_device->deviceid, xcb_device->name, name, v->max);
      device_attr = geis_attr_new(name, GEIS_ATTR_TYPE_FLOAT, &f);
      if (!device_attr)
      {
	geis_error("failed to create device attr");
      }
      else
      {
	geis_device_add_attr(geis_device, device_attr);
      }

      sprintf(name, "%.48s %d resolution", label, v->number);
      i = v->resolution;
      geis_debug("touch valuator for device %d \"%s\": label \"%s\" %d",
                 xcb_device->deviceid, xcb_device->name, name, i);
      device_attr = geis_attr_new(name, GEIS_ATTR_TYPE_INTEGER, &i);
      if (!device_attr)
      {
	geis_error("failed to create device attr");
      }
      else
      {
	geis_device_add_attr(geis_device, device_attr);
      }
    }
  }
  geis_post_event(be->geis, device_event);
  goto final_exit;

unroll_device:
  geis_device_unref(geis_device);
unroll_event:
  geis_event_delete(device_event);
final_exit:
  return;
}


/*
 * Enumerates all input devices known to the X server and reports those that may
 * give gestural input.
 */
static void
_report_xcb_devices(GeisXcbBackend be)
{
  XIDeviceInfo *devices;
  int           device_index;
  int           num_devices;

  devices = XIQueryDevice(be->x11_display, XIAllDevices, &num_devices);
  for (device_index = 0; device_index < num_devices; ++device_index)
  {
    int class_index;
    for (class_index = 0;
         class_index < devices[device_index].num_classes;
         ++class_index)
    {
      XIAnyClassInfo *any = devices[device_index].classes[class_index];
      if (any->type == XITouchClass)
      {
	_report_an_xcb_device(be, &devices[device_index]);
	break;
      }
    }
  }

  XIFreeDeviceInfo(devices);
}


/*
 * Generates the events for gesture classes.
 */
static void
_report_grail_classes(GeisXcbBackend be)
{
  GeisGestureClass drag_class = geis_gesture_class_new(GEIS_GESTURE_DRAG,
                                                       GRAIL_TYPE_DRAG1);
  geis_xcb_backend_add_drag_attrs(drag_class);
  GeisEvent event_drag = geis_event_new(GEIS_EVENT_CLASS_AVAILABLE);
  GeisAttr  attr_drag = geis_attr_new(GEIS_EVENT_ATTRIBUTE_CLASS,
                                      GEIS_ATTR_TYPE_POINTER,
                                      drag_class);
  geis_event_add_attr(event_drag, attr_drag);
  geis_post_event(be->geis, event_drag);

  GeisGestureClass pinch_class = geis_gesture_class_new(GEIS_GESTURE_DRAG,
                                                        GRAIL_TYPE_PINCH1);
  geis_xcb_backend_add_pinch_attrs(pinch_class);
  GeisEvent event_pinch = geis_event_new(GEIS_EVENT_CLASS_AVAILABLE);
  GeisAttr  attr_pinch = geis_attr_new(GEIS_EVENT_ATTRIBUTE_CLASS,
                                      GEIS_ATTR_TYPE_POINTER,
                                      pinch_class);
  geis_event_add_attr(event_pinch, attr_pinch);
  geis_post_event(be->geis, event_pinch);

  GeisGestureClass rotate_class = geis_gesture_class_new(GEIS_GESTURE_DRAG,
                                                         GRAIL_TYPE_ROTATE1);
  geis_xcb_backend_add_rotate_attrs(rotate_class);
  GeisEvent event_rotate = geis_event_new(GEIS_EVENT_CLASS_AVAILABLE);
  GeisAttr  attr_rotate = geis_attr_new(GEIS_EVENT_ATTRIBUTE_CLASS,
                                      GEIS_ATTR_TYPE_POINTER,
                                      rotate_class);
  geis_event_add_attr(event_rotate, attr_rotate);
  geis_post_event(be->geis, event_rotate);

  GeisGestureClass tap_class = geis_gesture_class_new(GEIS_GESTURE_DRAG,
                                                      GRAIL_TYPE_TAP1);
  geis_xcb_backend_add_tap_attrs(tap_class);
  GeisEvent event_tap = geis_event_new(GEIS_EVENT_CLASS_AVAILABLE);
  GeisAttr  attr_tap = geis_attr_new(GEIS_EVENT_ATTRIBUTE_CLASS,
                                      GEIS_ATTR_TYPE_POINTER,
                                      tap_class);
  geis_event_add_attr(event_tap, attr_tap);
  geis_post_event(be->geis, event_tap);
}


GeisBackend
geis_xcb_backend_new(Geis geis,
                     GeisBoolean track_devices,
                     GeisBoolean track_classes)
{
  GeisXcbBackend be = calloc(1, sizeof(struct _GeisXcbBackend));
  if (!be)
  {
    geis_error("failed to allocate GEIS XCB back end");
    goto final_exit;
  }

  geis_backend_init_base(&be->base, &be_vtbl, "GEIS2 XCB");
  be->geis = geis;

  be->x11_display = XOpenDisplay(NULL);
  if (!be->x11_display)
  {
    geis_error("error opening X server.");
    goto unwind_be;
  }

  be->xcb_connection = XGetXCBConnection(be->x11_display);
  if (!be->xcb_connection)
  {
    geis_error("error connecting to X server.");
    goto unwind_x11;
  }
  if (!_verify_xcb_version(be->xcb_connection))
  {
    goto unwind_x11;
  }

  be->xcb_fd = xcb_get_file_descriptor(be->xcb_connection);
  geis_multiplex_fd(be->geis, be->xcb_fd, _fd_callback, be);

  if (track_devices)
  {
    _report_xcb_devices(be);
  }

  if (track_classes)
  {
    _report_grail_classes(be);
  }

  goto final_exit;

unwind_x11:
  /* XCB has a bug that causes XCloseDisplay to fire a fatal IO error if the xcb
   * connection has an error. The best we can do right now is to leave the
   * connection dangling. */
  if (!be->xcb_connection || !xcb_connection_has_error(be->xcb_connection))
    XCloseDisplay(be->x11_display);
unwind_be:
  free(be);
  be = NULL;
final_exit:
  return be;
}



