/*****************************************************************************
 *
 * utouch-frame - Touch Frame Library
 *
 * Copyright (C) 2010-2012 Canonical Ltd.
 *
 * This library is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License version 3
 * as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#ifndef UTOUCH_FRAME_XI2_H
#define UTOUCH_FRAME_XI2_H

#include <utouch/frame.h>
#include <X11/Xlib.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @addtogroup v2
 * @{
 *
 * @defgroup v2-x11 X11
 * @{
 */

/**
 * Create a new uTouch Frame context for an X11 window server
 *
 * @param [in] display The X11 server connection
 * @param [out] handle The object for the new uTouch Frame instance
 * @return UFStatusSuccess, UFStatusErrorResources, or UFStatusErrorGeneric
 */
FRAME_PUBLIC
UFStatus frame_x11_new(Display *display, UFHandle *handle);

/**
 * Delete an X11 uTouch Frame instance
 *
 * @param [in] handle The object for the uTouch Frame instance
 */
FRAME_PUBLIC
void frame_x11_delete(UFHandle handle);

/**
 * Process an X11 input event into the uTouch Frame instance
 *
 * @param [in] handle The uTouch Frame context
 * @param [in] event The X11 generic input event cookie
 * @return UFStatusSuccess or UFStatusErrorGeneric
 *
 * The uTouch-Frame library can process XIDeviceEvent and
 * XIHierarchyEvent events. Processing these events requires additional event
 * data. This data is obtained by calling XGetEventData on the XEvent. See the
 * XGetEventData and XFreeEventData man pages for more details.
 *
 * This function will silently ignore any events other than those listed above.
 */
FRAME_PUBLIC
UFStatus frame_x11_process_event(UFHandle handle, XGenericEventCookie *xcookie);

/**
 * Accept ownership of a touch
 *
 * @param [in] device The device object for the touch (const)
 * @param [in] window The window to accept the touch for
 * @param [in] touch_id The touch ID object for the touch
 * @return UFStatusSuccess, UFStatusErrorInvalidTouch
 */
FRAME_PUBLIC
UFStatus frame_x11_accept_touch(UFDevice device, UFWindowId window,
                                UFTouchId touch_id);

/**
 * Reject ownership of a touch
 *
 * @param [in] device The device object for the touch (const)
 * @param [in] window The window to reject the touch for
 * @param [in] touch_id The touch ID object for the touch
 * @return UFStatusSuccess, UFStatusErrorInvalidTouch
 */
FRAME_PUBLIC
UFStatus frame_x11_reject_touch(UFDevice device, UFWindowId window,
                                UFTouchId touch_id);

/**
 * Get the X11 Window ID of a uTouch Frame window
 *
 * @param [in] window_id The uTouch Frame window ID object (const)
 * @return The X11 window ID
 */
FRAME_PUBLIC
Window frame_x11_get_window_id(UFWindowId window_id);

/**
 * Create a new uTouch Frame window ID object for an X11 window ID
 *
 * @param [in] id The X11 ID of the window
 * @return The new uTouch Frame window ID object
 */
FRAME_PUBLIC
UFWindowId frame_x11_create_window_id(Window id);

/**
 * Get the X11 touch ID of a uTouch Frame touch
 *
 * @param [in] touch_id The uTouch Frame touch ID object (const)
 * @return The X11 touch ID
 */
FRAME_PUBLIC
unsigned int frame_x11_get_touch_id(UFTouchId touch_id);

/**
 * Create a new uTouch Frame touch ID object for an X11 touch ID
 *
 * @param [in] id The X11 ID of the touch
 * @return The new uTouch Frame touch ID object
 */
FRAME_PUBLIC
UFTouchId frame_x11_create_touch_id(unsigned int id);

/**
 * @}
 *
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif
