/*****************************************************************************
 *
 * utouch-frame - Touch Frame Library
 *
 * Copyright (C) 2010-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *      Henrik Rydberg <rydberg@bitmath.org>
 *
 ****************************************************************************/

#define MTDEV_NO_LEGACY_API

#include <utouch/frame-mtdev.h>
#include <mtdev.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include "common-defs.h"

struct frame_test {
	struct evemu_device *evemu;
	struct mtdev *mtdev;
	utouch_frame_handle fh;
};

static int init_evemu(struct frame_test *test, int fd)
{
	test->evemu = evemu_new(NULL);
	if (!test->evemu)
		return -1;
	return evemu_extract(test->evemu, fd);
}

static int init_mtdev(struct frame_test *test, int fd)
{
	test->mtdev = mtdev_new_open(fd);
	if (!test->mtdev)
		return -1;
	return 0;
}

static int init_frame(struct frame_test *test, int fd)
{
	test->fh = utouch_frame_new_engine(100, 32, 100);
	if (!test->fh)
		return -1;
	return utouch_frame_init_mtdev(test->fh, test->evemu);
}

static void destroy_all(struct frame_test *test)
{
	utouch_frame_delete_engine(test->fh);
	mtdev_close_delete(test->mtdev);
	evemu_delete(test->evemu);
	memset(test, 0, sizeof(*test));
}

static void loop_device(struct frame_test *test, int fd)
{
	const struct utouch_frame *frame;
	struct input_event ev;

	while (!mtdev_idle(test->mtdev, fd, 5000)) {
		while (mtdev_get(test->mtdev, fd, &ev, 1) > 0) {
			frame = utouch_frame_pump_mtdev(test->fh, &ev);
			if (frame)
				report_frame(frame);
		}
	}
}

int main(int argc, char *argv[])
{
	struct frame_test test;
	int fd;

	if (argc < 2) {
		fprintf(stderr, "Usage: %s <device>\n", argv[0]);
		return -1;
	}

	fd = open(argv[1], O_RDONLY | O_NONBLOCK);
	if (fd < 0) {
		fprintf(stderr, "error: could not open device\n");
		return -1;
	}
	if (ioctl(fd, EVIOCGRAB, 1)) {
		fprintf(stderr, "error: could not grab the device\n");
		return -1;
	}

	if (init_evemu(&test, fd)) {
		fprintf(stderr, "error: could not describe device\n");
		return -1;
	}
	if (!utouch_frame_is_supported_mtdev(test.evemu)) {
		fprintf(stderr, "error: unsupported device\n");
		return -1;
	}

	fprintf(stderr, "device: %s\n", evemu_get_name(test.evemu));

	if (init_mtdev(&test, fd)) {
		fprintf(stderr, "error: could not init mtdev\n");
		return -1;
	}
	if (init_frame(&test, fd)) {
		fprintf(stderr, "error: could not init frame\n");
		return -1;
	}

	report_device_caps(test.fh);

	loop_device(&test, fd);

	destroy_all(&test);
	return 0;
}
