/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

namespace Unity.FilesPlace {

  public class Bookmarks : Object
  {
    
    private List<Bookmark> bookmarks;
    private string bookmarks_file;
    private FileMonitor monitor;
    
    public signal void changed ();
    
    public Bookmarks ()
    {
      bookmarks_file = @"$(Environment.get_home_dir())/.gtk-bookmarks";
      update();
      
      /* Update the bookmarks list whener the bookmarks file changes */
      try {
        monitor = File.new_for_path (bookmarks_file).monitor (FileMonitorFlags.NONE);
        monitor.set_rate_limit (2000); // max 1 update every 2s
        monitor.changed.connect ((mon, file, other_file, event_type) => {
          debug ("Bookmarks changed. Updating.");
          update ();
          changed ();
        });
      } catch (Error e) {
        warning ("Failed to install file monitor on %s. Bookmarks monitoring disabled: %s",
                 bookmarks_file, e.message);
      }
    }
    
    private void update ()
    {
      bookmarks = new List<Bookmark> ();
      string contents;
      
      try {
        FileUtils.get_contents (bookmarks_file, out contents);
        
      } catch (FileError e) {
        warning ("Failed to read favorites: %s", e.message);
        return;
      }
      
      string[] favorites = contents.split ("\n");
      string mimetype = "inode/directory";
      
      
      foreach (var uri in favorites)
      {
        if (uri == "")
          continue;
              
        string[] parts = uri.split (" ", 2);
        string display_name;
        
        if (parts.length == 1)
          {
            display_name = Uri.unescape_string (uri);
            display_name = Filename.display_basename (display_name);
          }
        else if (parts.length == 2)
          {
            uri = parts[0];
            display_name = parts[1];
          }
        else
          {
            warning ("Internal error computing display name for favorite '%s'",
                     uri);
            display_name = uri;
          }
        
        var bookmark = new Bookmark (uri, mimetype, display_name);
        bookmarks.append (bookmark);
      }
    }
    
    public unowned List<Bookmark> list ()
    {
      return bookmarks;
    }
    
    public List<Bookmark> prefix_search (string search)
    {
      var prefix = Utils.normalize_string (search);
      var matches = new List<Bookmark> ();
      
      foreach (var bookmark in bookmarks)
      {
        foreach (var term in bookmark.get_index_terms ())
        {
          if (term.has_prefix (prefix))
            {
              /* Register a hit for this bookmark */
              matches.append (bookmark);
              break;
            }
        }
      }
      
      return (owned) matches;
    }
  
  }
  
  private class Bookmark : Object
  {
    public string uri { get; set construct; }
    public string icon { get; set construct; }
    public string mimetype { get; set construct; }
    public string display_name { get; set construct; }
    
    private List<string> index_terms;
  
    public Bookmark (string uri, string mimetype, string display_name)
    {
      Object (uri:uri, icon:Utils.get_icon_for_uri (uri, mimetype),
              mimetype:mimetype, display_name:display_name);
      
      index_terms = new List<string> ();
      index_terms.append (Utils.normalize_string (Path.get_basename (uri)));
      index_terms.append (Utils.normalize_string (display_name));
    }
    
    public unowned List<string> get_index_terms ()
    {
      return index_terms;
    }
  }
  

} /* end: namespace */
