/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Posix;
using Unity;
using Gee;
 
namespace Unity.ApplicationsLens.Utils
{ 
  
  public AppInfo? get_app_info_for_actor (string actor)
  {
    string desktop_id = get_desktop_id_for_actor (actor);
    return AppInfoManager.get_instance().lookup (desktop_id);
  }
  
  public bool check_actor_has_category (string actor, Set<string>? categories)
  {
    if (categories == null)
      return false;
  
    AppInfoManager appman = AppInfoManager.get_instance ();
    string desktop_id = get_desktop_id_for_actor (actor);
    AppInfo? appinfo =  appman.lookup (desktop_id);
    
    if (appinfo == null)
      return false;
    
    unowned string[]? actor_cats = appman.get_categories (desktop_id);
    if (actor_cats == null)
      return false;
    
    foreach (var cat in actor_cats)
    {
      if (cat in categories)
        return true;
    }
    
    return false;
  }
  
  public string get_desktop_id_for_actor (string actor)
  {
    if (actor.has_prefix("application://"))
      return actor.substring(14);
    else if (actor.has_prefix("app://"))
      return actor.substring(6);
    else if (actor.has_prefix("/"))
      {
        return Path.get_basename (actor);
      }
    
    /* At this point we assume that @actor is already the basename of the
     * .desktop file */
    return actor;
  }
  
  public bool search_is_invalid (Unity.LensSearch? search)
  {
    /* This boolean expression is unfolded as we seem to get
    * some null dereference if we join them in a big || expression */
    if (search == null)
        return true;
    else if (search.search_string == null)
        return true;
  
    return search.search_string == "";
  }
  
  /* Sloppy and null-safe equality checking. null == "" and strings
   * are stripped of whitespace before comparison */
  private bool search_has_really_changed (Unity.LensSearch? old_search,
                                          Unity.LensSearch? new_search)
  {
    if (old_search == null && new_search == null)
      return false;
    
    string s1, s2;
   
    if (old_search == null)
      {
        return true;
      }
    else if (new_search == null)
      {
        s2 = old_search.search_string;
        if (s2 == null || s2.strip() == "")
          return false;
        else
          return true;
      }
    else
      {
        s1 = new_search.search_string;
        s2 = old_search.search_string;
        if (s1 == null)
          {
            if (s2 == null || s2.strip() == "")
              return false;
            else
              return true;
          }
        else if (s2 == null)
          {
            if (s1 == null || s1.strip() == "")
              return false;
            else
              return true;
          }
        else
          return s1.strip () != s2.strip ();
      }
  }

  /* Substitute tilde character in @s by the home directory.
   * Expansion of ~username also works if 'username' is found. */
  public string subst_tilde (string s)
  {
    int k;
    string name;
    unowned Passwd? pw;

    if (s[0] != '~')
      return s;

    if (s.length == 1 || s[1] == '/')
      return Environment.get_home_dir () + s.substring (1, -1);

    k = s.index_of ("/");
    if (k == -1)
      name = s.substring (1, -1);
    else
      name = s.substring (1, k-1);

    pw = Posix.getpwnam (name);
    if (pw == null)
      return s;

    if (k == -1)
      return pw.pw_dir;
    else
      return pw.pw_dir + s.substring (k, -1);
  }

}
