// -*- mode: js; js-indent-level: 4; indent-tabs-mode: nil -*-
var Cu = Components.utils;
var Ci = Components.interfaces;
var Cc = Components.classes;
var Cr = Components.results;

Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/ctypes.jsm");

Cu.import("resource://unity/unity-global-property-initializer.js");

var pbs = Cc["@mozilla.org/privatebrowsing;1"].getService(Ci.nsIPrivateBrowsingService);
var observerService = Cc["@mozilla.org/observer-service;1"].getService(Components.interfaces.nsIObserverService);

function BrandingRemover(mainWindow) {
    this._init(mainWindow);
}

BrandingRemover.prototype = {
    _value: '',
    _active: false,

    _init: function (mainWindow) {
	this._mainWindow = mainWindow;
    },

    _func: function (id, oldValue, newValue) {
	this._value = newValue;
	if (this._mainWindow.gBrowser.contentWindow.document.title)
	    return this._mainWindow.gBrowser.contentWindow.document.title;
	return newValue;
    },

    activate: function(aWindow) {
        this._active = true;
	this._mainWindow.document.watch('title', this._func.bind(this));
	this._mainWindow.document.title = this._mainWindow.document.title;
    },

    deactivate: function(aWindow) {
        if (!this._active)
            return;
        this._active = false;
	this._mainWindow.document.unwatch('title');
	if (this._value)
	    this._mainWindow.document.title = this._value;
	this._value = undefined;
    }
};

function WindowMonitor(domWindow) {
    this._init(domWindow);
}

WindowMonitor.prototype = {
    _active: null,
    _destroyed: false,
    _brandingRemover: null,

    _init: function(domWindow) {
        this._events = [];
        this._mainWindow = domWindow;
        this._brandingRemover = new BrandingRemover(domWindow);

        if (!this._mainWindow.gBrowser) {
            domWindow.addEventListener("load", function() {
                domWindow.removeEventListener("load", arguments.callee, false);
                this._realInit();
            }.bind(this), false);
        } else {
            this._realInit();
        }
    },

    _realInit: function() {
        if (this._destroyed)
            return;

	this._listenEvent(this._mainWindow.gBrowser.tabContainer, "TabSelect", this._onTabSelect);
        this._onTabSelect();
    },

    _onTabSelect: function() {
	let window = this._mainWindow.gBrowser.contentWindow;

        if (this._active)
            this._active.setActive(false);

        this._active = UnityGlobalPropertyInitializer.getExistingForWindow(window);

        var active = false;
        if (this._active) {
            this._active.setActive(true);
            active = this._active.hasContext();
        }

        if (active) {
            this._brandingRemover.activate();
        } else {
            this._brandingRemover.deactivate();
        }
    },

    _listenEvent: function(target, name, cb) {
        var descr = { target: target, name: name, cb: cb.bind(this) };

        target.addEventListener(name, descr.cb, false);

        this._events.push(descr);
    },

    _removeAllListeners: function() {
        for (var i = 0; i < this._events.length; i++) {
            let descr = this._events[i];
            descr.target.removeEventListener(descr.name, descr.cb, false);
        }
        this._events = [];
    },

    onNewContext: function() {
        this._onTabSelect();
    },

    deactivate: function() {
        this._destroyed = true;
        this._removeAllListeners();
        this._active = null;
    }
};

function forEachWindow(func) {
    var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
        .getService(Components.interfaces.nsIWindowMediator);
    var browserEnumerator = wm.getEnumerator("navigator:browser");

    var found = false;
    while (!found && browserEnumerator.hasMoreElements()) {
        var browserWin = browserEnumerator.getNext();

        func(browserWin);
    }

}

function Observer() {
    this._init();
}

Observer.prototype = {
    _init: function () {
        this._windows = new WeakMap();
    },

    activate: function() {
        forEachWindow(function(aWindow) {
            this._onWindowRegistered(aWindow.QueryInterface(Ci.nsIInterfaceRequestor).getInterface(Ci.nsIDOMWindowInternal || Ci.nsIDOMWindow));
        }.bind(this));

        observerService.addObserver(this, 'content-document-global-created', false);
        observerService.addObserver(this, 'xul-window-registered', false);
        observerService.addObserver(this, 'xul-window-destroyed', false);

        UnityGlobalPropertyInitializer.connect('on-new-context', function(aWindow) {
            let mainWindow = aWindow.QueryInterface(Components.interfaces.nsIInterfaceRequestor)
	        .getInterface(Components.interfaces.nsIWebNavigation)
	        .QueryInterface(Components.interfaces.nsIDocShellTreeItem)
	        .rootTreeItem
	        .QueryInterface(Components.interfaces.nsIInterfaceRequestor)
                .getInterface(Components.interfaces.nsIDOMWindow);
            this._windows.get(mainWindow).onNewContext();
        }.bind(this));
    },

    deactivate: function() {
        forEachWindow(function(aWindow) {
            this._onWindowDestroyed(aWindow.QueryInterface(Ci.nsIInterfaceRequestor).getInterface(Ci.nsIDOMWindowInternal || Ci.nsIDOMWindow));
        }.bind(this));

        UnityGlobalPropertyInitializer.disconnectAll();

        observerService.removeObserver(this, 'content-document-global-created');
        observerService.removeObserver(this, 'xul-window-registered');
        observerService.removeObserver(this, 'xul-window-destroyed');
    },

    _onWindowRegistered: function(domWindow) {
        this._windows.set(domWindow, new WindowMonitor(domWindow));
    },

    _onWindowDestroyed: function(domWindow) {
        if (this._windows.has(domWindow)) {
            this._windows.get(domWindow).deactivate();
            this._windows.delete(domWindow);
        }
    },

    observe : function(aWindow, aTopic, aData) {
        if (aWindow instanceof Ci.nsIDOMWindow && aTopic == 'content-document-global-created') {
            if (pbs.privateBrowsingEnabled)
                return;

            UnityGlobalPropertyInitializer.handle(aWindow);
        }
        if (aWindow instanceof Ci.nsIXULWindow && aTopic == 'xul-window-registered') {
            let domWindow = aWindow.QueryInterface(Ci.nsIInterfaceRequestor).getInterface(Ci.nsIDOMWindowInternal || Ci.nsIDOMWindow);

            this._onWindowRegistered(domWindow);
        }
        if (aWindow instanceof Ci.nsIXULWindow && aTopic == 'xul-window-destroyed') {
            let domWindow = aWindow.QueryInterface(Ci.nsIInterfaceRequestor).getInterface(Ci.nsIDOMWindowInternal || Ci.nsIDOMWindow);

            this._onWindowDestroyed(domWindow);
        }
    }
};

var EXPORTED_SYMBOLS = [ "Observer" ];
