# -*- coding: utf-8 -*-
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""A QApplication that starts a single instance."""

import sys

from PyQt4 import QtNetwork, QtGui, QtCore

from ubuntuone.controlpanel.logger import setup_logging


logger = setup_logging('uniqueapp')

# Arg with valid value
SOCKET_MESSAGES = {
    '--switch-to': ["folders", "share_links", "devices",
                    "settings", "account"],
}


class UniqueApplication(QtGui.QApplication):

    """A dummy UniqueApplication class."""

    new_instance = QtCore.pyqtSignal()
    switch_to = QtCore.pyqtSignal(unicode)
    activate_window = QtCore.pyqtSignal()

    def __init__(self, argv, key):
        super(UniqueApplication, self).__init__(argv)
        self.mapping_signals = {
            '--switch-to': self.switch_to,
        }
        self.key = key
        self.server = QtNetwork.QLocalServer(self)
        self.server.newConnection.connect(self.new_instance.emit)
        self.aboutToQuit.connect(self.cleanup)
        # Try to connect to existing app
        socket = QtNetwork.QLocalSocket()
        socket.connectToServer(key, QtCore.QIODevice.WriteOnly)
        if socket.waitForConnected(500):
            # Connected, exit
            self._send_messages(socket, argv)
            sys.exit()

        # Not connected, start server
        self.cleanup()
        self.ready = self.server.listen(key)
        if not self.ready:
            logger.debug(self.server.errorString())
        self.server.newConnection.connect(self._process_messages)

    def cleanup(self):
        """Remove the socket when we die."""
        self.server.removeServer(self.key)

    def _send_messages(self, socket, argv):
        """Send messages to the running application."""
        # This only take care of those args that are defined in SOCKET_MESSAGES
        # at this moment just "--switch-to", sending a message to the already
        # running client with: "arg=arg_value". If we have more than a single
        # arg, each arg is concatenated with "|".
        try:
            data = []
            size_argv = len(argv)
            for index in range(2, size_argv):
                if (argv[index] in SOCKET_MESSAGES and index < size_argv and
                   argv[index + 1] in SOCKET_MESSAGES[argv[index]]):
                    data.append('%s=%s' % (argv[index], argv[index + 1]))
            message = '|'.join(data)
            socket.write(message)
            socket.flush()
            socket.close()
        except:
            # The message couldn't be send through the socket,
            # but we avoided to open multiple instances of control panel.
            pass

    def _process_messages(self):
        """Get the messages from the other instances."""
        connection = self.server.nextPendingConnection()
        connection.waitForReadyRead()
        data = unicode(connection.readAll()).split('|')
        connection.close()
        for item in data:
            item_value = item.split('=')
            signal = self.mapping_signals.get(item_value[0])
            if signal:
                signal.emit(item_value[1])
        self.activate_window.emit()
