# -*- coding: utf-8 *-*

# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The search and popup widgets for the Share Links tab."""

import os

from PyQt4 import QtGui, QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.platform import expand_user

from ubuntuone.controlpanel import cache
from ubuntuone.controlpanel.logger import setup_logging

logger = setup_logging('qt.share_links_search')


def get_system_icon_for_filename(file_path):
    """Return the icon used for the system to represent this file."""
    fileinfo = QtCore.QFileInfo(expand_user(file_path))
    icon_provider = QtGui.QFileIconProvider()
    icon = icon_provider.icon(fileinfo)
    return icon


# pylint: disable=C0103

class SearchBox(QtGui.QLineEdit, cache.Cache):
    """Search widget for the synced files."""

    itemSelected = QtCore.pyqtSignal(unicode)

    def __init__(self, parent=None):
        super(SearchBox, self).__init__(parent)
        # The search box should be disabled until
        # the file list is obtained
        self.setEnabled(False)
        self.popup = FilesPopup()
        self.home_dir = ''
        self.temp_u1_files = []
        self.items_per_page = 0
        self.items_step = 20
        self.prefix = ''
        self._thread_explore = None
        self._pre_key_event = {
            QtCore.Qt.Key_Space: self._key_space_pressed,
        }
        self._post_key_event = {
            QtCore.Qt.Key_Escape: lambda *args: self.popup.hide(),
            QtCore.Qt.Key_Down: self._key_down_pressed,
            QtCore.Qt.Key_Up: self._key_up_pressed,
            QtCore.Qt.Key_Return: self._key_return_pressed,
            QtCore.Qt.Key_Enter: self._key_return_pressed,
        }

        self.popup.list_widget.itemPressed.connect(self._set_selected_item)
        self.popup.list_widget.verticalScrollBar().valueChanged.connect(
            self._scroll_fetch_more)
        self.textChanged.connect(self.filter)

        self._get_volumes_info()

    def _scroll_fetch_more(self, value):
        """Fetch more items into the list on scroll."""
        if self.popup.list_widget.verticalScrollBar().maximum() == value:
            filenames = self._get_filtered_list(self.temp_u1_files)
            self.popup.fetch_more(filenames)

    @inlineCallbacks
    def _get_volumes_info(self):
        """Get the volumes info."""
        self.home_dir = yield self.backend.get_home_dir()
        logger.info('Starting file list collection')
        self._thread_explore = ThreadExploreFolder(self.home_dir)
        info = yield self.backend.volumes_info()
        self._process_volumes_info(info)

    def _process_volumes_info(self, info):
        """Get the volumes paths and process them."""
        folders = []
        for _, _, data in info:
            for d in data:
                folder = d.get('path', d.get('realpath'))
                folders.append(folder)
        self.get_folders_files(folders)

    def get_folders_files(self, folders):
        """Get the list of files in each folder and index them."""
        self._thread_explore.set_folders(folders)
        self._thread_explore.folderDataObtained.connect(
            self._folder_content_obtained)
        self._thread_explore.start()

    def _folder_content_obtained(self):
        """Receive the content of the folders from the thread."""
        files = self._get_filtered_list(self._thread_explore.u1_files)
        logger.info('Finished file list collection, found %d files.',
            len(self._thread_explore.u1_files))
        self.setEnabled(True)
        self.popup.load_items(files)
        text = self.text()
        self.filter(text)

    def filter(self, text):
        """Filter the content of the popup with the text entered."""
        text = unicode(text)
        self.items_per_page = 0
        if text and (self.prefix == '' or self.prefix != text[:-1]):
            self.prefix = text
            self.temp_u1_files = self._thread_explore.u1_files
            self._show_filter()
        elif text != '':
            self.prefix = text
            self._show_filter()
        else:
            self.popup.hide()

    def _show_filter(self):
        """Load the items in the popup and display it."""
        if not self.popup.isVisible():
            self.popup.setFixedWidth(self.width())
            point = self.parent().mapToGlobal(self.pos())
            self.popup.show()
            self.popup.move(point.x(), point.y() + self.height())

        self.temp_u1_files = [filename for filename in self.temp_u1_files
            if os.path.basename(filename).find(self.prefix) > -1]
        files = self._get_filtered_list(self.temp_u1_files)
        self.popup.load_items(files)

    def _get_filtered_list(self, filenames):
        """Get pages of results."""
        begin = self.items_per_page
        self.items_per_page += self.items_step
        files = [filename for filename in filenames[begin:self.items_per_page]]
        return files

    def _key_space_pressed(self):
        """The user pressed the space key."""
        item = self.popup.list_widget.currentItem()
        if item is None:
            return False
        widget = self.popup.list_widget.itemWidget(item)
        self.setText(widget.name)
        self.popup.hide()
        return True

    def _key_down_pressed(self, current):
        """The user pressed the down key."""
        #While the current position is lower that the list size go to next
        if current != self.popup.list_widget.count() - 1:
            self.popup.list_widget.setCurrentRow(
                self.popup.list_widget.currentRow() + 1)
        #If the current position is greater than the amount of items in
        #the list - 6, then try to fetch more items in the list.
        if current >= (self.popup.list_widget.count() - 6):
            filenames = self._get_filtered_list(self.temp_u1_files)
            self.popup.fetch_more(filenames)

    def _key_up_pressed(self, current):
        """The user pressed the up key."""
        #while the current position is greater than 0, go to previous
        if current > 0:
            self.popup.list_widget.setCurrentRow(
                self.popup.list_widget.currentRow() - 1)

    def _key_return_pressed(self, current):
        """The user pressed the return key."""
        #If the user pressed enter, go to the item selected
        item = self.popup.list_widget.currentItem()
        self._set_selected_item(item)

    def keyPressEvent(self, event):
        """Process the different behaviour for the keyPress event."""
        if self._pre_key_event.get(event.key(), lambda: False)():
            return

        super(SearchBox, self).keyPressEvent(event)
        current = self.popup.list_widget.currentRow()
        self._post_key_event.get(event.key(), lambda *args: None)(current)

    def focusOutEvent(self, event):
        """Hide the popup when the window loses the focus."""
        super(SearchBox, self).focusOutEvent(event)
        self.popup.hide()

    def _set_selected_item(self, item):
        """Notify of the selected item."""
        widget = self.popup.list_widget.itemWidget(item)
        self.itemSelected.emit(widget.file_path)
        self.setText('')
        self.popup.hide()

    def moveEvent(self, event):
        """Move the popup when the windows is moved."""
        super(SearchBox, self).moveEvent(event)
        if self.popup.isVisible():
            point = self.mapToGlobal(self.line_search.pos())
            self.popup.move(point.x(), point.y() + self.line_search.height())


class FileItem(QtGui.QLabel):
    """Create a styled QLabel that will show the info."""

    def __init__(self, file_path):
        super(FileItem, self).__init__()
        self.name = os.path.basename(file_path)
        self.file_path = file_path
        self.text_style = (u"<span style='color: {2};'>{0}</span><br>"
            "<span style='font-size: 13px; color: {3};'>({1})</span>")
        self.setText(self.text_style.format(
                self.name, self.file_path, '#333333', 'grey'))

    def set_selected(self):
        """Set a style for the text when the item is selected."""
        self.setText(self.text_style.format(
                self.name, self.file_path, 'white', 'white'))

    def set_not_selected(self):
        """Set a style for the text when the item is not selected."""
        self.setText(self.text_style.format(
                self.name, self.file_path, '#333333', 'grey'))


class FilesPopup(QtGui.QFrame):
    """Filter popup where the file names are shown."""

    popupShown = QtCore.pyqtSignal()
    popupHidden = QtCore.pyqtSignal()

    def __init__(self):
        super(FilesPopup, self).__init__(None,
            QtCore.Qt.FramelessWindowHint | QtCore.Qt.ToolTip)
        vbox = QtGui.QVBoxLayout(self)
        vbox.setContentsMargins(0, 0, 0, 0)
        vbox.setSpacing(0)
        self.list_widget = QtGui.QListWidget()
        self.list_widget.setMinimumHeight(270)
        vbox.addWidget(self.list_widget)

        self.list_widget.currentItemChanged.connect(self._repaint_items)

    def _repaint_items(self, current, previous):
        """Set the proper style for the current and previous items."""
        if current is not None:
            widget = self.list_widget.itemWidget(current)
            widget.set_selected()
        if previous is not None:
            widget = self.list_widget.itemWidget(previous)
            widget.set_not_selected()

    def load_items(self, file_items):
        """Load the initial items."""
        self.list_widget.clear()
        for file_ in file_items:
            item = QtGui.QListWidgetItem("\n")
            file_widget = FileItem(file_)
            self.list_widget.addItem(item)
            self.list_widget.setItemWidget(item, file_widget)
            icon = get_system_icon_for_filename(file_.encode('utf-8'))
            item.setIcon(icon)
        if file_items:
            self.list_widget.setCurrentRow(0)

    def fetch_more(self, file_items):
        """Add more items to the list on user scroll."""
        for file_ in file_items:
            item = QtGui.QListWidgetItem("\n")
            file_widget = FileItem(file_)
            self.list_widget.addItem(item)
            self.list_widget.setItemWidget(item, file_widget)
            icon = get_system_icon_for_filename(file_.encode('utf-8'))
            item.setIcon(icon)

    def showEvent(self, event):
        """Notify when the popup is shown."""
        super(FilesPopup, self).showEvent(event)
        self.popupShown.emit()

    def hideEvent(self, event):
        """Notify when the popup is hidden."""
        super(FilesPopup, self).hideEvent(event)
        self.popupHidden.emit()


class ThreadExploreFolder(QtCore.QThread):
    """Retrieve the data of the folders asynchronously."""

    folderDataObtained = QtCore.pyqtSignal()

    def __init__(self, home_dir=''):
        super(ThreadExploreFolder, self).__init__()
        self.home_dir = home_dir
        self.u1_files = []
        self.folders = []

    def set_folders(self, folders):
        """Set the folders to be explored."""
        self.folders = folders

    def run(self):
        """Execute the thread to obtain the folders data."""
        folders_data = []
        for folder in self.folders:
            folders_data += self.get_folder_info(folder)
        temp_files = []
        for file_ in folders_data:
            if file_.startswith(self.home_dir):
                new_path = file_[len(self.home_dir):]
                if new_path.startswith(os.path.sep):
                    new_path = new_path[1:]
                path = os.path.join('~', new_path)
                temp_files.append(path)
            else:
                temp_files.append(file_)
        folders_data = temp_files
        folders_data.sort()
        self.u1_files = folders_data
        self.folderDataObtained.emit()

    def get_folder_info(self, folder):
        """Return the list of files in the proper folder."""
        files_list = []
        if os.path.exists(folder):
            for root, _, files in os.walk(folder):
                files_list.extend([os.path.join(root, f) for f in files])
        return files_list
