# -*- coding: utf-8 -*-

# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the show_quota_warning function."""

from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.controlpanel.gui import show_quota_warning


GB5 = 5368709120  # 5 Gb
GB25 = 26843545600  # 25 Gb


class QuotaWarningTestCase(TestCase):

    """Test cases for the humanize function."""

    def test_quota_warning_with_used_0_free_account(self):
        """Check the return value for show_quota_warning with 0 bytes used."""
        self.assertFalse(show_quota_warning(0, GB5))

    def test_quota_warning_with_free_account_used_medium(self):
        """Check the return value for show_quota_warning with 2.5 gb used."""
        used = 2.5 * (1024 ** 3)  # 2.5 Gb
        self.assertFalse(show_quota_warning(used, GB5))

    def test_quota_warning_with_free_account_used_almost_full(self):
        """Check the return value for show_quota_warning with 4.5 gb used."""
        # For free accounts the warning should be activate when the user
        # has equal or less than 500 mb remaining
        used = 4.5 * (1024 ** 3)  # 4.5 Gb
        self.assertTrue(show_quota_warning(used, GB5))

    def test_quota_warning_with_free_account_used_full(self):
        """Check the return value for show_quota_warning with 5 gb used."""
        # For free accounts the warning should be activate when the user
        # has equal or less than 500 mb remaining
        used = 5.0 * (1024 ** 3)  # 5.0 Gb
        self.assertTrue(show_quota_warning(used, GB5))

    def test_quota_warning_with_used_0_pay_account(self):
        """Check the return value for show_quota_warning with 0 bytes used."""
        self.assertFalse(show_quota_warning(0, GB25))

    def test_quota_warning_with_pay_account_used_medium(self):
        """Check the return value for show_quota_warning with 12.5 gb used."""
        used = 12.5 * (1024 ** 3)  # 12.5 Gb
        self.assertFalse(show_quota_warning(used, GB25))

    def test_quota_warning_with_pay_account_used_almost_full(self):
        """Check the return value for show_quota_warning with 22 gb used."""
        # For pay accounts the warning should be activate when the user
        # has equal or less than 3 Gb remaining
        used = 22 * (1024 ** 3)  # 22 Gb
        self.assertTrue(show_quota_warning(used, GB25))

    def test_quota_warning_with_pay_account_used_full(self):
        """Check the return value for show_quota_warning with 25 gb used."""
        # For free accounts the warning should be activate when the user
        # has equal or less than 3 gb remaining
        used = 25 * (1024 ** 3)  # 25 Gb
        self.assertTrue(show_quota_warning(used, GB25))
