# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the UniqueApplication class."""

from PyQt4 import QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.controlpanel.gui.qt.uniqueapp import windows as uniqueapp
from ubuntuone.controlpanel.tests import TestCase


#pylint: disable=C0103
class FakeLocalSocket(object):
    """A fake QLocalSocket."""

    def __init__(self):
        self.connect_calls = []
        self.connect_timeouts = []
        self.connect_succeeds = True

    def connectToServer(self, *args, **kwargs):
        """Fake connectToServer."""
        self.connect_calls.append((args, kwargs))

    def waitForConnected(self, timeout):
        """Fake waitForConnected."""
        self.connect_timeouts.append(timeout)
        return self.connect_succeeds


class FakeSignal(object):

    """A fake PyQt signal."""

    def __init__(self, *args, **kwargs):
        """Initialize."""
        self.target = None

    def connect(self, target):
        """Fake connect."""
        self.target = target

    def disconnect(self, *args):
        """Fake disconnect."""
        self.target = None

    def emit(self, *args):
        """Fake emit."""
        if self.target:
            self.target(*args)


class FakeLocalServer(object):

    """A fake QLocalServer."""

    def __init__(self):
        self.newConnection = FakeSignal()
        self.listen_args = []

    def listen(self, *args, **kwargs):
        """Fake listen."""
        self.listen_args.append((args, kwargs))


class FakeApplication(object):
    """A fake QApplication."""


class UniqueAppTestCase(TestCase):
    """Test the UniqueAppplication class."""

    @inlineCallbacks
    def setUp(self):
        yield super(UniqueAppTestCase, self).setUp()
        self.local_socket = FakeLocalSocket()
        self.patch(uniqueapp.QtNetwork, "QLocalSocket",
            lambda: self.local_socket)
        self.local_server = FakeLocalServer()
        self.patch(uniqueapp.QtNetwork, "QLocalServer",
            lambda parent: self.local_server)
        self.patch(uniqueapp.sys, "exit", self._set_called)
        self.fake_quit = FakeSignal()
        self.patch(uniqueapp.UniqueApplication, "aboutToQuit", self.fake_quit)
        self.patch(uniqueapp.QtGui, "QApplication", FakeApplication)

    def test_client_socket(self):
        """Check that the client socket is used correctly."""
        self.local_socket.connect_succeeds = True
        uniqueapp.UniqueApplication([], "key")
        self.assertEqual(self.local_socket.connect_calls,
        [(("key", QtCore.QIODevice.WriteOnly), {})])
        self.assertEqual(self.local_socket.connect_timeouts,
        [500])
        # The connection succeeds, so it should stop
        self.assertEqual(self._called, ((), {}))

    def test_server_socket(self):
        """Check that the server socket is used correctly."""
        self.local_socket.connect_succeeds = False
        uniqueapp.UniqueApplication([], "key")
        self.assertEqual(self.local_socket.connect_calls,
        [(("key", QtCore.QIODevice.WriteOnly), {})])
        self.assertEqual(self.local_socket.connect_timeouts,
        [500])

        # Should not exit
        self.assertEqual(self._called, False)

    def test_signal_connection(self):
        """Check that new_instance is correctly connected."""
        app = uniqueapp.UniqueApplication([], "key")
        # Yes, this is ugly. I can't find any other meaningful
        # way to compare them though.
        self.assertEqual(str(app.server.newConnection.target.__self__),
            str(app.new_instance))

    def test_cleanup(self):
        """Check that cleanup is called with the right key."""
        app = uniqueapp.UniqueApplication([], "key")
        self.assertEqual(self.fake_quit.target, app.cleanup)
