# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

# Unused import QtGui
# pylint: disable=W0611
from PyQt4 import QtGui, QtCore
# pylint: enable=W0611
from twisted.internet import defer

from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    DEVICES_LOCAL_LABEL,
    DEVICES_MANAGE_LABEL,
    DEVICES_REMOTE_LABEL,
    EDIT_DEVICES_LINK,
)
from ubuntuone.controlpanel.gui.qt.ui import devices_ui
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt import device


logger = setup_logging('qt.devices')


class DevicesPanel(UbuntuOneBin):
    """The DevicesFolders Tab Panel widget"""

    ui_class = devices_ui
    logger = logger

    localDeviceRemoved = QtCore.pyqtSignal()

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(DevicesPanel, self)._setup()
        self.ui.local_device_box.setTitle(DEVICES_LOCAL_LABEL)
        self.ui.other_devices.setTitle(DEVICES_REMOTE_LABEL)
        self.ui.manage_devices_button.setText(DEVICES_MANAGE_LABEL)
        self.ui.manage_devices_button.uri = EDIT_DEVICES_LINK

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load info."""
        self.is_processing = True
        info = yield self.backend.device_names_info()
        self.process_info(info)

    @log_call(logger.debug)
    def process_info(self, info):
        """Process and display the devices info."""
        self.ui.list_devices.clear()

        for device_info in info:
            self.update_device_info(device_info)

        self.is_processing = False

    def on_local_device_removed(self):
        """When the local device is removed, clear the box and emit signal."""
        self.ui.local_device.clear()
        self.localDeviceRemoved.emit()

    def update_device_info(self, device_info):
        """Update one device."""
        if device_info["is_local"]:
            self.update_local_device(device_info)
        else:
            self.create_remote_device(device_info)

    def update_local_device(self, device_info):
        """Update the info for the local device."""
        self.ui.local_device.update_device_info(device_info)
        self.ui.local_device.removed.connect(self.on_local_device_removed)

    def create_remote_device(self, device_info):
        """Add a remote device to the list."""
        widget = device.RemoteDeviceWidget()
        widget.update_device_info(device_info)
        item = QtGui.QListWidgetItem()

        self.ui.list_devices.addItem(item)
        self.ui.list_devices.setItemWidget(item, widget)
