# -*- coding: utf-8 -*-

# Authors: Roberto Alsina <roberto.alsina@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the url signing function."""

from urlparse import urlparse, parse_qs

from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.controlpanel.gui import sign_url, UBUNTUONE_FROM_OAUTH

TOKEN = {u'consumer_key': u'consumer_key',
         u'consumer_secret': u'consumer_secret',
         u'token_name': u'test_token',
         u'token': u'GkInOfSMGwTXAUoVQwLUoPxElEEUdhsLVNTPhxHJDUIeHCPNEo',
         u'token_secret': u'qFYImEtlczPbsCnYyuwLoPDlPEnvNcIktZphPQklAWrvyfFMV'}

SAMPLE_SIGNED = UBUNTUONE_FROM_OAUTH + '?oauth_nonce=' \
    '36886134&oauth_timestamp=1310671062&oauth_consumer_key=consumer_key&' \
    'oauth_signature_method=HMAC-SHA1&next=%2Fblah&oauth_version=1.0&' \
    'oauth_token=GkInOfSMGwTXAUoVQwLUoPxElEEUdhsLVNTPhxHJDUIeHCPNEo&' \
    'oauth_signature=s6h0LRBiWchTADrTJWaJUSuaGpo%3D'

#pylint: disable=E1101


class SignURLTestCase(TestCase):

    """Test cases for the URL signing function."""

    def test_is_correct_domain(self):
        """Test that we are using the right domain."""
        signed = sign_url("/blah", TOKEN)
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)
        self.assertEqual(parsed_signed.netloc, parsed_sample.netloc)

    def test_is_correct_path(self):
        """Test that we are using the right path in the URL."""
        signed = sign_url("/blah", TOKEN)
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)
        self.assertEqual(parsed_signed.path, parsed_sample.path)

    def test_is_correct_scheme(self):
        """Test that we are using the right scheme."""
        signed = sign_url("/blah", TOKEN)
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)

        self.assertEqual(parsed_signed.scheme, parsed_sample.scheme)

    def test_correct_query(self):
        """Test the invariant parts of the signed URL."""
        signed = sign_url("/blah", TOKEN)
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)
        signed_query = parse_qs(parsed_signed.query)
        sample_query = parse_qs(parsed_sample.query)

        for key in ('next',
            'oauth_consumer_key',
            'oauth_signature_method',
            'oauth_token',
            'oauth_version'):
            self.assertEqual("%s=%s" % (key, signed_query[key]),
            "%s=%s" % (key, sample_query[key]))

    def test_url_with_query(self):
        """Test that we are using the right scheme."""
        signed = sign_url("/blah?foo=bar", TOKEN)
        parsed_signed = urlparse(signed)
        signed_query = parse_qs(parsed_signed.query)

        self.assertEqual(signed_query['next'], ['/blah?foo=bar'])
