# -*- coding: utf-8 -*-

# Author: Roberto Alsina <roberto.alsina@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the start function."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import gui
from ubuntuone.controlpanel.gui.qt.tests import NO_OP
from ubuntuone.controlpanel.tests import TestCase


class FakeThing(object):

    """A fake thing."""

    def __init__(self):
        self.args = []
        self.shown = False

    def __call__(self, *args, **kwargs):
        self.args.append((args, kwargs))
        return self

    def show(self):
        """Show."""
        self.shown = True


class FakeReactor(object):
    """A fake reactor."""

    def run(self):
        """Start."""

    def stop(self):
        """Stop."""


class StartTestCase(TestCase):
    """Test the qt control panel."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(StartTestCase, self).setUp()
        self.main_window = FakeThing()
        self.tray_icon = FakeThing()
        self.patch(gui, "MainWindow", self.main_window)
        self.patch(gui, "TrayIcon", self.tray_icon)

    def test_minimized(self):
        """Test behaviour with minimized=True."""
        gui.start(NO_OP, minimized=True, with_icon=True)
        self.assertEqual(self.tray_icon.args, [((), {'window': None})])
        self.assertEqual(self.main_window.args, [])

    def test_with_icon(self):
        """Test behaviour with with_icon=True."""
        gui.start(NO_OP, with_icon=True, minimized=False)
        self.assertEqual(self.main_window.args, [((), {})])
        self.assertEqual(self.tray_icon.args, [((),
            {'window': self.main_window})])

    def test_both_false(self):
        """Test behaviour when with_icon and minimized are False."""
        gui.start(NO_OP, with_icon=False, minimized=False)
        # Should be called
        self.assertNotEqual(self.main_window.args, [])
        # Should not be called
        self.assertEqual(self.tray_icon.args, [])

    def test_both_true(self):
        """Test behaviour when with_icon and minimized are True."""
        gui.start(NO_OP, with_icon=True, minimized=True)
        self.assertEqual(self.tray_icon.args, [((), {'window': None})])
        self.assertEqual(self.main_window.args, [])
