# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Control Panel."""

from __future__ import division

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import controlpanel as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    SAMPLE_ACCOUNT_INFO, SAMPLE_NAME, TOKEN,
)
from ubuntuone.controlpanel.gui.qt.tests.test_ubuntuonebin import (
    UbuntuOneBinTestCase,
)


class ControlPanelTestCase(UbuntuOneBinTestCase):
    """Test the qt control panel."""

    innerclass_ui = gui.controlpanel_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.ControlPanel

    @defer.inlineCallbacks
    def setUp(self):
        yield super(ControlPanelTestCase, self).setUp()
        self.patch(self.ui.ui.folders_tab, 'process_info', lambda info: None)
        self.ui.backend.next_result = SAMPLE_ACCOUNT_INFO

    @defer.inlineCallbacks
    def test_is_processing_while_asking_info(self):
        """The ui is processing while the contents are loaded."""
        def check():
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')
            return {}

        self.patch(self.ui.backend, 'get_credentials', check)

        yield self.ui.load()  # trigger the info request
        self.assertFalse(self.ui.is_processing, 'ui must not be processing')

    @defer.inlineCallbacks
    def test_credentials_are_requested_on_load(self):
        """The info is requested to the backend."""
        yield self.ui.load()
        self.assert_backend_called('get_credentials')

    @defer.inlineCallbacks
    def test_on_credentials_not_found_called(self):
        """If no credentials, on_credentials_not_found is called."""
        self.patch(self.ui, 'on_credentials_not_found', self._set_called)
        self.patch(self.ui.backend, 'get_credentials', lambda: {})
        yield self.ui.load()

        self.assertEqual(self._called, ((), {}))

    def test_on_credentials_not_found(self):
        """The signin panel is shown."""
        self.ui.on_credentials_found()
        self.ui.on_credentials_not_found()
        self.assertIs(self.ui.ui.switcher.currentWidget(), self.ui.ui.signin)

    @defer.inlineCallbacks
    def test_on_credentials_found_called(self):
        """If credentials, on_credentials_not_found is called."""
        self.patch(self.ui, 'on_credentials_found', self._set_called)
        yield self.ui.load()

        self.assertEqual(self._called, ((), {}))

    def test_on_credentials_found(self):
        """The management panel is shown."""
        self.patch(self.ui, 'connect_file_sync', self._set_called)
        self.ui.on_credentials_not_found()
        self.ui.on_credentials_found()

        self.assertIs(self.ui.ui.switcher.currentWidget(),
                      self.ui.ui.management)
        self.assertEqual(self._called, ((), {}))

    @defer.inlineCallbacks
    def test_info_is_requested_on_load(self):
        """The info is requested to the backend."""
        self.patch(self.ui, 'process_info', self._set_called)
        yield self.ui.load()
        self.assert_backend_called('account_info')
        self.assertEqual(self._called, ((SAMPLE_ACCOUNT_INFO,), {}))

    def test_process_info(self):
        """The info is processed when ready."""
        info = SAMPLE_ACCOUNT_INFO
        self.ui.process_info(info)

        name = gui.NAME_STYLE % SAMPLE_NAME
        greeting = gui.GREETING % {'user_display_name': name}
        self.assertEqual(self.ui.ui.greeting_label.text(), greeting)

        used = int(info['quota_used'])
        total = int(info['quota_total'])
        percentage = round((used / total) * 100, 2)
        percentage_usage = {'percentage': gui.PERCENTAGE_STYLE % percentage}
        expected = {'used': gui.humanize(used),
                    'total': gui.humanize(total)}
        msg = gui.USAGE_LABEL % expected
        self.assertEqual(self.ui.ui.quota_usage_label.text(), msg)
        msg = gui.PERCENTAGE_LABEL % percentage_usage
        self.assertEqual(self.ui.ui.percentage_usage_label.text(), msg)

    def test_update_over_quota(self):
        """Check the labels state when the user exceed the quota."""
        percentage_value = 100
        # pylint: disable=W0212
        self.ui._update_quota({'percentage': percentage_value})
        # pylint: enable=W0212

        self.assertTrue(
            self.ui.ui.percentage_usage_label.property("OverQuota").toBool())
        self.assertTrue(
            self.ui.ui.quota_usage_label.property("OverQuota").toBool())

    def test_update_quota_in_range(self):
        """Check the labels state when the quota is under the threshold."""
        percentage_value = 50
        # pylint: disable=W0212
        self.ui._update_quota({'percentage': percentage_value})
        # pylint: enable=W0212

        self.assertFalse(
            self.ui.ui.percentage_usage_label.property("OverQuota").toBool())
        self.assertFalse(
            self.ui.ui.quota_usage_label.property("OverQuota").toBool())

    def test_on_local_device_removed(self):
        """On DeviesPanel's localDeviceRemoved, emit credentialsNotFound."""
        self.ui.ui.devices_tab.localDeviceRemoved.emit()
        self.assertIs(self.ui.ui.switcher.currentWidget(), self.ui.ui.signin)

    def test_on_signin_credentials_found(self):
        """On SignInPanel's credentialsFound, the management panel is shown."""
        self.patch(self.ui, 'load', self._set_called)
        self.ui.ui.signin.credentialsFound.emit(TOKEN)

        self.assertEqual(self._called, ((), {}))

    @defer.inlineCallbacks
    def test_connect_file_sync_with_autoconnect(self):
        """Connect is called if autoconnect is enabled."""
        settings = {gui.AUTOCONNECT_KEY: True}
        self.patch(self.ui.backend, 'file_sync_settings_info',
                   lambda: defer.succeed(settings))

        yield self.ui.connect_file_sync()

        self.assert_backend_called('connect_files')

    @defer.inlineCallbacks
    def test_connect_file_sync_without_autoconnect(self):
        """Connect is called if autoconnect is disabled."""
        settings = {gui.AUTOCONNECT_KEY: False}
        self.patch(self.ui.backend, 'file_sync_settings_info',
                   lambda: defer.succeed(settings))

        yield self.ui.connect_file_sync()

        # pylint: disable=W0212
        self.assertNotIn('connect_files', self.ui.backend._called)


class ExternalLinkButtonsTestCase(ControlPanelTestCase):
    """The link in the go-to-web buttons are correct."""

    def test_get_more_space_button(self):
        """When clicking the get more GB button, the proper url is opened."""
        self.assert_uri_hook_called(self.ui.ui.get_more_space_button,
                                    gui.EDIT_SERVICES_LINK)

    def test_help_button(self):
        """When clicking the help button, the proper url is opened."""
        self.assert_uri_hook_called(self.ui.ui.help_button,
                                    gui.GET_SUPPORT_LINK)

    def test_twitter_button(self):
        """When clicking the twitter button, the proper url is opened."""
        self.assert_uri_hook_called(self.ui.ui.twitter_button,
                                    gui.TWITTER_LINK)

    def test_facebook_button(self):
        """When clicking the facebook button, the proper url is opened."""
        self.assert_uri_hook_called(self.ui.ui.facebook_button,
                                    gui.FACEBOOK_LINK)
