# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The signin page."""

from PyQt4 import QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.gui import RESET_PASSWORD_LINK
from ubuntuone.controlpanel.gui.qt import icon_from_name, handle_errors
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt.ui import signin_ui
from ubuntuone.controlpanel.logger import setup_logging, log_call


logger = setup_logging('qt.signin')


class SignInPanel(UbuntuOneBin):
    """The widget for signing in."""

    ui_class = signin_ui
    logger = logger

    signinCanceled = QtCore.pyqtSignal()
    credentialsFound = QtCore.pyqtSignal(dict)

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(SignInPanel, self)._setup()

        self.ui.forgot_password_button.uri = RESET_PASSWORD_LINK
        icon = icon_from_name('external_icon_orange')
        self.ui.forgot_password_button.setIcon(icon)

        self.ui.signin_button.setEnabled(False)
        for entry in (self.ui.email_entry, self.ui.password_entry):
            entry.textChanged.connect(self.validate)
            entry.returnPressed.connect(self.ui.signin_button.click)

    def validate(self, *a, **kw):
        """Enable sign in button only if email and password are non empty."""
        email = unicode(self.ui.email_entry.text())
        password = unicode(self.ui.password_entry.text())
        self.ui.signin_button.setEnabled(bool(email and password))
        self.ui.signin_button.style().unpolish(self.ui.signin_button)
        self.ui.signin_button.style().polish(self.ui.signin_button)

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load specific tab info."""
        yield self.backend.get_credentials()

    @QtCore.pyqtSlot()
    @handle_errors(logger=logger)
    @log_call(logger.debug)
    @defer.inlineCallbacks
    def on_signin_button_clicked(self):
        """The 'Sign in' button was clicked."""
        email = unicode(self.ui.email_entry.text())
        password = unicode(self.ui.password_entry.text())
        self.is_processing = True
        try:
            result = yield self.backend.login(email=email, password=password)
            logger.info('Emitting credentialsFound for email %r.', email)
            self.credentialsFound.emit(result)
        finally:
            self.is_processing = False

    @QtCore.pyqtSlot()
    def on_cancel_button_clicked(self):
        """The 'Cancel' button was clicked."""
        self.signinCanceled.emit()
