# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from __future__ import division

from PyQt4 import QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.backend import AUTOCONNECT_KEY
from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    humanize,
    EDIT_SERVICES_LINK,
    FACEBOOK_LINK,
    GET_SUPPORT_LINK,
    GREETING,
    PERCENTAGE_LABEL,
    QUOTA_THRESHOLD,
    TWITTER_LINK,
    USAGE_LABEL,
)
from ubuntuone.controlpanel.gui import qt
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt.ui import controlpanel_ui


logger = setup_logging('qt.controlpanel')

NAME_STYLE = '<br><span style=" font-size:16pt;">%s!</span>'
PERCENTAGE_STYLE = '<span style=" font-size:16pt;">%.0f%%</span>'


class ControlPanel(UbuntuOneBin):
    """The Control Panel widget"""

    ui_class = controlpanel_ui
    logger = logger

    def _setup(self):
        """Do some extra setupping for the UI."""
        self.ui.get_more_space_button.uri = EDIT_SERVICES_LINK
        self.ui.help_button.uri = GET_SUPPORT_LINK
        self.ui.devices_tab.localDeviceRemoved.connect(
            self.on_credentials_not_found)
        self.ui.signin.credentialsFound.connect(lambda creds: self.load())

    @defer.inlineCallbacks
    def connect_file_sync(self):
        """Connect file sync service if the setting autoconnect is enabled."""
        settings = yield self.backend.file_sync_settings_info()
        if AUTOCONNECT_KEY in settings and settings[AUTOCONNECT_KEY]:
            yield self.backend.connect_files()

    @log_call(logger.debug)
    def on_credentials_not_found(self):
        """Credentials are not found or were removed."""
        self.ui.switcher.setCurrentWidget(self.ui.signin)
        self.is_processing = False

    @log_call(logger.debug)
    def on_credentials_found(self):
        """Credentials are not found or were removed."""
        self.ui.switcher.setCurrentWidget(self.ui.management)
        self.connect_file_sync()
        self.is_processing = False

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load info."""
        self.is_processing = True
        credentials = yield self.backend.get_credentials()
        if not credentials:
            self.on_credentials_not_found()
        else:
            self.on_credentials_found()
            info = yield self.backend.account_info()
            self.process_info(info)

    @log_call(logger.debug)
    def process_info(self, info):
        """Process and display the account info."""
        name = NAME_STYLE % info['name']
        self.ui.greeting_label.setText(GREETING % {'user_display_name': name})

        used = int(info['quota_used'])
        total = int(info['quota_total'])
        percentage_value = ((used / total) * 100)
        percentage = {'percentage': PERCENTAGE_STYLE % percentage_value}
        data = {'used': humanize(used), 'total': humanize(total)}
        self.ui.percentage_usage_label.setText(PERCENTAGE_LABEL % percentage)
        self.ui.quota_usage_label.setText(USAGE_LABEL % data)
        self._update_quota({'percentage': percentage_value})

    @log_call(logger.debug)
    def _update_quota(self, data=None):
        """Update the quota info."""
        fraction = 0.0
        if data is not None:
            fraction = data.get('percentage', 0.0) / 100
            if fraction > 0 and fraction < 0.05:
                fraction = 0.05
            else:
                fraction = round(fraction, 2)

        logger.debug('ManagementPanel: updating quota to %r.', fraction)
        self.ui.percentage_usage_label.setProperty("OverQuota",
            fraction >= QUOTA_THRESHOLD)
        self.ui.quota_usage_label.setProperty("OverQuota",
            fraction >= QUOTA_THRESHOLD)
        self.ui.percentage_usage_label.style().unpolish(
            self.ui.percentage_usage_label)
        self.ui.percentage_usage_label.style().polish(
            self.ui.percentage_usage_label)
        self.ui.quota_usage_label.style().unpolish(
            self.ui.quota_usage_label)
        self.ui.quota_usage_label.style().polish(self.ui.quota_usage_label)

    @QtCore.pyqtSlot()
    def on_twitter_button_clicked(self):
        """The twitter button was clicked."""
        qt.uri_hook(TWITTER_LINK)

    @QtCore.pyqtSlot()
    def on_facebook_button_clicked(self):
        """The facebook button was clicked."""
        qt.uri_hook(FACEBOOK_LINK)
