# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
# Authors: Natalia B. Bidart <nataliabidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the DBus service when accessing Single Sign On."""

# pylint: disable=C0103
# DBus signals have CamelCased names

import dbus

from ubuntu_sso import (DBUS_BUS_NAME,
    DBUS_CREDENTIALS_PATH, DBUS_CREDENTIALS_IFACE)
from twisted.internet.defer import inlineCallbacks

from ubuntuone.controlpanel import dbus_client
from ubuntuone.controlpanel.integrationtests import (MockDBusNoMethods,
    DBusClientTestCase)


SAMPLE_CREDS = {
    "token": "ABCDEF12345678",
    "access_token": "DEADCAFE2010",
}
OTHER_CREDS = {"token": "other!"}
SAMPLE_ERROR = {"error message": "test", "detailed_error": "error details"}

# pylint: disable=C0322
# pylint, you have to go to decorator's school


class MockDBusSSOService(dbus.service.Object):
    """A mock object that mimicks ussoc."""

    found = True
    wrong_app = None
    error = None

    @dbus.service.method(dbus_interface=DBUS_CREDENTIALS_IFACE,
                         in_signature='sa{ss}', out_signature='')
    def find_credentials(self, app_name, args):
        """Get creds from the keyring, login/register if needed."""
        if self.wrong_app is None and self.error is None:
            if self.found:
                self.CredentialsFound(app_name, SAMPLE_CREDS)
            else:
                self.CredentialsNotFound(app_name)
        elif self.wrong_app is not None and self.error is None:
            self.CredentialsFound(self.wrong_app, OTHER_CREDS)
        elif self.wrong_app is None and self.error is not None:
            self.CredentialsError(app_name, self.error)
        else:
            self.CredentialsError(self.wrong_app, self.error)

        self.CredentialsFound(app_name, SAMPLE_CREDS)

    @dbus.service.method(dbus_interface=DBUS_CREDENTIALS_IFACE,
                         in_signature='sa{ss}', out_signature='')
    def clear_credentials(self, app_name, args):
        """Clear the credentials for an application."""
        self.found = False

        if self.wrong_app is None and self.error is None:
            self.CredentialsCleared(app_name)
        elif self.wrong_app is not None and self.error is None:
            self.CredentialsCleared(self.wrong_app)
        elif self.wrong_app is None and self.error is not None:
            self.CredentialsError(app_name, self.error)
        else:
            self.CredentialsError(self.wrong_app, self.error)

        self.CredentialsCleared(app_name)

    @dbus.service.signal(DBUS_CREDENTIALS_IFACE, signature='sa{ss}')
    def CredentialsFound(self, app_name, credentials):
        """Signal thrown when the credentials are found."""

    @dbus.service.signal(DBUS_CREDENTIALS_IFACE, signature='s')
    def CredentialsNotFound(self, app_name):
        """Signal thrown when the credentials are not found."""

    @dbus.service.signal(DBUS_CREDENTIALS_IFACE, signature='s')
    def CredentialsCleared(self, app_name):
        """Signal thrown when the credentials were cleared."""

    @dbus.service.signal(DBUS_CREDENTIALS_IFACE, signature='s')
    def CredentialsStored(self, app_name):
        """Signal thrown when the credentials were cleared."""

    @dbus.service.signal(DBUS_CREDENTIALS_IFACE, signature='sa{ss}')
    def CredentialsError(self, app_name, error_dict):
        """Signal thrown when there is a problem getting the credentials."""


class SSOClientTestCase(DBusClientTestCase):
    """Test for the SSO dbus client."""

    def setUp(self):
        super(SSOClientTestCase, self).setUp()
        self.register_mockserver(DBUS_BUS_NAME, DBUS_CREDENTIALS_PATH,
                                 MockDBusSSOService)
        MockDBusSSOService.wrong_app = None
        MockDBusSSOService.error = None
        MockDBusSSOService.found = True

    # get_credentials

    @inlineCallbacks
    def test_get_credentials_ok(self):
        """Test the success case for get_credentials."""
        creds = yield dbus_client.get_credentials()
        self.assertEqual(creds, SAMPLE_CREDS)

    @inlineCallbacks
    def test_get_credentials_not_found(self):
        """Credentials were not found."""
        yield dbus_client.clear_credentials()  # not found will be sent
        yield self.assertFailure(dbus_client.get_credentials(),
                                 dbus_client.CredentialsError)

    @inlineCallbacks
    def test_get_credentials_other(self):
        """Creds for other apps are ignored."""
        MockDBusSSOService.wrong_app = 'other app!'
        creds = yield dbus_client.get_credentials()
        self.assertEqual(creds, SAMPLE_CREDS)

    @inlineCallbacks
    def test_get_credentials_error(self):
        """Test what happens when the creds can't be retrieved."""
        MockDBusSSOService.error = SAMPLE_ERROR
        yield self.assertFailure(dbus_client.get_credentials(),
                                 dbus_client.CredentialsError)

    @inlineCallbacks
    def test_get_credentials_other_error(self):
        """Other creds err before ours are retrieved."""
        MockDBusSSOService.wrong_app = 'other app!'
        MockDBusSSOService.error = SAMPLE_ERROR
        creds = yield dbus_client.get_credentials()
        self.assertEqual(creds, SAMPLE_CREDS)

    # clear_credentials

    @inlineCallbacks
    def test_clear_credentials_ok(self):
        """Test the success case for clear_credentials."""
        result = yield dbus_client.clear_credentials()
        self.assertEqual(result, dbus_client.APP_NAME)

    @inlineCallbacks
    def test_clear_credentials_other(self):
        """Creds for other apps are ignored."""
        MockDBusSSOService.wrong_app = 'other app!'
        result = yield dbus_client.clear_credentials()
        self.assertEqual(result, dbus_client.APP_NAME)

    @inlineCallbacks
    def test_clear_credentials_error(self):
        """Test what happens when the creds can't be retrieved."""
        MockDBusSSOService.error = SAMPLE_ERROR
        yield self.assertFailure(dbus_client.clear_credentials(),
                                 dbus_client.CredentialsError)

    @inlineCallbacks
    def test_clear_credentials_other_error(self):
        """Other creds err before ours are retrieved."""
        MockDBusSSOService.wrong_app = 'other app!'
        MockDBusSSOService.error = SAMPLE_ERROR
        result = yield dbus_client.clear_credentials()
        self.assertEqual(result, dbus_client.APP_NAME)


class NoMethodsSSOClientTestCase(DBusClientTestCase):
    """Test for the SSO dbus client when the service provides no methods."""

    def setUp(self):
        super(NoMethodsSSOClientTestCase, self).setUp()
        self.register_mockserver(DBUS_BUS_NAME, DBUS_CREDENTIALS_PATH,
                                 MockDBusNoMethods)

    @inlineCallbacks
    def test_get_credentials_dbus_error(self):
        """Test what happens when there's a DBus error."""
        yield self.assertFailure(dbus_client.get_credentials(),
                                 dbus.DBusException)

    @inlineCallbacks
    def test_clear_credentials_dbus_error(self):
        """Test what happens when there's a DBus error."""
        yield self.assertFailure(dbus_client.clear_credentials(),
                                 dbus.DBusException)
