# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the extra widgets."""

from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.controlpanel.gtk import widgets


class LoadingTestCase(TestCase):
    """Test suite for the Loading widget (a spinner plus a label)."""

    def setUp(self):
        super(LoadingTestCase, self).setUp()
        self.label = 'A loading tests...'
        self.loading = widgets.Loading(self.label)

    def test_creation(self):
        """A Loading can be created."""
        self.assertEqual(self.loading.label.get_text(), self.label)
        self.assertEqual(self.loading.spinner.get_property('active'), True)

    def test_visibility(self):
        """All widgets are visible."""
        self.assertTrue(self.loading.label.get_visible())
        self.assertTrue(self.loading.spinner.get_visible())

    def test_children(self):
        """A Loading can be created."""
        children = self.loading.get_children()
        self.assertEqual(2, len(children))
        self.assertTrue(self.loading.spinner is children[0])
        self.assertTrue(self.loading.label is children[-1])

    def test_fg_color(self):
        """Foreground color can be set."""
        expected = '#654321'
        widget = widgets.Loading(self.label, fg_color=expected)

        win = widgets.gtk.Window()
        win.add(widget)
        win.realize()  # ensure realization

        self.assertEqual(widget.label.style.fg[widgets.gtk.STATE_NORMAL],
                         widgets.gtk.gdk.Color(expected))


class LabelLoadingTestCase(TestCase):
    """Test suite for the LabelLoading widget.

    (a label that shows a Loading until the text is set).

    """

    def setUp(self):
        super(LabelLoadingTestCase, self).setUp()
        self.label = 'A label loading tests...'
        self.widget = widgets.LabelLoading(self.label)

    def test_creation(self):
        """A LabelLoading can be created."""
        self.assertEqual(self.widget.label.get_text(), '')
        self.assertTrue(self.widget.label.get_visible())

        self.assertIsInstance(self.widget.loading, widgets.Loading)
        self.assertTrue(self.widget.loading.get_visible())
        self.assertTrue(self.widget.active)  # loading label is packed

    def test_stop(self):
        """Stop hides the Loading widget."""
        self.widget.stop()

        self.assertTrue(self.widget.get_child() is self.widget.label)
        self.assertFalse(self.widget.active)

    def test_start(self):
        """Start shows the Loading widget."""
        self.widget.start()

        self.assertTrue(self.widget.get_child() is self.widget.loading)
        self.assertTrue(self.widget.active)

    def test_get_text(self):
        """Text can be queried."""
        text = 'Test me.'
        self.widget.label.set_text(text)
        self.assertEqual(self.widget.get_text(), text)

    def test_set_text(self):
        """Text can be set."""
        text = 'Test me.'
        self.widget.set_text(text)
        self.assertEqual(self.widget.label.get_text(), text)

    def test_get_label(self):
        """Label (markup text) can be queried."""
        text = 'Test <b>me</b>.'
        self.widget.label.set_markup(text)
        self.assertEqual(self.widget.get_label(), text)

    def test_set_markup(self):
        """Markup can be set."""
        text = 'Test <b>me</b>.'
        self.widget.set_markup(text)
        self.assertEqual(self.widget.label.get_label(), text)

    def test_fg_color(self):
        """Foreground color can be set."""
        expected = '#123456'
        widget = widgets.LabelLoading(self.label, fg_color=expected)
        widget.stop()  # show label instead of loading

        win = widgets.gtk.Window()
        win.add(widget)
        win.realize()  # ensure realization

        assert widget.label.get_visible()
        self.assertEqual(widget.label.style.fg[widgets.gtk.STATE_NORMAL],
                         widgets.gtk.gdk.Color(expected))

    def test_fg_color_loading(self):
        """Foreground color can be set to loading label."""
        expected = '#123456'
        widget = widgets.LabelLoading(self.label, fg_color=expected)

        win = widgets.gtk.Window()
        win.add(widget)
        win.realize()  # ensure realization

        assert widget.loading.get_visible()
        actual = widget.loading.label.style.fg[widgets.gtk.STATE_NORMAL]
        self.assertEqual(actual, widgets.gtk.gdk.Color(expected))
        actual = widget.loading.spinner.style.fg[widgets.gtk.STATE_NORMAL]
        self.assertEqual(actual, widgets.gtk.gdk.Color(expected))

    def test_selectable(self):
        """The widget is selectable."""
        self.assertTrue(self.widget.label.get_selectable())


class PanelTitleTestCase(TestCase):
    """Tets case for a special title for each management panel."""

    # pylint: disable=W0612

    TITLE = '<b>Foo Bar</b>'

    def setUp(self):
        super(PanelTitleTestCase, self).setUp()
        self.widget = widgets.PanelTitle(markup=self.TITLE)

        win = widgets.gtk.Window()
        win.add(self.widget)
        win.realize()  # ensure realization

    def test_visibility(self):
        """The widget and children visibility is correct."""
        self.assertTrue(self.widget.get_visible())

    def test_label_markup(self):
        """The label markup is correct."""
        self.assertEqual(self.widget.get_label(), self.TITLE)

    def test_label_xalign(self):
        """The label is left aligned."""
        self.assertEqual(self.widget.get_property('xalign'), 0.0)

    def test_label_line_wrap(self):
        """The label is left aligned."""
        self.assertTrue(self.widget.get_line_wrap())
        self.assertEqual(self.widget.get_line_wrap_mode(),
                         widgets.pango.WRAP_WORD)

    def test_label_padding(self):
        """The label padding is correct."""
        self.assertEqual(self.widget.get_padding(),
                         widgets.DEFAULT_PADDING)

    def test_selectable(self):
        """The widget is selectable."""
        self.assertTrue(self.widget.get_selectable())
