# -*- coding: utf-8 -*-

# Authors: Natalia B. Bidart <nataliabidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the package manager service."""

import collections

try:
    # Unable to import 'defer', pylint: disable=F0401,E0611,W0404
    from aptdaemon import defer
except ImportError:
    # Unable to import 'defer', pylint: disable=F0401,E0611,W0404
    import defer

from ubuntuone.controlpanel.gtk import package_manager
from ubuntuone.controlpanel.tests import TestCase


FAKED_CACHE = {}
SUCCESS = package_manager.aptdaemon.enums.EXIT_SUCCESS
FAILURE = package_manager.aptdaemon.enums.EXIT_FAILED


class FakedPackage(object):
    """Fake a package."""

    def __init__(self, name=None, is_installed=False):
        self.name = name
        self.is_installed = is_installed


class FakedTransaction(object):
    """Fake a transaction."""

    failure = None

    def __init__(self, packages, cache=None):
        self._signals = collections.defaultdict(list)
        self._cache = cache
        self.was_run = False
        self.packages = packages
        self.connect = lambda sig, f: self._signals[sig].append(f)

    def run(self):
        """Run!"""
        self.was_run = True

        if self._cache is not None:
            for package in self.packages:
                FAKED_CACHE[package].is_installed = True

        if self.failure is None:
            code = SUCCESS
        else:
            code = FAILURE

        for listener in self._signals['finished']:
            listener(self, code)

        d = defer.Deferred()
        d.callback(code)
        return d


class FakedClient(object):
    """Fake an apt client."""

    def install_packages(self, packages_names, **kwargs):
        """Install packages listed in 'package_names'.

        package_names - a list of package names
        wait - if True run the transaction immediately and return its exit
            state instead of the transaction itself.
        reply_handler - callback function. If specified in combination with
            error_handler the method will be called asynchrounsouly.
        error_handler - in case of an error the given callback gets the
            corresponding DBus exception instance

        """
        if kwargs.get('wait', False):
            return package_manager.aptdaemon.enums.EXIT_FAILED
        d = defer.Deferred()
        d.callback(FakedTransaction(packages_names, cache=FAKED_CACHE))
        return d


class PackageManagerTestCase(TestCase):
    """Test for the package manager."""

    timeout = 2

    def setUp(self):
        FAKED_CACHE.clear()  # clean cache
        self.patch(package_manager.apt, 'Cache', lambda: FAKED_CACHE)
        self.patch(package_manager.aptdaemon.client, 'AptClient', FakedClient)
        self.obj = package_manager.PackageManager()

    def test_is_installed(self):
        """Check that a package is installed."""
        name = 'test'
        FAKED_CACHE[name] = FakedPackage(name, is_installed=True)

        result = self.obj.is_installed(name)

        self.assertTrue(result, 'must be installed')

    def test_is_not_installed(self):
        """Check if a package is installed."""
        name = 'test'
        FAKED_CACHE[name] = FakedPackage(name)  # not installed by default

        result = self.obj.is_installed(name)

        self.assertFalse(result, 'must not be installed')

    def test_is_not_installed_if_key_error(self):
        """Check if a package is installed when cache raises KeyError."""
        name = 'test'  # is not in the cache
        result = self.obj.is_installed(name)

        self.assertFalse(result, 'must not be installed')

    def test_progress_bar(self):
        """The progress bar class is correct."""
        self.assertIsInstance(package_manager.PackageManagerProgressBar(),
                              package_manager.AptProgressBar)

    @package_manager.inline_callbacks
    def test_install(self):
        """Install is correct."""
        name = 'test'
        FAKED_CACHE[name] = FakedPackage(name)  # not installed by default

        result = yield self.obj.install(name)

        self.assertIsInstance(result, FakedTransaction)
        self.assertFalse(result.was_run, 'transaction must not be run')

    @package_manager.inline_callbacks
    def test_transaction_install(self):
        """Install is correct."""
        name = 'test'
        FAKED_CACHE[name] = FakedPackage(name)  # not installed by default

        trans = yield self.obj.install(name)

        trans.connect('finished', self._set_called)
        trans.run()

        self.assertEqual(self._called, ((trans, SUCCESS), {}))
        self.assertTrue(trans.was_run, 'transaction must be run')
        self.assertTrue(self.obj.is_installed(name))

    @package_manager.inline_callbacks
    def test_install_if_installed(self):
        """Install does nothing is package is already installed."""
        name = 'test'
        FAKED_CACHE[name] = FakedPackage(name, is_installed=True)

        result = yield self.obj.install(name)

        self.assertEqual(result, SUCCESS)
