# -*- coding: utf-8 -*-
#
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the Zeitgeist logging."""

import os
import shutil
import subprocess
import tempfile
import time

from distutils.spawn import find_executable

from twisted.internet.defer import Deferred, inlineCallbacks
from zeitgeist.client import ZeitgeistClient
from zeitgeist.datamodel import Event, Subject, Interpretation, Manifestation

from tests.platform.linux.test_dbus import DBusTwistedTestCase
from ubuntuone.eventlog.zglog import ZeitgeistLogger

SRCDIR = os.environ.get('SRCDIR', os.getcwd())


class NotFoundError(Exception):
    """Not found error."""


class ZeitgeistNotStartedTests(DBusTwistedTestCase):
    """Tests for the zeitgeist logging module."""

    def test_log_does_not_err_when_daemon_not_started(self):
        """zeitgeist-daemon was not started."""
        timestamp = int(time.time() * 1000)
        subject = Subject.new_for_values(
                uri="file:///tmp/folder1",
                interpretation=Interpretation.FOLDER,
                manifestation=Manifestation.FILE_DATA_OBJECT,
                origin="ubuntuone:uuid",
                mimetype="inode/directory",
                text="sample folder"
        )
        sample_event = Event.new_for_values(
            timestamp=timestamp,
            interpretation=Interpretation.ACCESS_EVENT,
            manifestation=Manifestation.USER_ACTIVITY,
            actor="mailto:sample_subject",
            subjects=[subject]
        )

        zg = ZeitgeistLogger()
        d = zg.log(sample_event)
        self.assertEqual(zg.client, None)

        def verify(result):
            """Stored result is the empty list, because zg not available."""
            self.assertEqual(result, [])
            return result

        d.addCallback(verify)
        return d


def wait_zeitgeist_started(seconds=10):
    """Wait a few seconds until zg is started, or fail if it can't."""
    client = None
    count = 0
    while client is None:
        count += 1
        try:
            client = ZeitgeistClient()
            break
        except RuntimeError:
            if count > seconds*10:
                raise
            time.sleep(0.1)


class ZeitgeistTestCase(DBusTwistedTestCase):
    """Tests for the zeitgeist logging module."""

    @inlineCallbacks
    def setUp(self):
        yield super(ZeitgeistTestCase, self).setUp()
        zgdaemon = find_executable("zeitgeist-daemon")
        if not zgdaemon:
            raise NotFoundError("zeitgeist-daemon was not found.")

        tempfolder = tempfile.mkdtemp(prefix="test-u1-zeitgeist-")
        tempstdout = tempfile.TemporaryFile(prefix="test-u1-stdout-")
        tempstderr = tempfile.TemporaryFile(prefix="test-u1-stderr-")
        os.environ["ZEITGEIST_DATA_PATH"] = tempfolder
        p = subprocess.Popen([zgdaemon], bufsize=4096, stdout=tempstdout,
                             stderr=tempstderr)
        def cleanup():
            """Wait for the process to finish."""
            p.terminate()
            p.wait()
            del(os.environ["ZEITGEIST_DATA_PATH"])
            shutil.rmtree(tempfolder)

        wait_zeitgeist_started()
        self.addCleanup(cleanup)

    def test_log_records_the_event(self):
        """The log method records the event in zg."""
        timestamp = int(time.time() * 1000)
        subject = Subject.new_for_values(
                uri="file:///tmp/folder1",
                interpretation=Interpretation.FOLDER,
                manifestation=Manifestation.FILE_DATA_OBJECT,
                origin="ubuntuone:uuid",
                mimetype="inode/directory",
                text="sample folder"
        )
        sample_event = Event.new_for_values(
            timestamp=timestamp,
            interpretation=Interpretation.ACCESS_EVENT,
            manifestation=Manifestation.USER_ACTIVITY,
            actor="mailto:sample_subject",
            subjects=[subject]
        )

        sample_template = Event.new_for_values()

        zg = ZeitgeistLogger()
        self.assertNotEqual(zg.client, None)

        d2 = Deferred()

        def logged(id_list):
            """The event was logged to zeitgeist."""

            def events_found(event_list):
                """zg returned the list of events."""
                self.assertEqual(event_list[0].id, id_list[0])
                d2.callback("ok")

            zg.client.find_events_for_template(sample_template, events_found)

        d = zg.log(sample_event)
        d.addCallbacks(logged, d2.errback)

        return d2
