# ubuntuone.platform.linux.vm_helper- vm helpers for linux.
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""vm helpers for linux."""
import os
import stat
import re
import shutil

from ubuntuone.storageprotocol import request

platform = "linux2"


def create_shares_link(source, dest):
    """Create the shares symlink."""
    if not os.path.exists(dest):
        # remove the symlink if it's broken
        if os.path.islink(dest) and \
           os.readlink(dest) != source:
            os.remove(dest)
        os.symlink(source, dest)
        return True
    else:
        return False


def get_udf_path_name(path):
    """Return (path, name) from path.

    path must be a path inside the user home direactory, if it's not
    a ValueError is raised.
    """
    if not path:
        raise ValueError("no path specified")

    user_home = os.path.expanduser('~')
    start_list = os.path.abspath(user_home).split(os.path.sep)
    path_list = os.path.abspath(path).split(os.path.sep)

    # Work out how much of the filepath is shared by user_home and path.
    common_prefix = os.path.commonprefix([start_list, path_list])
    if os.path.sep + os.path.join(*common_prefix) != user_home:
        raise ValueError("path isn't inside user home: %r" % path)

    i = len(common_prefix)
    rel_list = [os.path.pardir] * (len(start_list)-i) + path_list[i:]
    relpath = os.path.join(*rel_list)
    head, tail = os.path.split(relpath)
    if head == '':
        head = '~'
    else:
        head = os.path.join('~', head)
    return head, tail


def get_udf_path(suggested_path):
    """Build the udf path using the suggested_path."""
    # Unicode boundary! the name is Unicode in protocol and server,
    # but here we use bytes for paths
    return os.path.expanduser(suggested_path).encode("utf8")


def get_share_path(share):
    """Builds the root path of a share using the share information."""
    if hasattr(share, 'volume_id'):
        share_id = share.volume_id
    elif hasattr(share, 'share_id'):
        share_id = share.share_id
    else:
        share_id = share.id

    if hasattr(share, 'name'):
        share_name = share.name
    else:
        share_name = share.share_name

    if hasattr(share, 'other_visible_name'):
        visible_name = share.other_visible_name
    else:
        visible_name = share.from_visible_name

    if visible_name:
        dir_name = '%s (%s, %s)' % (share_name, visible_name, share_id)
    else:
        dir_name = '%s (%s)' % (share_name, share_id)

    # Unicode boundary! the name is Unicode in protocol and server,
    # but here we use bytes for paths
    dir_name = dir_name.encode("utf8")

    return dir_name


class VMMetadataUpgraderMixIn(object):

    def _guess_metadata_version(self):
        """Try to guess the metadata version based on current metadata
        and layout, fallbacks to md_version = None if can't guess it.

        """
        from ubuntuone.syncdaemon.volume_manager import LegacyShareFileShelf, _Share
        md_version = None
        if os.path.exists(self._shares_md_dir) \
           and os.path.exists(self._shared_md_dir):
            # we have shares and shared dirs
            # md_version >= 1
            old_root_dir = os.path.join(self._root_dir, 'My Files')
            old_share_dir = os.path.join(self._root_dir, 'Shared With Me')
            if os.path.exists(old_share_dir) and os.path.exists(old_root_dir) \
               and not os.path.islink(old_share_dir):
                # md >= 1 and <= 3
                # we have a My Files dir, 'Shared With Me' isn't a
                # symlink and ~/.local/share/ubuntuone/shares doesn't
                # exists.
                # md_version <= 3, set it to 2 as it will migrate
                # .conflict to .u1conflict, and we don't need to upgrade
                # from version 1 any more as the LegacyShareFileShelf
                # takes care of that.
                md_version = '2'
            else:
                try:
                    target = os.readlink(self._shares_dir_link)
                except OSError:
                    target = None
                if os.path.islink(self._shares_dir_link) \
                   and os.path.normpath(target) == self._shares_dir_link:
                    # broken symlink, md_version = 4
                    md_version = '4'
                else:
                    # md_version >= 5
                    shelf = LegacyShareFileShelf(self._shares_md_dir)
                    # check a pickled value to check if it's in version
                    # 5 or 6
                    md_version = '5'
                    versions = {'5':0, '6':0}
                    for key in shelf:
                        share = shelf[key]
                        if isinstance(share, _Share):
                            versions['5'] += 1
                        else:
                            versions['6'] += 1
                    if versions['5'] > 0:
                        md_version = '5'
                    elif versions['6'] > 0:
                        md_version = '6'
        else:
            # this is metadata 'None'
            md_version = None
        return md_version

    def _upgrade_metadata_None(self, md_version):
        """Upgrade the shelf layout, for *very* old clients."""
        from ubuntuone.syncdaemon.volume_manager import LegacyShareFileShelf
        self.log.debug('Upgrading the share shelf layout')
        # the shelf already exists, and don't have a .version file
        # first backup the old data
        backup = os.path.join(self._data_dir, '0.bkp')
        if not os.path.exists(backup):
            os.makedirs(backup)
        # pylint: disable-msg=W0612
        # filter 'shares' and 'shared' dirs, in case we are in the case of
        # missing version but existing .version file
        filter_known_dirs = lambda d: d != os.path.basename(self._shares_md_dir)\
                and d != os.path.basename(self._shared_md_dir)
        for dirname, dirs, files in os.walk(self._data_dir):
            if dirname == self._data_dir:
                for dir in filter(filter_known_dirs, dirs):
                    if dir != os.path.basename(backup):
                        shutil.move(os.path.join(dirname, dir),
                                    os.path.join(backup, dir))
        # regenerate the shelf using the new layout using the backup as src
        old_shelf = LegacyShareFileShelf(backup)
        if not os.path.exists(self._shares_dir):
            os.makedirs(self._shares_dir)
        new_shelf = LegacyShareFileShelf(self._shares_md_dir)
        for key, share in old_shelf.iteritems():
            new_shelf[key] = share
        # now upgrade to metadata 2
        self._upgrade_metadata_2(md_version)

    def _upgrade_metadata_1(self, md_version):
        """Upgrade to version 2.

        Upgrade all pickled Share to the new package/module layout.

        """
        from ubuntuone.syncdaemon.volume_manager import LegacyShareFileShelf
        self.log.debug('upgrading share shelfs from metadata 1')
        shares = LegacyShareFileShelf(self._shares_md_dir)
        for key, share in shares.iteritems():
            shares[key] = share
        shared = LegacyShareFileShelf(self._shared_md_dir)
        for key, share in shared.iteritems():
            shared[key] = share
        # now upgrade to metadata 3
        self._upgrade_metadata_2(md_version)

    def _upgrade_metadata_2(self, md_version):
        """Upgrade to version 3

        Renames foo.conflict files to foo.u1conflict, foo.conflict.N
        to foo.u1conflict.N, foo.partial to .u1partial.foo, and
        .partial to .u1partial.

        """
        from ubuntuone.platform.linux import allow_writes
        self.log.debug('upgrading from metadata 2 (bogus)')
        for top in self._root_dir, self._shares_dir:
            for dirpath, dirnames, filenames in os.walk(top):
                with allow_writes(dirpath):
                    for names in filenames, dirnames:
                        self._upgrade_names(dirpath, names)
        self._upgrade_metadata_3(md_version)

    def _upgrade_names(self, dirpath, names):
        """Do the actual renaming for _upgrade_metadata_2."""
        for pos, name in enumerate(names):
            new_name = name
            if re.match(r'.*\.partial$|\.u1partial(?:\..+)?', name):
                if name == '.partial':
                    new_name = '.u1partial'
                else:
                    new_name = re.sub(r'^(.+)\.partial$',
                                      r'.u1partial.\1', name)
                if new_name != name:
                    while os.path.lexists(os.path.join(dirpath, new_name)):
                        # very, very strange
                        self.log.warning('Found a .partial and .u1partial'
                                         ' for the same file: %s!', new_name)
                        new_name += '.1'
            elif re.search(r'\.(?:u1)?conflict(?:\.\d+)?$', name):
                new_name = re.sub(r'^(.+)\.conflict((?:\.\d+)?)$',
                                  r'\1.u1conflict\2', name)
                if new_name != name:
                    while os.path.lexists(os.path.join(dirpath, new_name)):
                        m = re.match(r'(.*\.u1conflict)((?:\.\d+)?)$', new_name)
                        base, num = m.groups()
                        if not num:
                            num = '.1'
                        else:
                            num = '.' + str(int(num[1:])+1)
                        new_name = base + num
            if new_name != name:
                old_path = os.path.join(dirpath, name)
                new_path = os.path.join(dirpath, new_name)
                self.log.debug('renaming %r to %r', old_path, new_path)
                os.rename(old_path, new_path)
                names[pos] = new_name


    def _upgrade_metadata_3(self, md_version):
        """Upgrade to version 4 (new layout!)

        move "~/Ubuntu One/Shared With" Me to XDG_DATA/ubuntuone/shares
        move "~/Ubuntu One/My Files" contents to "~/Ubuntu One"

        """
        from ubuntuone.syncdaemon.volume_manager import LegacyShareFileShelf
        self.log.debug('upgrading from metadata 3 (new layout)')
        old_share_dir = os.path.join(self._root_dir, 'Shared With Me')
        old_root_dir = os.path.join(self._root_dir, 'My Files')
        # change permissions
        os.chmod(self._root_dir, 0775)

        def move(src, dst):
            """Move a file/dir taking care if it's read-only."""
            prev_mode = stat.S_IMODE(os.stat(src).st_mode)
            os.chmod(src, 0755)
            shutil.move(src, dst)
            os.chmod(dst, prev_mode)

        # update the path's in metadata and move the folder
        if os.path.exists(old_share_dir) and not os.path.islink(old_share_dir):
            os.chmod(old_share_dir, 0775)
            if not os.path.exists(os.path.dirname(self._shares_dir)):
                os.makedirs(os.path.dirname(self._shares_dir))
            self.log.debug('moving shares dir from: %r to %r',
                           old_share_dir, self._shares_dir)
            for path in os.listdir(old_share_dir):
                src = os.path.join(old_share_dir, path)
                dst = os.path.join(self._shares_dir, path)
                move(src, dst)
            os.rmdir(old_share_dir)

        # update the shares metadata
        shares = LegacyShareFileShelf(self._shares_md_dir)
        for key, share in shares.iteritems():
            if share.path is not None:
                if share.path == old_root_dir:
                    share.path = share.path.replace(old_root_dir,
                                                    self._root_dir)
                else:
                    share.path = share.path.replace(old_share_dir,
                                                    self._shares_dir)
                shares[key] = share

        shared = LegacyShareFileShelf(self._shared_md_dir)
        for key, share in shared.iteritems():
            if share.path is not None:
                share.path = share.path.replace(old_root_dir, self._root_dir)
            shared[key] = share
        # move the My Files contents, taking care of dir/files with the same
        # name in the new root
        if os.path.exists(old_root_dir):
            self.log.debug('moving My Files contents to the root')
            # make My Files rw
            os.chmod(old_root_dir, 0775)
            path_join = os.path.join
            for relpath in os.listdir(old_root_dir):
                old_path = path_join(old_root_dir, relpath)
                new_path = path_join(self._root_dir, relpath)
                if os.path.exists(new_path):
                    shutil.move(new_path, new_path+'.u1conflict')
                if relpath == 'Shared With Me':
                    # remove the Shared with Me symlink inside My Files!
                    self.log.debug('removing shares symlink from old root')
                    os.remove(old_path)
                else:
                    self.log.debug('moving %r to %r', old_path, new_path)
                    move(old_path, new_path)
            self.log.debug('removing old root: %r', old_root_dir)
            os.rmdir(old_root_dir)

        # fix broken symlink (md_version 4)
        self._upgrade_metadata_4(md_version)

    def _upgrade_metadata_4(self, md_version):
        """Upgrade to version 5 (fix the broken symlink!)."""
        self.log.debug('upgrading from metadata 4 (broken symlink!)')
        if os.path.islink(self._shares_dir_link):
            target = os.readlink(self._shares_dir_link)
            if os.path.normpath(target) == self._shares_dir_link:
                # the symnlink points to itself
                self.log.debug('removing broken shares symlink: %r -> %r',
                               self._shares_dir_link, target)
                os.remove(self._shares_dir_link)
        self._upgrade_metadata_5(md_version)

    def _upgrade_metadata_5(self, md_version):
        """Upgrade to version 6 (plain dict storage)."""
        from ubuntuone.syncdaemon.volume_manager import VMFileShelf, LegacyShareFileShelf, UDF
        self.log.debug('upgrading from metadata 5')
        bkp_dir = os.path.join(os.path.dirname(self._data_dir), '5.bkp')
        new_md_dir = os.path.join(os.path.dirname(self._data_dir), 'md_6.new')
        new_shares_md_dir = os.path.join(new_md_dir, 'shares')
        new_shared_md_dir = os.path.join(new_md_dir, 'shared')
        new_udfs_md_dir = os.path.join(new_md_dir, 'udfs')
        try:
            # upgrade shares
            old_shares = LegacyShareFileShelf(self._shares_md_dir)
            shares = VMFileShelf(new_shares_md_dir)
            for key, share in old_shares.iteritems():
                shares[key] = self._upgrade_share_to_volume(share)
            # upgrade shared folders
            old_shared = LegacyShareFileShelf(self._shared_md_dir)
            shared = VMFileShelf(new_shared_md_dir)
            for key, share in old_shared.iteritems():
                shared[key] = self._upgrade_share_to_volume(share, shared=True)
            # upgrade the udfs
            old_udfs = LegacyShareFileShelf(self._udfs_md_dir)
            udfs = VMFileShelf(new_udfs_md_dir)
            for key, udf in old_udfs.iteritems():
                udfs[key] = UDF(udf.id, udf.node_id, udf.suggested_path,
                                udf.path, udf.subscribed)
            # move md dir to bkp
            os.rename(self._data_dir, bkp_dir)
            # move new to md dir
            os.rename(new_md_dir, self._data_dir)
            self._upgrade_metadata_6(md_version)
        except Exception:
            # something bad happend, remove partially upgraded metadata
            shutil.rmtree(new_md_dir)
            raise

    def _upgrade_share_to_volume(self, share, shared=False):
        """Upgrade from _Share to new Volume hierarchy."""
        from ubuntuone.syncdaemon.volume_manager import VMFileShelf, Root, Share, Shared
        def upgrade_share_dict(share):
            """Upgrade share __dict__ to be compatible with the
            new Share.__init__.

            """
            if 'subtree' in share.__dict__:
                share.node_id = share.__dict__.pop('subtree')
            if 'id' in share.__dict__:
                share.volume_id = share.__dict__.pop('id')
            if 'free_bytes' in share.__dict__:
                share.free_bytes = share.__dict__.pop('free_bytes')
            else:
                share.free_bytes = None
            return share

        if isinstance(share, dict):
            # oops, we have mixed metadata. fix it!
            clazz = VMFileShelf.classes[share[VMFileShelf.TYPE]]
            share_dict = share.copy()
            del share_dict[VMFileShelf.TYPE]
            return clazz(**share_dict)
        elif share.path == self._root_dir or share.id == '':
            # handle the root special case
            return Root(volume_id=request.ROOT,
                        node_id=share.subtree, path=share.path)
        else:
            share = upgrade_share_dict(share)
            if shared:
                return Shared(**share.__dict__)
            else:
                return Share(**share.__dict__)

    def _upgrade_metadata_6(self, md_version):
        """Upgrade to version 7, tritcask!."""
        from ubuntuone.syncdaemon.volume_manager import (
            VMFileShelf, VMTritcaskShelf,
            SHARE_ROW_TYPE, SHARED_ROW_TYPE, UDF_ROW_TYPE
        )
        self.log.debug('upgrading from metadata 6')
        old_shares = VMFileShelf(self._shares_md_dir)
        old_shared = VMFileShelf(self._shared_md_dir)
        old_udfs = VMFileShelf(self._udfs_md_dir)
        shares = VMTritcaskShelf(SHARE_ROW_TYPE, self.db)
        shared = VMTritcaskShelf(SHARED_ROW_TYPE, self.db)
        udfs = VMTritcaskShelf(UDF_ROW_TYPE, self.db)
        for share_id, share in old_shares.iteritems():
            shares[share_id] = share
        for share_id, share in old_shared.iteritems():
            shared[share_id] = share
        for udf_id, udf in old_udfs.iteritems():
            udfs[udf_id] = udf
        # update the metadata version
        self.update_metadata_version()
        # now delete the old metadata
        shutil.rmtree(self._shares_md_dir)
        shutil.rmtree(self._shared_md_dir)
        shutil.rmtree(self._udfs_md_dir)

