# -*- coding: utf-8 -*-
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Test the windows keyring implementation."""

from json import dumps

from twisted.internet import defer
from twisted.trial.unittest import TestCase

from ubuntu_sso.keyring import windows


class FakePyKeyring(object):

    """Fake Keyring"""

    def __init__(self):
        """Initialize object."""
        self.data = {}

    def set_password(self, *args):
        """Fake set_password for keyring."""
        self.data['set_password'] = args
        self.data['app_name'] = args[0][0]
        return self

    def get_password(self, app_name, username):
        """Fake get_password."""
        self.data['get_password'] = True
        password = dict(password='password', app_name=app_name)
        return dumps(password)

    def delete_password(self, *args):
        """Fake delete_password."""
        self.data['delete_password'] = True


class FakeDefer(object):

    """Fake Deferred."""

    def fake_defer_to_thread(self, func, *args):
        """Fake deferToThread."""
        return func(args)


class TestWindowsKeyring(TestCase):
    """Test the windows keyring implementation."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup  tests."""
        yield super(TestWindowsKeyring, self).setUp()
        self.app_name = 'app_name'
        self.fake_keyring = FakePyKeyring()
        self.keyring = windows.Keyring(self.fake_keyring)
        self.fake_defer = FakeDefer()
        self.patch(windows, "deferToThread",
            self.fake_defer.fake_defer_to_thread)

    # pylint: disable=E1103
    def test_set_credentials(self):
        """Test setting the credentials."""
        self.keyring = windows.Keyring(self.fake_keyring)
        d = self.keyring.set_credentials(self.app_name, 'password')
        self.assertEqual(d.data['set_password'],
            (('app_name', 'ubuntu_sso', '"password"'), ))

    def test_get_credentials(self):
        """Test getting the credentials."""
        d = self.keyring.get_credentials(self.app_name)
        self.assertTrue('password' in d)
        self.assertEqual(d['password'], 'password')
        self.assertTrue('app_name' in d)
        self.assertEqual(d['app_name'], ['app_name'])
        self.assertTrue(self.fake_keyring.data.get('get_password', False))

    def test_get_credentials_not_present(self):
        """Test getting creds that are not present."""
        d = self.keyring.get_credentials('app_name_not_there')
        self.assertTrue('password' in d)
        self.assertEqual(d['password'], 'password')
        self.assertTrue('app_name' in d)
        self.assertNotEqual(d['app_name'], [self.app_name])
        self.assertTrue(self.fake_keyring.data.get('get_password', False))

    def test_delete_credentials(self):
        """Test deleting the credentials."""
        result = self.keyring.delete_credentials(self.app_name)
        self.assertTrue(result is None)
        self.assertTrue(self.fake_keyring.data.get('delete_password', False))
    # pylint: enable=E1103
