# -*- coding: utf-8 -*-

# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2010, 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Utility modules that may find use outside ubuntu_sso."""

import cgi
import time
import urllib2

from oauth import oauth
from urlparse import urlparse
from twisted.web import http

from ubuntu_sso.logger import setup_logging
logger = setup_logging("ubuntu_sso.utils")


class RequestHead(urllib2.Request):
    """A request with the method set to HEAD."""

    _request_method = "HEAD"

    def get_method(self):
        """Return the desired method."""
        return self._request_method


class SyncTimestampChecker(object):
    """A timestamp that's regularly checked with a server."""

    CHECKING_INTERVAL = 60 * 60  # in seconds
    ERROR_INTERVAL = 30  # in seconds
    SERVER_URL = "http://one.ubuntu.com/api/time"

    def __init__(self):
        """Initialize this instance."""
        self.next_check = time.time()
        self.skew = 0

    def get_server_time(self):
        """Get the time at the server."""
        headers = {"Cache-Control": "no-cache"}
        request = RequestHead(self.SERVER_URL, headers=headers)
        response = urllib2.urlopen(request)
        date_string = response.info()["Date"]
        timestamp = http.stringToDatetime(date_string)
        return timestamp

    def get_faithful_time(self):
        """Get an accurate timestamp."""
        local_time = time.time()
        if local_time >= self.next_check:
            try:
                server_time = self.get_server_time()
                self.next_check = local_time + self.CHECKING_INTERVAL
                self.skew = server_time - local_time
                logger.debug("Calculated server-local time skew: %r",
                             self.skew)
            #pylint: disable=W0703
            except Exception, server_error:
                logger.debug("Error while verifying the server time skew: %r",
                             server_error)
                self.next_check = local_time + self.ERROR_INTERVAL
        logger.debug("Using corrected timestamp: %r",
                     http.datetimeToString(local_time + self.skew))
        return int(local_time + self.skew)


# pylint: disable=C0103
timestamp_checker = SyncTimestampChecker()
# pylint: enable=C0103


def oauth_headers(url, credentials, http_method='GET'):
    """Sign 'url' using 'credentials'.

    * 'url' must be a valid unicode url.
    * 'credentials' must be a valid OAuth token.

    Return oauth headers that can be pass to any Request like object.

    """
    assert isinstance(url, unicode)
    url = url.encode('utf-8')
    _, _, _, _, query, _ = urlparse(url)
    parameters = dict(cgi.parse_qsl(query))
    parameters["oauth_timestamp"] = timestamp_checker.get_faithful_time()

    consumer = oauth.OAuthConsumer(credentials['consumer_key'],
                                   credentials['consumer_secret'])
    token = oauth.OAuthToken(credentials['token'],
                             credentials['token_secret'])
    kwargs = dict(oauth_consumer=consumer, token=token,
                  http_method=http_method, http_url=url,
                  parameters=parameters)
    get_request = oauth.OAuthRequest.from_consumer_and_token
    oauth_req = get_request(**kwargs)
    hmac_sha1 = oauth.OAuthSignatureMethod_HMAC_SHA1()
    oauth_req.sign_request(hmac_sha1, consumer, token)
    headers = oauth_req.to_header()

    return headers
