# -*- coding: utf-8 -*-
# Author: Diego Sarmentero <diego.sarmentero@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Test the EnhancedLineEdit."""

import collections

from twisted.trial.unittest import TestCase

from PyQt4 import QtGui
from PyQt4 import QtCore

from ubuntu_sso.qt.gui import EnhancedLineEdit


class EnhancedLineEditTestCase(TestCase):
    """Tests EnhancedLineEdit for valid and invalid states."""

    def get_pixmap_data(self, pixmap):
        """Get the raw data of a QPixmap."""
        byte_array = QtCore.QByteArray()
        array_buffer = QtCore.QBuffer(byte_array)
        pixmap.save(array_buffer, "PNG")
        return byte_array

    # pylint: disable=C0103
    def assertEqualPixmaps(self, pixmap1, pixmap2):
        """Compare two Qt pixmaps."""
        d1 = self.get_pixmap_data(pixmap1)
        d2 = self.get_pixmap_data(pixmap2)
        self.assertEqual(d1, d2)
    # pylint: enable=C0103

    def test_enhanced_line_edit_init(self):
        """Tests the initial state of the enchanced line edit."""
        line_edit = QtGui.QLineEdit()
        line_edit.show()
        enhanced = EnhancedLineEdit(line_edit)
        self.assertIsInstance(enhanced.valid_cb, collections.Callable)
        self.assertTrue(enhanced.clear_label.property("lineEditWarning"))
        self.assertFalse(enhanced.clear_label.isVisible())
        self.assertEqual(enhanced.clear_label.cursor().shape(),
            QtCore.Qt.ArrowCursor)
        self.assertFalse(enhanced.clear_label.isVisible())
        self.assertEqual(enhanced.clear_label.pixmap(), None)

    def test_enhanced_line_edit_init_with_icon(self):
        """Tests the initial state of the enchanced line edit."""
        line_edit = QtGui.QLineEdit()
        line_edit.show()
        enhanced = EnhancedLineEdit(line_edit, warning_sign=True)
        self.assertIsInstance(enhanced.valid_cb, collections.Callable)
        edit_property = enhanced.clear_label.property("lineEditWarning")
        self.assertTrue(edit_property)
        self.assertFalse(enhanced.clear_label.isVisible())
        self.assertEqual(enhanced.clear_label.cursor().shape(),
            QtCore.Qt.ArrowCursor)
        self.assertFalse(enhanced.clear_label.isVisible())
        icon = QtGui.QApplication.style().standardIcon(
            QtGui.QStyle.SP_MessageBoxWarning)
        self.assertEqualPixmaps(enhanced.clear_label.pixmap(),
            icon.pixmap(16, 16))

    def test_enhanced_line_edit_function_show_true(self):
        """Tests the EnchancedLineEdit show method with validation = True."""
        line_edit = QtGui.QLineEdit()
        line_edit.show()
        func = lambda x: False
        enhanced = EnhancedLineEdit(line_edit, valid_cb=func,
                                    warning_sign=True)
        line_edit.setText("testing")
        self.assertTrue(enhanced.clear_label.isVisible())

    def test_enhanced_line_edit_function_show_false(self):
        """Tests the EnchancedLineEdit show method with validation = False."""
        line_edit = QtGui.QLineEdit()
        line_edit.show()
        func = lambda x: True
        enhanced = EnhancedLineEdit(line_edit, valid_cb=func)
        line_edit.setText("testing")
        self.assertFalse(enhanced.clear_label.isVisible())
