# -*- coding: utf-8 -*-
#
# test_networkstate - tests for ubuntu_sso.networkstate
#
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the network state detection code."""

from ubuntu_sso.networkstate import (NetworkManagerState,
                                     ONLINE, OFFLINE, UNKNOWN)
from ubuntu_sso.networkstate.linux import  (DBUS_UNKNOWN_SERVICE,
                                            NM_STATE_DISCONNECTED,
                                            NM_STATE_DISCONNECTED_OLD,
                                            NM_STATE_CONNECTING,
                                            NM_STATE_CONNECTING_OLD,
                                            NM_STATE_CONNECTED_OLD,
                                            NM_STATE_CONNECTED_LOCAL,
                                            NM_STATE_CONNECTED_SITE,
                                            NM_STATE_CONNECTED_GLOBAL)

from mocker import ARGS, KWARGS, ANY, MockerTestCase


class TestException(Exception):
    """An exception to test error conditions."""
    def get_dbus_name(self):
        """A fake dbus name for this exception."""
        return "Test Exception Message"


class TestNmNotAvailableException(Exception):
    """An exception to test unavailability conditions."""
    def get_dbus_name(self):
        """The real name of the dbus error when NM is not running."""
        return DBUS_UNKNOWN_SERVICE


class NetworkManagerBaseTestCase(MockerTestCase):
    """Base test case for NM state tests."""

    def setUp(self):
        """Setup the mocker dbus object tree."""
        super(NetworkManagerBaseTestCase, self).setUp()

        self.dbusmock = self.mocker.mock()
        self.dbusmock.SystemBus()
        sysbusmock = self.mocker.mock()
        self.mocker.result(sysbusmock)

        sysbusmock.get_object(ARGS, KWARGS)

    def connect_proxy(self, exc=None):
        """Get a proxy mock object, and allow failing with specified exc."""
        proxymock = self.mocker.mock()
        self.mocker.result(proxymock)

        self.dbusmock.Interface(proxymock, ANY)
        ifmock = self.mocker.mock()
        self.mocker.result(ifmock)

        ifmock.connect_to_signal(ARGS, KWARGS)
        signalmock = self.mocker.mock()
        self.mocker.result(signalmock)

        proxymock.Get(ARGS, KWARGS)
        if exc is not None:
            self.expect(self.dbusmock.exceptions.DBusException)
            self.mocker.result(exc)
        signalmock.remove()
        self.mocker.replay()

    # Invalid name "assert[A-Z].*"
    # pylint: disable=C0103
    def assertOnline(self, state):
        """Check that the state given is ONLINE."""
        self.assertEquals(state, ONLINE)

    def assertOffline(self, state):
        """Check that the state given is OFFLINE."""
        self.assertEquals(state, OFFLINE)

    def assertUnknown(self, state):
        """Check that the state was UNKNOWN."""
        self.assertEquals(state, UNKNOWN)
    # pylint: enable=C0103

    def get_nms(self, callback):
        """Get the NetworkManagerState object."""
        nms = NetworkManagerState(callback, self.dbusmock)
        nms.find_online_state()
        return nms

    def check_nm_error(self, callback, error):
        """Check that the error handling is correct."""
        nms = self.get_nms(callback)
        nms.got_error(error)

    def check_nm_state(self, callback, state):
        """Check the state handling is correct."""
        nms = self.get_nms(callback)
        nms.got_state(state)

    def check_nm_state_change(self, callback, fmstate, tostate):
        """Check the state change handling is correct."""
        nms = self.get_nms(callback)
        nms.got_state(fmstate)
        nms.state_changed(tostate)


class NetworkManagerStateTestCase(NetworkManagerBaseTestCase):
    """Test NetworkManager state retrieval code."""

    def setUp(self):
        """Setup the mocker dbus object tree."""
        super(NetworkManagerStateTestCase, self).setUp()
        self.connect_proxy()

    def test_nm_online_old(self):
        """Check the connected, old status, case."""
        self.check_nm_state(self.assertOnline, NM_STATE_CONNECTED_OLD)

    def test_nm_offline_local(self):
        """Check the connected, local status, case."""
        self.check_nm_state(self.assertOffline, NM_STATE_CONNECTED_LOCAL)

    def test_nm_offline_site(self):
        """Check the connected, site status, case."""
        self.check_nm_state(self.assertOffline, NM_STATE_CONNECTED_SITE)

    def test_nm_online_global(self):
        """Check the connected, global status, case."""
        self.check_nm_state(self.assertOnline, NM_STATE_CONNECTED_GLOBAL)

    def test_nm_offline_old(self):
        """Check the disconnected, old status, case."""
        self.check_nm_state(self.assertOffline, NM_STATE_DISCONNECTED_OLD)

    def test_nm_offline(self):
        """Check the disconnected case."""
        self.check_nm_state(self.assertOffline, NM_STATE_DISCONNECTED)

    def test_nm_connecting_then_online_old(self):
        """Check the waiting for connection, old status, case."""
        self.check_nm_state_change(self.assertOnline,
                                   NM_STATE_CONNECTING_OLD,
                                   NM_STATE_CONNECTED_OLD)

    def test_nm_connecting_then_online(self):
        """Check the waiting for connection case."""
        self.check_nm_state_change(self.assertOnline,
                                   NM_STATE_CONNECTING,
                                   NM_STATE_CONNECTED_GLOBAL)

    def test_nm_connecting_then_offline_old(self):
        """Check the waiting but fail, old status, case."""
        self.check_nm_state_change(self.assertOffline,
                                   NM_STATE_CONNECTING_OLD,
                                   NM_STATE_DISCONNECTED_OLD)

    def test_nm_connecting_then_offline(self):
        """Check the waiting but fail case."""
        self.check_nm_state_change(self.assertOffline,
                                   NM_STATE_CONNECTING, NM_STATE_DISCONNECTED)


class NetworkManagerStateErrorsTestCase(NetworkManagerBaseTestCase):
    """Test NetworkManager state retrieval code."""

    def mock_except_while_getting_proxy(self, exc):
        """Simulate an exception while getting the DBus proxy object."""
        self.mocker.throw(exc)
        self.expect(self.dbusmock.exceptions.DBusException)
        self.mocker.result(exc)
        self.mocker.replay()

    def test_nm_not_running(self):
        """Check the case when NM is not running."""
        self.connect_proxy(TestNmNotAvailableException)
        self.check_nm_error(self.assertUnknown, TestNmNotAvailableException())

    def test_dbus_problem(self):
        """Check the case when DBus throws some other exception."""
        self.mock_except_while_getting_proxy(TestException)
        self.get_nms(self.assertUnknown)
