# -*- coding: utf-8 -*-
#
# test_main - tests for ubuntu_sso.main
#
# Author: Natalia Bidart <natalia.bidart@canonical.com>
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2009-2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the main SSO client code."""

import logging
import os

from mocker import Mocker, MockerTestCase, ARGS, KWARGS, MATCH
from twisted.internet import defer
from twisted.internet.defer import Deferred, inlineCallbacks
from ubuntuone.devtools.handlers import MementoHandler

import ubuntu_sso.keyring
import ubuntu_sso.main
import ubuntu_sso.main.linux

from ubuntu_sso import DBUS_CREDENTIALS_IFACE
from ubuntu_sso.keyring import U1_APP_NAME
from ubuntu_sso.main import (U1_PING_URL, except_to_errdict,
    CredentialsManagement, SSOCredentials, SSOLogin, TIMEOUT_INTERVAL)
from ubuntu_sso.main.linux import blocking
from ubuntu_sso.main import (HELP_TEXT_KEY, PING_URL_KEY,
    TC_URL_KEY, UI_CLASS_KEY, UI_MODULE_KEY, WINDOW_ID_KEY,
    SUCCESS_CB_KEY, ERROR_CB_KEY, DENIAL_CB_KEY)
from ubuntu_sso.tests import (APP_NAME, TC_URL, HELP_TEXT, CAPTCHA_ID,
    CAPTCHA_SOLUTION, EMAIL, EMAIL_TOKEN, NAME, PASSWORD, PING_URL, TOKEN,
    TOKEN_NAME, WINDOW_ID, TestCase)


# Access to a protected member 'yyy' of a client class
# pylint: disable=W0212


class BlockingSampleException(Exception):
    """The exception that will be thrown by the fake blocking."""


def fake_ok_blocking(f, app, cb, eb):
    """A fake blocking function that succeeds."""
    cb(app, f())


def fake_err_blocking(f, app, cb, eb):
    """A fake blocking function that fails."""
    try:
        f()
    except Exception, e:  # pylint: disable=W0703
        eb(app, except_to_errdict(e))
    else:
        eb(app, except_to_errdict(BlockingSampleException()))


class SsoDbusTestCase(TestCase):
    """Test the SSOLogin DBus interface."""

    timeout = 2

    @defer.inlineCallbacks
    def setUp(self):
        """Create the mocking bus."""
        yield super(SsoDbusTestCase, self).setUp()
        self.mocker = Mocker()
        self.mockbusname = self.mocker.mock()
        mockbus = self.mocker.mock()
        self.mockbusname.get_bus()
        self.mocker.result(mockbus)
        mockbus._register_object_path(ARGS)
        self.mockprocessorclass = None

        def ksc(keyring, k, val):
            """Assert over token and app_name."""
            self.assertEqual(k, APP_NAME)
            self.assertEqual(val, TOKEN)
            self.keyring_was_set = True
            self.keyring_values = k, val
            return defer.succeed(None)

        self.patch(ubuntu_sso.main.Keyring, "set_credentials", ksc)
        self.keyring_was_set = False
        self.keyring_values = None

    @defer.inlineCallbacks
    def tearDown(self):
        """Verify the mocking bus and shut it down."""
        self.mocker.verify()
        self.mocker.restore()
        yield super(SsoDbusTestCase, self).tearDown()

    def test_creation(self):
        """Test that the object creation is successful."""
        self.mocker.replay()
        SSOLogin(self.mockbusname)

    def create_mock_processor(self):
        """Create a mock processor from a dummy processor class."""
        self.mockprocessorclass = self.mocker.mock()
        mockprocessor = self.mocker.mock()
        self.mockprocessorclass(ARGS, KWARGS)
        self.mocker.result(mockprocessor)
        return mockprocessor

    def test_generate_captcha(self):
        """Test that the captcha method works ok."""
        d = Deferred()
        filename = "sample filename"
        expected_result = "expected result"
        self.create_mock_processor().generate_captcha(filename)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main.linux, "blocking", fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, expected_result)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "CaptchaGenerated", verify)
        self.patch(client, "CaptchaGenerationError", d.errback)
        client.generate_captcha(APP_NAME, filename)
        return d

    def test_generate_captcha_error(self):
        """Test that the captcha method fails as expected."""
        d = Deferred()
        filename = "sample filename"
        expected_result = "expected result"
        self.create_mock_processor().generate_captcha(filename)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "thread_execute", fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "CaptchaGenerated", d.errback)
        self.patch(client, "CaptchaGenerationError", verify)
        client.generate_captcha(APP_NAME, filename)
        return d

    def test_register_user(self):
        """Test that the register_user method works ok."""
        d = Deferred()
        expected_result = "expected result"
        self.create_mock_processor().register_user(EMAIL, PASSWORD, NAME,
            CAPTCHA_ID, CAPTCHA_SOLUTION)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main.linux, "blocking", fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, expected_result)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "UserRegistered", verify)
        self.patch(client, "UserRegistrationError", d.errback)
        client.register_user(APP_NAME, EMAIL, PASSWORD, NAME, CAPTCHA_ID,
                             CAPTCHA_SOLUTION)
        return d

    def test_register_user_error(self):
        """Test that the register_user method fails as expected."""
        d = Deferred()
        expected_result = "expected result"
        self.create_mock_processor().register_user(EMAIL, PASSWORD, NAME,
            CAPTCHA_ID, CAPTCHA_SOLUTION)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "thread_execute", fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "UserRegistered", d.errback)
        self.patch(client, "UserRegistrationError", verify)
        client.register_user(APP_NAME, EMAIL, PASSWORD, NAME, CAPTCHA_ID,
                             CAPTCHA_SOLUTION)
        return d

    def test_login(self):
        """Test that the login method works ok."""
        d = Deferred()
        processor = self.create_mock_processor()
        processor.login(EMAIL, PASSWORD, TOKEN_NAME)
        self.mocker.result(TOKEN)
        processor.is_validated(TOKEN)
        self.mocker.result(True)
        self.patch(ubuntu_sso.main.linux, "blocking", fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            self.assertTrue(self.keyring_was_set, "The keyring should be set")
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "LoggedIn", verify)
        self.patch(client, "LoginError", d.errback)
        self.patch(client, "UserNotValidated", d.errback)
        client.login(APP_NAME, EMAIL, PASSWORD)
        return d

    def test_login_user_not_validated(self):
        """Test that the login sends EmailNotValidated signal."""
        d = Deferred()
        processor = self.create_mock_processor()
        processor.login(EMAIL, PASSWORD, TOKEN_NAME)
        self.mocker.result(TOKEN)
        processor.is_validated(TOKEN)
        self.mocker.result(False)
        self.patch(ubuntu_sso.main.linux, "blocking", fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, email):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(email, EMAIL)
            self.assertFalse(self.keyring_was_set, "Keyring should not be set")
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "LoggedIn", d.errback)
        self.patch(client, "LoginError", d.errback)
        self.patch(client, "UserNotValidated", verify)
        client.login(APP_NAME, EMAIL, PASSWORD)
        return d

    def test_login_error_get_token_name(self):
        """The login method fails as expected when get_token_name fails."""
        d = Deferred()
        self.create_mock_processor()
        self.patch(ubuntu_sso.main.linux, "blocking", fake_err_blocking)

        def fake_gtn(*args):
            """A fake get_token_name that fails."""
            raise BlockingSampleException()

        self.patch(ubuntu_sso.main, "get_token_name", fake_gtn)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertFalse(self.keyring_was_set, "Keyring should not be set")
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "LoggedIn", d.errback)
        self.patch(client, "LoginError", verify)
        self.patch(client, "UserNotValidated", d.errback)
        client.login(APP_NAME, EMAIL, PASSWORD)
        return d

    def test_login_error_set_credentials(self):
        """The login method fails as expected when set_credentials fails."""
        d = Deferred()
        processor = self.create_mock_processor()
        processor.login(EMAIL, PASSWORD, TOKEN_NAME)
        self.mocker.result(TOKEN)
        processor.is_validated(TOKEN)
        self.mocker.result(True)

        self.patch(ubuntu_sso.main.linux, "blocking", fake_ok_blocking)

        def fake_set_creds(*args):
            """A fake Keyring.set_credentials that fails."""
            return defer.fail(BlockingSampleException())

        self.patch(ubuntu_sso.main.Keyring, "set_credentials", fake_set_creds)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertFalse(self.keyring_was_set, "Keyring should not be set")
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        fail = lambda app, res: d.errback((app, res))
        self.patch(client, "LoggedIn", fail)
        self.patch(client, "LoginError", verify)
        self.patch(client, "UserNotValidated", fail)
        client.login(APP_NAME, EMAIL, PASSWORD)
        return d

    def test_validate_email(self):
        """Test that the validate_email method works ok."""
        d = Deferred()
        self.create_mock_processor().validate_email(EMAIL, PASSWORD,
                                                    EMAIL_TOKEN, TOKEN_NAME)
        self.mocker.result(TOKEN)
        self.patch(ubuntu_sso.main.linux, "blocking", fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            self.assertTrue(self.keyring_was_set, "The keyring should be set")
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "EmailValidated", verify)
        self.patch(client, "EmailValidationError", d.errback)
        client.validate_email(APP_NAME, EMAIL, PASSWORD, EMAIL_TOKEN)
        return d

    def test_validate_email_error(self):
        """Test that the validate_email method fails as expected."""
        d = Deferred()
        self.create_mock_processor()
        self.patch(ubuntu_sso.main.linux, "blocking", fake_err_blocking)

        def fake_gtn(*args):
            """A fake get_token_name that fails."""
            raise BlockingSampleException()

        self.patch(ubuntu_sso.main, "get_token_name", fake_gtn)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertFalse(self.keyring_was_set, "Keyring should not be set")
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "EmailValidated", d.errback)
        self.patch(client, "EmailValidationError", verify)
        client.validate_email(APP_NAME, EMAIL, PASSWORD, EMAIL_TOKEN)
        return d

    def test_request_password_reset_token(self):
        """Test that the request_password_reset_token method works ok."""
        d = Deferred()
        processor = self.create_mock_processor()
        processor.request_password_reset_token(EMAIL)
        self.patch(ubuntu_sso.main, "thread_execute", fake_ok_blocking)
        self.mocker.result(EMAIL)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordResetTokenSent", verify)
        self.patch(client, "PasswordResetError", d.errback)
        client.request_password_reset_token(APP_NAME, EMAIL)
        return d

    def test_request_password_reset_token_error(self):
        """Test the request_password_reset_token method fails as expected."""
        d = Deferred()

        self.create_mock_processor().request_password_reset_token(EMAIL)
        self.mocker.result(EMAIL)
        self.patch(ubuntu_sso.main, "thread_execute", fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordResetTokenSent", d.errback)
        self.patch(client, "PasswordResetError", verify)
        client.request_password_reset_token(APP_NAME, EMAIL)
        return d

    def test_set_new_password(self):
        """Test that the set_new_password method works ok."""
        d = Deferred()
        self.create_mock_processor().set_new_password(EMAIL, EMAIL_TOKEN,
                                                      PASSWORD)
        self.mocker.result(EMAIL)
        self.patch(ubuntu_sso.main.linux, "blocking", fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordChanged", verify)
        self.patch(client, "PasswordChangeError", d.errback)
        client.set_new_password(APP_NAME, EMAIL, EMAIL_TOKEN, PASSWORD)
        return d

    def test_set_new_password_error(self):
        """Test that the set_new_password method fails as expected."""
        d = Deferred()
        expected_result = "expected result"

        self.create_mock_processor().set_new_password(EMAIL, EMAIL_TOKEN,
                                                      PASSWORD)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "thread_execute", fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordChanged", d.errback)
        self.patch(client, "PasswordChangeError", verify)
        client.set_new_password(APP_NAME, EMAIL, EMAIL_TOKEN, PASSWORD)
        return d


class BlockingFunctionTestCase(TestCase):
    """Tests for the "blocking" function."""

    timeout = 5

    def test_blocking(self):
        """Test the normal behaviour."""
        d = Deferred()
        expected_result = "expected result"

        def f():
            """No failure."""
            return expected_result

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, expected_result)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        blocking(f, APP_NAME, verify, d.errback)
        return d

    def test_blocking_error(self):
        """Test the behaviour when an Exception is raised."""
        d = Deferred()
        expected_error_message = "expected error message"

        def f():
            """Failure."""
            raise BlockingSampleException(expected_error_message)

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(errdict["message"], expected_error_message)
            d.callback("Ok")

        blocking(f, APP_NAME, d.errback, verify)
        return d


class TestExceptToErrdictException(Exception):
    """A dummy exception for the following testcase."""


class ExceptToErrdictTestCase(TestCase):
    """Tests for the except_to_errdict function."""

    def test_first_arg_is_dict(self):
        """If the first arg is a dict, use it as the base dict."""
        sample_dict = {
            "errorcode1": "error message 1",
            "errorcode2": "error message 2",
            "errorcode3": "error message 3",
        }
        e = TestExceptToErrdictException(sample_dict)
        result = except_to_errdict(e)

        self.assertEqual(result["errtype"], e.__class__.__name__)
        for k in sample_dict.keys():
            self.assertIn(k, result)
            self.assertEqual(result[k], sample_dict[k])

    def test_first_arg_is_str(self):
        """If the first arg is a str, use it as the message."""
        sample_string = "a sample string"
        e = TestExceptToErrdictException(sample_string)
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)
        self.assertEqual(result["message"], sample_string)

    def test_first_arg_is_unicode(self):
        """If the first arg is a unicode, use it as the message."""
        sample_string = u"a sample string"
        e = TestExceptToErrdictException(sample_string)
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)
        self.assertEqual(result["message"], sample_string)

    def test_no_args_at_all(self):
        """If there are no args, use the class docstring."""
        e = TestExceptToErrdictException()
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)
        self.assertEqual(result["message"], e.__class__.__doc__)

    def test_some_other_thing_as_first_arg(self):
        """If first arg is not basestring nor dict, then repr all args."""
        sample_args = (None, u"unicode2\ufffd", "errorcode3")
        e = TestExceptToErrdictException(*sample_args)
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)


class RegisterSampleException(Exception):
    """A mock exception thrown just when testing."""


class ApplicationCredentialsTestCase(TestCase, MockerTestCase):
    """Tests for the ApplicationCredentials related DBus methods."""

    timeout = 5

    @inlineCallbacks
    def setUp(self):
        yield super(ApplicationCredentialsTestCase, self).setUp()

        self.client = SSOCredentials(self.mocker.mock())

        mock_class = self.mocker.replace("ubuntu_sso.credentials.Credentials")
        mock_class(app_name=APP_NAME)
        self.creds_obj = self.mocker.mock()
        self.mocker.result(self.creds_obj)

    @inlineCallbacks
    def test_find_credentials(self):
        """find_credentials immediately returns the token when found."""
        expected_creds = "expected creds"
        self.creds_obj.find_credentials()
        self.mocker.result(defer.succeed(expected_creds))
        self.mocker.replay()

        d = Deferred()
        self.client.find_credentials(APP_NAME, d.callback, d.errback)
        creds = yield d
        self.assertEqual(creds, expected_creds)

    @inlineCallbacks
    def test_credentials_not_found(self):
        """find_credentials immediately returns {} when no creds found."""
        expected_creds = {}
        self.creds_obj.find_credentials()
        self.mocker.result(defer.succeed(expected_creds))
        self.mocker.replay()

        d = Deferred()
        self.client.find_credentials(APP_NAME, d.callback, d.errback)
        creds = yield d
        self.assertEqual(creds, expected_creds)


class ApplicationCredentialsGUITestCase(TestCase, MockerTestCase):
    """Tests for the ApplicationCredentials register/login DBus method."""

    app_name = APP_NAME
    ping_url = None

    @inlineCallbacks
    def setUp(self):
        yield super(ApplicationCredentialsGUITestCase, self).setUp()
        self.client = SSOCredentials(self.mocker.mock())
        self.args = {PING_URL_KEY: self.ping_url,
                     TC_URL_KEY: TC_URL, HELP_TEXT_KEY: HELP_TEXT,
                     WINDOW_ID_KEY: WINDOW_ID,
                     SUCCESS_CB_KEY: self.client.CredentialsFound,
                     ERROR_CB_KEY: self.client._process_error,
                     DENIAL_CB_KEY: self.client.AuthorizationDenied}

    def test_login_or_register(self):
        """login_or_register is correct."""
        self.args[UI_MODULE_KEY] = MATCH(lambda x: isinstance(x, str))
        mock_class = self.mocker.replace("ubuntu_sso.credentials.Credentials")
        mock_class(app_name=self.app_name, **self.args)
        creds_obj = self.mocker.mock()
        self.mocker.result(creds_obj)

        creds_obj.register()
        self.mocker.replay()

        args = (self.app_name, TC_URL, HELP_TEXT, WINDOW_ID)
        self.client.login_or_register_to_get_credentials(*args)

    def test_login_only(self):
        """login_or_register is correct."""
        self.args[TC_URL_KEY] = None
        self.args[UI_MODULE_KEY] = MATCH(lambda x: isinstance(x, str))
        mock_class = self.mocker.replace("ubuntu_sso.credentials.Credentials")
        mock_class(app_name=self.app_name, **self.args)
        creds_obj = self.mocker.mock()
        self.mocker.result(creds_obj)

        creds_obj.login()
        self.mocker.replay()

        args = (self.app_name, HELP_TEXT, WINDOW_ID)
        self.client.login_to_get_credentials(*args)


class ApplicationCredentialsU1TestCase(ApplicationCredentialsGUITestCase):
    """Tests for the ApplicationCredentials register/login DBus method.

    Specifically for APP_NAME == U1_APP_NAME.

    """

    app_name = U1_APP_NAME
    ping_url = U1_PING_URL


class ApplicationCredentialsClearTokenTestCase(TestCase, MockerTestCase):
    """Tests for the ApplicationCredentials related DBus methods."""

    def test_clear_token(self):
        """Check that clear_token tries removing the correct token."""
        mock_class = self.mocker.replace("ubuntu_sso.credentials.Credentials")
        mock_class(app_name=APP_NAME)
        creds_obj = self.mocker.mock()
        self.mocker.result(creds_obj)

        creds_obj.clear_credentials()
        self.mocker.result(defer.succeed(None))
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        client.clear_token(APP_NAME)


class EnvironOverridesTestCase(TestCase):
    """Some URLs can be set from the environment for testing/QA purposes."""

    def test_override_ping_url(self):
        """The ping url can be set from the environ via USSOC_PING_URL."""
        fake_url = 'this is not really a URL'
        old_url = os.environ.get('USSOC_PING_URL')
        os.environ['USSOC_PING_URL'] = fake_url
        try:
            creds = SSOCredentials(None)
            self.assertEqual(creds.root.ping_url, fake_url)
        finally:
            if old_url:
                os.environ['USSOC_PING_URL'] = old_url
            else:
                del os.environ['USSOC_PING_URL']

    def test_no_override_ping_url(self):
        """If the environ is unset, the default ping url is used."""
        creds = SSOCredentials(None)
        self.assertEqual(creds.root.ping_url, U1_PING_URL)


class CredentialsManagementTestCase(TestCase):
    """Tests for the CredentialsManagement DBus interface."""

    timeout = 2
    base_args = {HELP_TEXT_KEY: HELP_TEXT, PING_URL_KEY: PING_URL,
                 TC_URL_KEY: TC_URL, WINDOW_ID_KEY: WINDOW_ID,
                 UI_CLASS_KEY: 'SuperUI', UI_MODULE_KEY: 'foo.bar.baz',
                }

    @defer.inlineCallbacks
    def setUp(self):
        yield super(CredentialsManagementTestCase, self).setUp()

        self.mocker = Mocker()
        self.client = CredentialsManagement(timeout_func=lambda *a: None,
                                            shutdown_func=lambda *a: None)
        self.args = {}
        self.cred_args = {}

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        ubuntu_sso.main.logger.addHandler(self.memento)

    @defer.inlineCallbacks
    def tearDown(self):
        """Verify the mocking stuff and shut it down."""
        self.mocker.verify()
        self.mocker.restore()
        yield super(CredentialsManagementTestCase, self).tearDown()

    def assert_dbus_method_correct(self, method):
        """Check that 'method' is a dbus method with proper signatures."""
        self.assertTrue(method._dbus_is_method)
        self.assertEqual(method._dbus_interface, DBUS_CREDENTIALS_IFACE)
        self.assertEqual(method._dbus_in_signature, 'sa{ss}')
        self.assertEqual(method._dbus_out_signature, '')

    def create_mock_backend(self):
        """Create a mock backend."""
        mock_class = self.mocker.replace("ubuntu_sso.credentials.Credentials")
        mock_class(APP_NAME, **self.cred_args)
        creds_obj = self.mocker.mock()
        self.mocker.result(creds_obj)

        return creds_obj

    def test_is_dbus_object(self):
        """CredentialsManagement is a Dbus object."""
        self.assertIsInstance(self.client,
                              ubuntu_sso.main.linux.dbus.service.Object)


class FakeCredentials(object):
    """A very dummy Credentials object."""

    def __init__(self, *a, **kw):
        self.find_credentials = lambda *a: defer.succeed(TOKEN)
        self.clear_credentials = lambda *a: defer.succeed(None)
        self.store_credentials = lambda *a: defer.succeed(None)
        self.login = self.register = lambda *a: None


class CredentialsManagementRefCountingTestCase(CredentialsManagementTestCase):
    """Tests for the CredentialsManagement ref counting."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(CredentialsManagementRefCountingTestCase, self).setUp()
        self.patch(ubuntu_sso.main, 'Credentials', FakeCredentials)

    def test_ref_counting(self):
        """Ref counting is in place."""
        self.assertEqual(self.client.root.ref_count, 0)

    def test_find_credentials(self):
        """Keep proper track of on going requests."""
        d = Deferred()

        def verify(*args):
            """Make the check."""
            self.assertEqual(self.client.root.ref_count, 1)
            d.callback(True)

        self.patch(self.client, 'CredentialsFound', verify)
        self.client.find_credentials(APP_NAME, self.args)

        return d

    def test_clear_credentials(self):
        """Keep proper track of on going requests."""
        d = Deferred()

        def verify(*args):
            """Make the check."""
            self.assertEqual(self.client.root.ref_count, 1)
            d.callback(True)

        self.patch(self.client, 'CredentialsCleared', verify)
        self.client.clear_credentials(APP_NAME, self.args)

        return d

    def test_store_credentials(self):
        """Keep proper track of on going requests."""
        d = Deferred()

        def verify(*args):
            """Make the check."""
            self.assertEqual(self.client.root.ref_count, 1)
            d.callback(True)

        self.patch(self.client, 'CredentialsStored', verify)
        self.client.store_credentials(APP_NAME, self.args)

        return d

    def test_register(self):
        """Keep proper track of on going requests."""
        self.client.register(APP_NAME, self.args)

        self.assertEqual(self.client.root.ref_count, 1)

    def test_login(self):
        """Keep proper track of on going requests."""
        self.client.login(APP_NAME, self.args)

        self.assertEqual(self.client.root.ref_count, 1)

    def test_several_requests(self):
        """Requests can be nested."""
        self.client.login(APP_NAME, self.args)
        self.client.register(APP_NAME, self.args)
        self.client.login(APP_NAME, self.args)
        self.client.register(APP_NAME, self.args)
        self.client.register(APP_NAME, self.args)

        self.assertEqual(self.client.root.ref_count, 5)

    def test_credentials_found(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root.ref_count = 3
        self.client.CredentialsFound(APP_NAME, TOKEN)

        self.assertEqual(self.client.root.ref_count, 2)

    def test_credentials_not_found(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root.ref_count = 3
        self.client.CredentialsNotFound(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 2)

    def test_credentials_cleared(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root.ref_count = 3
        self.client.CredentialsCleared(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 2)

    def test_credentials_stored(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root.ref_count = 3
        self.client.CredentialsStored(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 2)

    def test_credentials_error(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root.ref_count = 3
        self.client.CredentialsError(APP_NAME, {'error_type': 'test'})

        self.assertEqual(self.client.root.ref_count, 2)

    def test_authorization_denied(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root.ref_count = 3
        self.client.AuthorizationDenied(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 2)

    def test_credentials_found_when_ref_count_is_not_positive(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root._ref_count = -3
        self.client.CredentialsFound(APP_NAME, TOKEN)

        self.assertEqual(self.client.root.ref_count, 0)
        msg = 'Attempting to decrease ref_count to a negative value (-4).'
        self.assertTrue(self.memento.check_warning(msg))

    def test_credentials_not_found_when_ref_count_is_not_positive(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root._ref_count = -3
        self.client.CredentialsNotFound(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 0)
        msg = 'Attempting to decrease ref_count to a negative value (-4).'
        self.assertTrue(self.memento.check_warning(msg))

    def test_credentials_cleared_when_ref_count_is_not_positive(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root._ref_count = -3
        self.client.CredentialsCleared(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 0)
        msg = 'Attempting to decrease ref_count to a negative value (-4).'
        self.assertTrue(self.memento.check_warning(msg))

    def test_credentials_stored_when_ref_count_is_not_positive(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root._ref_count = -3
        self.client.CredentialsStored(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 0)
        msg = 'Attempting to decrease ref_count to a negative value (-4).'
        self.assertTrue(self.memento.check_warning(msg))

    def test_credentials_error_when_ref_count_is_not_positive(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root._ref_count = -3
        self.client.CredentialsError(APP_NAME, {'error_type': 'test'})

        self.assertEqual(self.client.root.ref_count, 0)
        msg = 'Attempting to decrease ref_count to a negative value (-4).'
        self.assertTrue(self.memento.check_warning(msg))

    def test_autorization_denied_when_ref_count_is_not_positive(self):
        """Ref counter is decreased when a signal is sent."""
        self.client.root._ref_count = -3
        self.client.AuthorizationDenied(APP_NAME)

        self.assertEqual(self.client.root.ref_count, 0)
        msg = 'Attempting to decrease ref_count to a negative value (-4).'
        self.assertTrue(self.memento.check_warning(msg))

    def test_on_zero_ref_count_shutdown(self):
        """When ref count reaches 0, queue shutdown op."""
        self.client.root.timeout_func = self._set_called
        self.client.login(APP_NAME, self.args)
        self.client.CredentialsFound(APP_NAME, TOKEN)

        self.assertEqual(self._called,
                         ((TIMEOUT_INTERVAL, self.client.root.shutdown), {}))

    def test_on_non_zero_ref_count_do_not_shutdown(self):
        """If ref count is not 0, do not queue shutdown op."""
        self.client.timeout_func = self._set_called
        self.client.login(APP_NAME, self.args)

        self.assertEqual(self._called, False)

    def test_on_non_zero_ref_count_after_zero_do_not_shutdown(self):
        """If the shutdown was queued, do not quit if counter is not zero."""

        def fake_timeout_func(interval, func):
            """Start a new request when the timer is started."""
            self.client.register(APP_NAME, self.args)
            assert self.client.root.ref_count > 0
            func()

        self.client.timeout_func = fake_timeout_func
        self.client.shutdown_func = self._set_called

        self.client.login(APP_NAME, self.args)
        self.client.CredentialsFound(APP_NAME, TOKEN)
        # counter reached 0, timeout_func was called

        self.assertEqual(self._called, False, 'shutdown_func was not called')

    def test_zero_ref_count_after_zero_do_shutdown(self):
        """If the shutdown was queued, do quit if counter is zero."""

        def fake_timeout_func(interval, func):
            """Start a new request when the timer is started."""
            assert self.client.root.ref_count == 0
            func()

        self.client.root.timeout_func = fake_timeout_func
        self.client.root.shutdown_func = self._set_called

        self.client.login(APP_NAME, self.args)
        self.client.CredentialsFound(APP_NAME, TOKEN)
        # counter reached 0, timeout_func was called

        self.assertEqual(self._called, ((), {}), 'shutdown_func was called')


class CredentialsManagementFindTestCase(CredentialsManagementTestCase):
    """Tests for the CredentialsManagement find method."""

    def test_find_credentials(self):
        """The credentials are asked and returned in signals."""
        self.create_mock_backend().find_credentials()
        self.mocker.result(defer.succeed(None))
        self.mocker.replay()

        self.client.find_credentials(APP_NAME, self.args)
        self.assert_dbus_method_correct(self.client.find_credentials)

    def test_find_credentials_does_not_block_when_found(self):
        """Calling find_credentials does not block but return thru signals.

        If the creds are found, CredentialsFound is emitted.

        """
        d = Deferred()

        def verify(app_name, creds):
            """The actual test."""
            try:
                self.assertEqual(app_name, APP_NAME)
                self.assertEqual(creds, TOKEN)
            except Exception, e:  # pylint: disable=W0703
                d.errback(e)
            else:
                d.callback(creds)

        self.patch(self.client, 'CredentialsFound', verify)
        self.patch(self.client, 'CredentialsNotFound', d.errback)

        self.create_mock_backend().find_credentials()
        self.mocker.result(defer.succeed(TOKEN))
        self.mocker.replay()

        self.client.find_credentials(APP_NAME, self.args)
        return d

    def test_find_credentials_does_not_block_when_not_found(self):
        """Calling find_credentials does not block but return thru signals.

        If the creds are not found, CredentialsNotFound is emitted.

        """
        d = Deferred()

        def verify(app_name):
            """The actual test."""
            try:
                self.assertEqual(app_name, APP_NAME)
            except Exception, e:  # pylint: disable=W0703
                d.errback(e)
            else:
                d.callback(app_name)

        self.patch(self.client, 'CredentialsFound',
                   lambda app, creds: d.errback(app))
        self.patch(self.client, 'CredentialsNotFound', verify)

        self.create_mock_backend().find_credentials()
        self.mocker.result(defer.succeed({}))
        self.mocker.replay()

        self.client.find_credentials(APP_NAME, self.args)
        return d

    def test_find_credentials_error(self):
        """If find_credentials fails, CredentialsError is sent."""
        d = Deferred()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        self.patch(self.client, 'CredentialsFound',
                   lambda app, creds: d.errback(app))
        self.patch(self.client, 'CredentialsNotFound', d.errback)
        self.patch(self.client, 'CredentialsError', verify)

        self.create_mock_backend().find_credentials()
        self.mocker.result(defer.fail(BlockingSampleException()))
        self.mocker.replay()

        self.client.find_credentials(APP_NAME, self.args)
        return d


class CredentialsManagementClearTestCase(CredentialsManagementTestCase):
    """Tests for the CredentialsManagement clear method."""

    def test_clear_credentials(self):
        """The credentials are removed."""
        self.create_mock_backend().clear_credentials()
        self.mocker.result(defer.succeed(APP_NAME))
        self.mocker.replay()

        self.client.clear_credentials(APP_NAME, self.args)
        self.assert_dbus_method_correct(self.client.clear_credentials)

    def test_clear_credentials_does_not_block(self):
        """Calling clear_credentials does not block but return thru signals."""
        d = Deferred()

        def verify(app_name):
            """The actual test."""
            try:
                self.assertEqual(app_name, APP_NAME)
            except Exception, e:  # pylint: disable=W0703
                d.errback(e)
            else:
                d.callback(app_name)

        self.patch(self.client, 'CredentialsCleared', verify)
        self.patch(self.client, 'CredentialsError',
                   lambda app, err: d.errback(app))

        self.create_mock_backend().clear_credentials()
        self.mocker.result(defer.succeed(APP_NAME))
        self.mocker.replay()

        self.client.clear_credentials(APP_NAME, self.args)
        return d

    def test_clear_credentials_error(self):
        """If clear_credentials fails, CredentialsError is sent."""
        d = Deferred()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        self.patch(self.client, 'CredentialsCleared', d.errback)
        self.patch(self.client, 'CredentialsError', verify)

        self.create_mock_backend().clear_credentials()
        self.mocker.result(defer.fail(BlockingSampleException()))
        self.mocker.replay()

        self.client.clear_credentials(APP_NAME, self.args)
        return d


class CredentialsManagementStoreTestCase(CredentialsManagementTestCase):
    """Tests for the CredentialsManagement store method."""

    def test_store_credentials(self):
        """The credentials are stored and the outcome is a signal."""
        self.create_mock_backend().store_credentials(TOKEN)
        self.mocker.result(defer.succeed(APP_NAME))
        self.mocker.replay()

        self.client.store_credentials(APP_NAME, TOKEN)
        self.assert_dbus_method_correct(self.client.store_credentials)

    def test_store_credentials_does_not_block(self):
        """Calling store_credentials does not block but return thru signals.

        If the creds are stored, CredentialsStored is emitted.

        """
        d = Deferred()

        def verify(app_name):
            """The actual test."""
            try:
                self.assertEqual(app_name, APP_NAME)
            except Exception, e:  # pylint: disable=W0703
                d.errback(e)
            else:
                d.callback(app_name)

        self.patch(self.client, 'CredentialsStored', verify)
        self.patch(self.client, 'CredentialsError',
                   lambda app, err: d.errback(app))

        self.create_mock_backend().store_credentials(TOKEN)
        self.mocker.result(defer.succeed(APP_NAME))
        self.mocker.replay()

        self.client.store_credentials(APP_NAME, TOKEN)
        return d

    def test_store_credentials_error(self):
        """If store_credentials fails, CredentialsError is sent."""
        d = Deferred()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        self.patch(self.client, 'CredentialsStored', d.errback)
        self.patch(self.client, 'CredentialsError', verify)

        self.create_mock_backend().store_credentials(TOKEN)
        self.mocker.result(defer.fail(BlockingSampleException()))
        self.mocker.replay()

        self.client.store_credentials(APP_NAME, TOKEN)
        return d


class CredentialsManagementOpsTestCase(CredentialsManagementTestCase):
    """Tests for the CredentialsManagement login/register methods."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(CredentialsManagementOpsTestCase, self).setUp()
        self.args = dict((k, str(v)) for k, v in self.base_args.iteritems())
        self.cred_args = self.base_args.copy()
        self.cred_args[SUCCESS_CB_KEY] = self.client.CredentialsFound
        self.cred_args[ERROR_CB_KEY] = self.client.CredentialsError
        self.cred_args[DENIAL_CB_KEY] = self.client.AuthorizationDenied

    def test_register(self):
        """The registration is correct."""
        self.create_mock_backend().register()
        self.mocker.replay()

        self.client.register(APP_NAME, self.args)
        self.assert_dbus_method_correct(self.client.register)

    def test_login(self):
        """The login is correct."""
        self.create_mock_backend().login()
        self.mocker.replay()

        self.client.login(APP_NAME, self.args)
        self.assert_dbus_method_correct(self.client.login)

    def test_login_email_password(self):
        """The login_email_password is correct."""
        self.create_mock_backend().login_email_password(email=EMAIL,
                                                        password=PASSWORD)
        self.mocker.replay()

        self.args['email'] = EMAIL
        self.args['password'] = PASSWORD
        self.client.login_email_password(APP_NAME, self.args)
        self.assert_dbus_method_correct(self.client.login_email_password)


class CredentialsManagementParamsTestCase(CredentialsManagementOpsTestCase):
    """Tests for the CredentialsManagement extra parameters handling."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(CredentialsManagementParamsTestCase, self).setUp()
        self.args['dummy'] = 'nothing useful'


class CredentialsManagementSignalsTestCase(TestCase):
    """Tests for the CredentialsManagement DBus signals."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set up."""
        yield super(CredentialsManagementSignalsTestCase, self).setUp()
        self.client = CredentialsManagement(timeout_func=lambda *a: None,
                                            shutdown_func=lambda *a: None)

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        ubuntu_sso.main.logger.addHandler(self.memento)

    def assert_dbus_signal_correct(self, signal, signature):
        """Check that 'signal' is a dbus signal with proper 'signature'."""
        self.assertTrue(signal._dbus_is_signal)
        self.assertEqual(signal._dbus_interface, DBUS_CREDENTIALS_IFACE)
        self.assertEqual(signal._dbus_signature, signature)

    def test_credentials_found(self):
        """The CredentialsFound signal."""
        self.client.CredentialsFound(APP_NAME, TOKEN)
        msgs = (self.client.__class__.__name__,
                self.client.CredentialsFound.__name__, APP_NAME)
        self.assertTrue(self.memento.check_info(*msgs))

        msg = 'credentials must not be logged (found %r in log).'
        for val in TOKEN.itervalues():
            self.assertFalse(self.memento.check_info(val), msg % val)

        self.assert_dbus_signal_correct(self.client.CredentialsFound, 'sa{ss}')

    def test_credentials_not_found(self):
        """The CredentialsNotFound signal."""
        self.client.CredentialsNotFound(APP_NAME)
        msgs = (self.client.__class__.__name__,
                self.client.CredentialsNotFound.__name__, APP_NAME)
        self.assertTrue(self.memento.check_info(*msgs))
        self.assert_dbus_signal_correct(self.client.CredentialsNotFound, 's')

    def test_credentials_cleared(self):
        """The CredentialsCleared signal."""
        self.client.CredentialsCleared(APP_NAME)
        msgs = (self.client.__class__.__name__,
                self.client.CredentialsCleared.__name__, APP_NAME)
        self.assertTrue(self.memento.check_info(*msgs))

        self.assert_dbus_signal_correct(self.client.CredentialsCleared, 's')

    def test_credentials_stored(self):
        """The CredentialsStored signal."""
        self.client.CredentialsStored(APP_NAME)
        msgs = (self.client.__class__.__name__,
                self.client.CredentialsStored.__name__, APP_NAME)
        self.assertTrue(self.memento.check_info(*msgs))

        self.assert_dbus_signal_correct(self.client.CredentialsStored, 's')

    def test_credentials_error(self):
        """The CredentialsError signal."""
        error = {'error_message': 'failed!', 'detailed error': 'yadda yadda'}
        self.client.CredentialsError(APP_NAME, error)
        msgs = (self.client.__class__.__name__,
                self.client.CredentialsError.__name__,
                APP_NAME, str(error))
        self.assertTrue(self.memento.check_error(*msgs))

        self.assert_dbus_signal_correct(self.client.CredentialsError, 'sa{ss}')

    def test_authorization_denied(self):
        """The AuthorizationDenied signal."""
        self.client.AuthorizationDenied(APP_NAME)
        msgs = (self.client.__class__.__name__,
                self.client.AuthorizationDenied.__name__, APP_NAME)
        self.assertTrue(self.memento.check_info(*msgs))

        self.assert_dbus_signal_correct(self.client.AuthorizationDenied, 's')
