# -*- coding: utf-8 -*-
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Windows tests."""

# pylint: disable=F0401
from _winreg import REG_SZ

from mocker import MATCH, Mocker, MockerTestCase

from twisted.internet import defer, reactor
from twisted.trial.unittest import TestCase
from twisted.spread.pb import (
    DeadReferenceError,
    PBClientFactory,
    PBServerFactory,
)
from ubuntu_sso.main import windows
from ubuntu_sso.main.windows import (
    signal,
    CredentialsManagement,
    CredentialsManagementClient,
    LOCALHOST,
    SignalBroadcaster,
    SSOCredentials,
    SSOCredentialsClient,
    SSOLogin,
    SSOLoginClient,
    UbuntuSSORoot,
    UbuntuSSOClient,
    get_sso_pb_port,
)

# because we are using twisted we have java like names C0103 and
# the issues that mocker brings with it like W0104
# pylint: disable=C0103,W0104


class SaveProtocolServerFactory(PBServerFactory):
    """A PBServerFactory that saves the latest connected client."""

    protocolInstance = None

    def clientConnectionMade(self, protocol):
        """Keep track of the given protocol."""
        self.protocolInstance = protocol


class FakeDecoratedObject(object):
    """An object that has decorators."""

    def __init__(self):
        """Create a new instance."""
        super(FakeDecoratedObject, self).__init__()

    @signal
    def on_no_args(self):
        """Get no args passwed."""

    @signal
    def on_just_args(self, *args):
        """Just get args."""

    @signal
    def on_just_kwargs(self, **kwargs):
        """Just get kwargs."""

    @signal
    def on_both_args(self, *args, **kwargs):
        """Both args."""


# pylint: disable=W0201
class SignalTestCase(MockerTestCase):
    """Test the signal decorator."""

    def setUp(self):
        """Set the tests."""
        super(SignalTestCase, self).setUp()
        self.fake_object = FakeDecoratedObject()
        self.cb = self.mocker.mock()

    def test_no_args(self):
        """Test when the cb should have no args."""
        self.fake_object.on_no_args_cb = self.cb
        self.cb()
        self.mocker.replay()
        self.fake_object.on_no_args()

    def test_just_args(self):
        """Test when the cb just has *args"""
        first = 'first'
        second = 'second'
        self.fake_object.on_just_args_cb = self.cb
        self.cb(first, second)
        self.mocker.replay()
        self.fake_object.on_just_args(first, second)

    def test_just_kwargs(self):
        """Test when the cb just has kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_just_kwargs_cb = self.cb
        self.cb(first=first, second=second)
        self.mocker.replay()
        self.fake_object.on_just_kwargs(first=first, second=second)

    def test_just_kwargs_empty(self):
        """Test when the cb just has kwargs."""
        self.fake_object.on_just_kwargs_cb = self.cb
        self.cb()
        self.mocker.replay()
        self.fake_object.on_just_kwargs()

    def test_both_args(self):
        """Test with args and kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_both_args_cb = self.cb
        self.cb(first, second, first=first, second=second)
        self.mocker.replay()
        self.fake_object.on_both_args(first, second, first=first,
                                      second=second)

    def test_both_args_no_kwargs(self):
        """Test with args and kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_both_args_cb = self.cb
        self.cb(first, second)
        self.mocker.replay()
        self.fake_object.on_both_args(first, second)

    def test_both_args_no_args(self):
        """Test with args and kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_both_args_cb = self.cb
        self.cb(first=first, second=second)
        self.mocker.replay()
        self.fake_object.on_both_args(first=first, second=second)
# pylint: enable=W0201


class FakeDeadRemoteClient(object):
    """A fake dead remote client."""

    def callRemote(self, signal_name, *args, **kwargs):
        """Fails with DeadReferenceError."""
        raise DeadReferenceError("Calling Stale Broker")


class FakeAliveRemoteClient(object):
    """A fake alive remote client."""

    def __init__(self):
        self.called = False

    def callRemote(self, signal_name, *args, **kwargs):
        """Returns a succeed."""
        self.called = True
        return defer.succeed(None)


class SignalBroadcasterTestCase(TestCase):
    """Test the SignalBroadcaster class."""

    def test_emit_signal_dead_reference(self):
        """Test dead reference while emiting the signal."""
        fake_remote_client = FakeDeadRemoteClient()
        sb = SignalBroadcaster()
        sb.remote_register_to_signals(fake_remote_client)
        self.assertIn(fake_remote_client, sb.clients)
        sb.emit_signal("sample_signal")
        self.assertNotIn(fake_remote_client, sb.clients)

    def test_emit_signal_some_dead_some_not(self):
        """Test a clean reference after a dead one."""
        fake_dead_remote = FakeDeadRemoteClient()
        fake_alive_remote = FakeAliveRemoteClient()
        sb = SignalBroadcaster()
        sb.remote_register_to_signals(fake_dead_remote)
        sb.remote_register_to_signals(fake_alive_remote)
        sb.emit_signal("sample_signal")
        self.assertTrue(fake_alive_remote.called, "The alive must be called.")


class SSOLoginTestCase(TestCase):
    """Test the login class."""

    def setUp(self):
        """Setup tests."""
        super(SSOLoginTestCase, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.except_to_errdict = self.mocker.replace(
                                        'ubuntu_sso.main.except_to_errdict')
        self.login = SSOLogin(None)
        self.login.root = self.root
        # start pb
        self.sso_root = UbuntuSSORoot(self.login, None, None)
        self.server_factory = SaveProtocolServerFactory(self.sso_root)
        # pylint: disable=E1101
        port = get_sso_pb_port()
        self.listener = reactor.listenTCP(port, self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = reactor.connectTCP(LOCALHOST, port,
                                            self.client_factory)
        # pylint: enable=E1101

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_sso_login')
        client = SSOLoginClient(remote)
        yield client.register_to_signals()
        # set the cb
        for signal_name in ['on_captcha_generated_cb',
                            'on_captcha_generation_error_cb',
                            'on_user_registered_cb',
                            'on_user_registration_error_cb',
                            'on_logged_in_cb',
                            'on_login_error_cb',
                            'on_user_not_validated_cb',
                            'on_email_validated_cb',
                            'on_email_validation_error_cb',
                            'on_password_reset_token_sent_cb',
                            'on_password_reset_error_cb',
                            'on_password_changed_cb',
                            'on_password_change_error_cb']:
            setattr(client, signal_name, self.mocker.mock())
        defer.returnValue(client)

    def test_emit_captcha_generated(self):
        """Test that the cb was called."""
        app_name = 'app'
        result = 'result'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_captcha_generated_cb(app_name, result)
            self.mocker.replay()
            self.login.emit_captcha_generated(app_name, result)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_captcha_generation_error(self):
        """Test that the cb was called."""
        app_name = 'app'
        error_value = 'error'
        raised_error = self.mocker.mock()

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            raised_error.value
            self.mocker.result(error_value)
            self.except_to_errdict(error_value)
            self.mocker.result(error_value)
            client.on_captcha_generation_error_cb(app_name, error_value)
            self.mocker.replay()
            self.login.emit_captcha_generation_error(app_name, raised_error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_generate_captcha(self):
        """Test the call from the client."""
        app_name = 'app'
        filename = 'file'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.generate_captcha(app_name, filename,
                                   self.login.emit_captcha_generated,
                                   self.login.emit_captcha_generation_error)
            self.mocker.replay()
            yield client.generate_captcha(app_name, filename)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_user_registered(self):
        """Test that the cb was called."""
        app_name = 'app'
        result = 'result'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_user_registered_cb(app_name, result)
            self.mocker.replay()
            self.login.emit_user_registered(app_name, result)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_user_registration_error(self):
        """Test that the cb was called."""
        app_name = 'app'
        error_value = 'error'
        raised_error = self.mocker.mock()

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            raised_error.value
            self.mocker.result(error_value)
            self.except_to_errdict(error_value)
            self.mocker.result(error_value)
            client.on_user_registration_error_cb(app_name, error_value)
            self.mocker.replay()
            self.login.emit_user_registration_error(app_name, raised_error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_register_user(self):
        """Test the call from the client."""
        app_name = 'app'
        email = 'email'
        password = 'password'
        displayname = 'name'
        captcha_id = 'captcha_id'
        captcha_solution = 'captcha_solution'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.register_user(app_name, email, password, displayname,
                                    captcha_id, captcha_solution,
                                    self.login.emit_user_registered,
                                    self.login.emit_user_registration_error)
            self.mocker.replay()
            yield client.register_user(app_name, email, password, displayname,
                                       captcha_id, captcha_solution)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_logged_in(self):
        """Test that the cb was called."""
        app_name = 'app'
        result = 'result'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_logged_in_cb(app_name, result)
            self.mocker.replay()
            self.login.emit_logged_in(app_name, result)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_login_error(self):
        """Test that the db was called."""
        app_name = 'app'
        error_value = 'error'
        raised_error = self.mocker.mock()

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            raised_error.value
            self.mocker.result(error_value)
            self.except_to_errdict(error_value)
            self.mocker.result(error_value)
            client.on_login_error_cb(app_name, error_value)
            self.mocker.replay()
            self.login.emit_login_error(app_name, raised_error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_user_not_validated(self):
        """Test that the cb was called."""
        app_name = 'app'
        result = 'result'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_user_not_validated_cb(app_name, result)
            self.mocker.replay()
            self.login.emit_user_not_validated(app_name, result)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_login(self):
        """Test the call from the client."""
        app_name = 'app'
        email = 'email'
        password = 'password'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.login(app_name, email, password,
                            self.login.emit_logged_in,
                            self.login.emit_login_error,
                            self.login.emit_user_not_validated)
            self.mocker.replay()
            yield client.login(app_name, email, password)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_email_validated(self):
        """Test the cb was called."""
        app_name = 'app'
        result = 'result'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_email_validated_cb(app_name, result)
            self.mocker.replay()
            self.login.emit_email_validated(app_name, result)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_email_validation_error(self):
        """Test the cb was called."""
        app_name = 'app'
        error_value = 'error'
        raised_error = self.mocker.mock()

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            raised_error.value
            self.mocker.result(error_value)
            self.except_to_errdict(error_value)
            self.mocker.result(error_value)
            client.on_email_validation_error_cb(app_name, error_value)
            self.mocker.replay()
            self.login.emit_email_validation_error(app_name, raised_error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_validate_email(self):
        """Test the client calll."""
        app_name = 'app'
        email = 'email'
        password = 'password'
        email_token = 'token'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.validate_email(app_name, email, password, email_token,
                                 self.login.emit_email_validated,
                                 self.login.emit_email_validation_error)
            self.mocker.replay()
            yield client.validate_email(app_name, email, password, email_token)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_password_reset_token_sent(self):
        """Test the cb was called."""
        app_name = 'app'
        result = 'result'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_password_reset_token_sent_cb(app_name, result)
            self.mocker.replay()
            self.login.emit_password_reset_token_sent(app_name, result)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_password_reset_error(self):
        """Test the cb was called."""
        app_name = 'app'
        error_value = 'error'
        raised_error = self.mocker.mock()

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            raised_error.value
            self.mocker.result(error_value)
            self.except_to_errdict(error_value)
            self.mocker.result(error_value)
            client.on_password_reset_error_cb(app_name, error_value)
            self.mocker.replay()
            self.login.emit_password_reset_error(app_name, raised_error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_request_password_reset_token(self):
        """Test the client call."""
        app_name = 'app'
        email = 'email'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.request_password_reset_token(app_name, email,
                                self.login.emit_password_reset_token_sent,
                                self.login.emit_password_reset_error)
            self.mocker.replay()
            client.request_password_reset_token(app_name, email)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_password_changed(self):
        """Test the cb was called."""
        app_name = 'app'
        result = 'result'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_password_changed_cb(app_name, result)
            self.mocker.replay()
            self.login.emit_password_changed(app_name, result)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_password_change_error(self):
        """Test the cb was called."""
        app_name = 'app'
        error_value = 'error'
        raised_error = self.mocker.mock()

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            raised_error.value
            self.mocker.result(error_value)
            self.except_to_errdict(error_value)
            self.mocker.result(error_value)
            client.on_password_change_error(app_name, error_value)
            self.mocker.replay()
            self.login.emit_password_change_error(app_name, raised_error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_set_new_password(self):
        """Test the client call."""
        app_name = 'app'
        email = 'email'
        token = 'token'
        new_password = 'password'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.set_new_password(app_name, email, token, new_password,
                                   self.login.emit_password_changed,
                                   self.login.emit_password_change_error)
            self.mocker.replay()
            yield client.set_new_password(app_name, email, token, new_password)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d


class SSOCredentialsTestCase(TestCase):
    """Test the credentials class."""

    def setUp(self):
        """Set up tests."""
        super(SSOCredentialsTestCase, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.except_to_errdict = self.mocker.replace(
                                        'ubuntu_sso.main.except_to_errdict')
        self.creds = SSOCredentials(None)
        self.creds.root = self.root
        # start pb
        self.sso_root = UbuntuSSORoot(None, self.creds, None)
        self.server_factory = SaveProtocolServerFactory(self.sso_root)
        # pylint: disable=E1101
        port = get_sso_pb_port()
        self.listener = reactor.listenTCP(port, self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = reactor.connectTCP(LOCALHOST, port,
                                            self.client_factory)
        # pylint: enable=E1101

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_sso_credentials')
        client = SSOCredentialsClient(remote)
        yield client.register_to_signals()
        # set the cb
        for signal_name in ['on_authorization_denied_cb',
                            'on_credentials_found_cb',
                            'on_credentials_error_cb']:
            setattr(client, signal_name, self.mocker.mock())
        defer.returnValue(client)

    def test_emit_authorization_denied(self):
        """Test that the cb is executed."""
        app_name = 'app'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_authorization_denied_cb(app_name)
            self.mocker.replay()
            self.creds.emit_authorization_denied(app_name)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_credentials_found(self):
        """Test that the cb was executed."""
        app_name = 'app'
        credentials = 'cred'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_credentials_found_cb(app_name, credentials)
            self.mocker.replay()
            self.creds.emit_credentials_found(app_name, credentials)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_credentials_error(self):
        """Test that the cb was executed."""
        app_name = 'app'
        raised_error = 'error'
        detail = 'detail'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_credentials_error_cb(app_name, raised_error, detail)
            self.mocker.replay()
            self.creds.emit_credentials_error(app_name, raised_error, detail)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_find_credentials_no_kword(self):
        """Ensure that the root is called."""
        app_name = 'app'
        callback = lambda: None
        errback = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.find_credentials(app_name, MATCH(callable),
                                       MATCH(callable))
            self.mocker.replay()
            yield client.find_credentials(app_name, callback, errback)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_find_credentials_(self):
        """Ensure that the root is called."""
        app_name = 'app'
        callback = lambda: None
        errback = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.find_credentials(app_name, MATCH(callable),
                                       MATCH(callable))
            self.mocker.replay()
            yield client.find_credentials(app_name, callback=callback,
                                          errback=errback)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_login_or_register_to_get_credentials(self):
        """Ensure that the root is called."""
        app_name = 'app'
        terms = 'terms'
        help_txt = 'help'
        window_id = 'window_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            # pylint: disable=W0212
            self.root.login_or_register_to_get_credentials(app_name, terms,
                                        help_txt, window_id,
                                        self.creds.emit_credentials_found,
                                        self.creds._process_error,
                                        self.creds.emit_authorization_denied,
                                        ui_module='ubuntu_sso.qt.gui')
            # pylint: enable=W0212
            self.mocker.replay()
            yield client.login_or_register_to_get_credentials(app_name, terms,
                                                          help_txt, window_id)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_login_to_get_credentials(self):
        """Ensure that the root is called."""
        app_name = 'app'
        help_txt = 'help'
        window_id = 'window_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            # pylint: disable=W0212
            self.root.login_to_get_credentials(app_name, help_txt, window_id,
                                        self.creds.emit_credentials_found,
                                        self.creds._process_error,
                                        self.creds.emit_authorization_denied,
                                        ui_module='ubuntu_sso.qt.gui')
            # pylint: enable=W0212
            self.mocker.replay()
            yield client.login_to_get_credentials(app_name, help_txt,
                                                  window_id)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_clear_token_no_kword(self):
        """Ensure that the root is called."""
        app_name = 'app'
        callback = lambda: None
        errback = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.clear_token(app_name, MATCH(callable), MATCH(callable))
            self.mocker.replay()
            yield client.clear_token(app_name, callback, errback)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_clear_token(self):
        """Ensure that the root is called."""
        app_name = 'app'
        callback = lambda: None
        errback = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.clear_token(app_name, MATCH(callable), MATCH(callable))
            self.mocker.replay()
            yield client.clear_token(app_name, callback=callback,
                                     errback=errback)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d


class CredentialsManagementTestCase(TestCase):
    """Test the management class."""

    def setUp(self):
        """Set up tests."""
        super(CredentialsManagementTestCase, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.except_to_errdict = self.mocker.replace(
                                        'ubuntu_sso.main.except_to_errdict')
        self.creds = CredentialsManagement(None, None)
        self.creds.root = self.root
        # start pb
        self.sso_root = UbuntuSSORoot(None, None, self.creds)
        self.server_factory = SaveProtocolServerFactory(self.sso_root)
        # pylint: disable=E1101
        port = get_sso_pb_port()
        self.listener = reactor.listenTCP(port, self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = reactor.connectTCP(LOCALHOST, port,
                                            self.client_factory)
        # pylint: enable=E1101

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_cred_manager')
        client = CredentialsManagementClient(remote)
        yield client.register_to_signals()
        # set the cb
        for signal_name in ['on_authorization_denied_cb',
                            'on_credentials_found_cb',
                            'on_credentials_not_found_cb',
                            'on_credentials_cleared_cb',
                            'on_credentials_stored_cb',
                            'on_credentials_error_cb']:
            setattr(client, signal_name, self.mocker.mock())
        defer.returnValue(client)

    def test_shutdown(self):
        """Test that root is called."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            # pylint: disable=W0104
            self.root.ref_count
            # pylint: enable=W0104
            self.mocker.result(1)
            self.root.shutdown()
            self.mocker.replay()
            yield client.shutdown()
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_authorization_denied(self):
        """Test the callback is called."""
        app_name = 'app'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            # pylint: disable=W0104
            self.root.ref_count
            # pylint: enable=W0104
            self.mocker.result(1)
            self.root.ref_count = 0
            client.on_authorization_denied_cb(app_name)
            self.mocker.replay()
            self.creds.emit_authorization_denied(app_name)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_credentials_found(self):
        """Test the callback is called."""
        app_name = 'app'
        creds = 'creds'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            # pylint: disable=W0104
            self.root.ref_count
            # pylint: enable=W0104
            self.mocker.result(1)
            self.root.ref_count = 0
            client.on_credentials_found_cb(app_name, creds)
            self.mocker.replay()
            self.creds.emit_credentials_found(app_name, creds)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_credentials_not_found(self):
        """Test the callback is called."""
        app_name = 'app'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            # pylint: disable=W0104
            self.root.ref_count
            # pylint: enable=W0104
            self.mocker.result(1)
            self.root.ref_count = 0
            client.on_credentials_not_found_cb(app_name)
            self.mocker.replay()
            self.creds.emit_credentials_not_found(app_name)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_credentials_cleared(self):
        """Test the callback is called."""
        app_name = 'app'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            # pylint: disable=W0104
            self.root.ref_count
            # pylint: enable=W0104
            self.mocker.result(1)
            self.root.ref_count = 0
            client.on_credentials_cleared_cb(app_name)
            self.mocker.replay()
            self.creds.emit_credentials_cleared(app_name)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_credentials_stored(self):
        """Test the callback is called."""
        app_name = 'app'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            # pylint: disable=W0104
            self.root.ref_count
            # pylint: enable=W0104
            self.mocker.result(1)
            self.root.ref_count = 0
            client.on_credentials_stored_cb(app_name)
            self.mocker.replay()
            self.creds.emit_credentials_stored(app_name)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_credentials_error(self):
        """Test the callback is called."""
        app_name = 'app'
        raised_error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            # pylint: disable=W0104
            self.root.ref_count
            # pylint: enable=W0104
            self.mocker.result(1)
            self.root.ref_count = 0
            client.on_credentials_error_cb(app_name, raised_error)
            self.mocker.replay()
            self.creds.emit_credentials_error(app_name, raised_error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_find_credentials(self):
        """Test that root is called."""
        app_name = 'app'
        args = 'args'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            # pylint: disable=W0212
            self.root.find_credentials(app_name, args, MATCH(callable),
                                       self.creds._process_failure)
            # pylint: enable=W0212
            self.root.shutdown()
            yield client.find_credentials(app_name, args)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_clear_credentials(self):
        """Test that root is called."""
        app_name = 'app'
        args = 'args'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            # pylint: disable=W0212
            self.root.clear_credentials(app_name, args, MATCH(callable),
                                        self.creds._process_failure)
            # pylint: enable=W0212
            self.mocker.replay()
            yield client.clear_credentials(app_name, args)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_store_credentials(self):
        """Test that root is called."""
        app_name = 'app'
        args = 'args'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            # pylint: disable=W0212
            self.root.store_credentials(app_name, args, MATCH(callable),
                               self.creds._process_failure)
            # pylint: enable=W0212
            self.mocker.replay()
            yield client.store_credentials(app_name, args)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_register(self):
        """Test that root is called."""
        app_name = 'app'
        args = 'args'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.register(app_name, args)
            self.mocker.replay()
            yield client.register(app_name, args)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_login(self):
        """Test that root is called."""
        app_name = 'app'
        args = 'args'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.login(app_name, args)
            self.mocker.replay()
            yield client.login(app_name, args)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_login_email_password(self):
        """Test that root is called."""
        app_name = 'app'
        args = 'args'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.login_email_password(app_name, args)
            self.mocker.replay()
            yield client.login_email_password(app_name, args)
            yield client.unregister_to_signals()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d


class MockRemoteObject(object):
    """A mock RemoteObject."""

    def __init__(self):
        """A place to store MockRemoteObjects created by this."""
        self.children = []

    def callRemote(self, method_name, *args, **kwargs):
        """Any call to a remote object returns one of us."""
        new_child = MockRemoteObject()
        self.children.append(new_child)
        return defer.succeed(new_child)


class MockPBClientFactory(object):
    """A mock PBClientFactory."""

    connected = False
    root_object = None

    def getRootObject(self):
        """Store that we were called; return a deferred."""
        assert self.connected == True
        self.root_object = MockRemoteObject()
        return defer.succeed(self.root_object)


class MockReactor(object):
    """A mock twisted.reactor."""

    def connectTCP(self, host, port, factory):
        """Set the factory as connected."""
        factory.connected = True


class MockActivationClient(object):
    """A mock tcpactivation.ActivationClient."""

    def __init__(self):
        """Initialize this mock instance."""
        self.config = None
        self.port_requested = False

    def _set_config(self, config):
        """Set the configuration."""
        self.config = config
        return self

    def get_active_port(self):
        """Return the port for tests."""
        self.port_requested = True
        return defer.succeed(self.config.port)


class UbuntuSSOClientTestCase(TestCase):
    """Tests for the UbuntuSSOClient class."""

    @defer.inlineCallbacks
    def test_connect(self):
        """Test the connect method."""
        ussoc = UbuntuSSOClient()
        mac = MockActivationClient()
        self.patch(windows, "reactor", MockReactor())
        self.patch(windows, "PBClientFactory", MockPBClientFactory)
        # pylint: disable=W0212
        self.patch(windows, "ActivationClient", mac._set_config)
        self.patch(windows, "get_activation_cmdline",
                                        lambda _: "sample executable")
        connect_result = yield ussoc.connect()
        self.assertEqual(connect_result, ussoc)
        self.assertTrue(mac.port_requested, "The port must be requested.")
        self.assertNotEqual(ussoc.sso_login, None)
        self.assertNotEqual(ussoc.cred_management, None)

    def test_get_activation_cmdline(self):
        """Test the get_activation_cmdline method."""
        SAMPLE_VALUE = "test 123"
        self.patch(windows, "OpenKey", lambda key, subkey: SAMPLE_VALUE)
        self.patch(windows, "QueryValueEx", lambda key, v: (key, REG_SZ))
        cmdline = windows.get_activation_cmdline(windows.SSO_SERVICE_NAME)
        self.assertEqual(SAMPLE_VALUE, cmdline)


class MockWin32APIs(object):
    """Some mock win32apis."""

    process_handle = object()
    TOKEN_ALL_ACCESS = object()
    TokenUser = object()

    def __init__(self, sample_token):
        """Initialize this mock instance."""
        self.sample_token = sample_token
        self.token_handle = object()

    def GetCurrentProcess(self):
        """Returns a fake process_handle."""
        return self.process_handle

    def OpenProcessToken(self, process_handle, access):
        """Open the process token."""
        assert process_handle is self.process_handle
        assert access is self.TOKEN_ALL_ACCESS
        return self.token_handle

    def GetTokenInformation(self, token_handle, info):
        """Get the information for this token."""
        assert token_handle == self.token_handle
        assert info == self.TokenUser
        return (self.sample_token, 0)


class AssortedTestCase(TestCase):
    """Tests for functions in the windows module."""

    def test_get_userid(self):
        """The returned user id is parsed ok."""
        expected_id = 1001
        sample_token = "abc-123-1001"

        win32apis = MockWin32APIs(sample_token)
        self.patch(windows, "win32process", win32apis)
        self.patch(windows, "win32security", win32apis)

        userid = windows.get_user_id()
        self.assertEqual(userid, expected_id)

    def _test_port_assignation(self, uid, expected_port):
        """Test a given uid/expected port combo."""
        self.patch(windows, "get_user_id", lambda: uid)
        self.assertEqual(windows.get_sso_pb_port(), expected_port)

    def test_get_sso_pb_port(self):
        """Test the get_sso_pb_port function."""
        uid = 1001
        uid_modulo = uid % windows.SSO_RESERVED_PORTS
        expected_port = (windows.SSO_BASE_PB_PORT +
                         uid_modulo * windows.SSO_PORT_ALLOCATION_STEP)
        self._test_port_assignation(uid, expected_port)

    def test_get_sso_pb_port_alt(self):
        """Test the get_sso_pb_port function."""
        uid = 2011 + windows.SSO_RESERVED_PORTS
        uid_modulo = uid % windows.SSO_RESERVED_PORTS
        expected_port = (windows.SSO_BASE_PB_PORT +
                         uid_modulo * windows.SSO_PORT_ALLOCATION_STEP)
        self._test_port_assignation(uid, expected_port)
