/* This file is part of Strigi Desktop Search
 *
 * Copyright (C) 2006 Jos van den Oever <jos@vandenoever.info>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include "strigilineedit.h"
#include "krun.h"
#include "hitsview.h"
#include "hitmenuitem.h"
#include <kpopupmenu.h>
#include <kmimetype.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <iostream>
#include <kdebug.h>
using namespace std;

StrigiLineEdit::StrigiLineEdit(QWidget*parent) : QLineEdit(parent),
        strigi(this) {
    connect(this, SIGNAL(returnPressed()), this, SLOT(openQuery()));
    connect(this, SIGNAL(textChanged(const QString&)),
        this, SLOT(slotTextChanged()));
    connect(&strigi,
        SIGNAL(gotHits(const QString&, const ClientInterface::Hits&)),
        this, SLOT(handleHits(const QString&, const ClientInterface::Hits&)));

    menu = new HitsView(this);
    hasmouse = false;
    animatedWidget = 0;
}
void
StrigiLineEdit::hideMenu() {
    if (!menu->hasMouse() && !hasmouse) {
        menu->hide();
        clearFocus();
    }
}
void
StrigiLineEdit::setAnimatedWidget(KAnimWidget* animWidget) {
  animatedWidget = animWidget;
}
QString
makeQuery(const QString q) {
    QString t = q;
    if (t.find('*') == -1 && t.find('?') == -1) {
        t += '*';
    }
    if (t.find(':') == -1) {
        t = "name:" + t;
    }
    return t;
}
void
StrigiLineEdit::slotTextChanged() {
    if (text().length() == 0) {
        lasttext = text();
        menu->clear();
        menu->hide();
        return;
    }
    QString t = makeQuery(text());
    if (t == lasttext) {
        return;
    }
    lasttext = t;
    strigi.query(t);
    if (animatedWidget) {
        animatedWidget->start();
        animatedWidget->show();
    }
}
void
StrigiLineEdit::openQuery() {
    if (text().length()) {
        QString cmd = "kfmclient newTab 'strigi:?q="
            + KURL::encode_string(makeQuery(text())) + "' 'text/html'";
        KRun::runCommand(cmd);
        menu->hide();
        clearFocus();
    }
}
void
StrigiLineEdit::keyPressEvent(QKeyEvent *e) {
    if (menu->isVisible()) {
        if (e->key() == Qt::Key_Up) {
            menu->setActiveItem(menu->count()-1);
        } else if (e->key() == Qt::Key_Down) {
            menu->setActiveItem(0);
        }
    }
    if (e->key() == Qt::Key_Escape) {
        menu->hide();
        clearFocus();
    }
    QLineEdit::keyPressEvent(e);
}
void
StrigiLineEdit::handleHits(const QString& query, const ClientInterface::Hits& hits) {
    if (query != lasttext) return;

    // transform 'name:abc*' into 'abc'
    QString input = query;
    if (input.find(':') >= 0) {
        input = input.mid(input.find(':')+1);
    }
    if (input.find('*') >= 0) {
        input = input.left(input.find('*'));
    }

    menu->clear();
    int last = (hits.hits.size() > 10) ? 10 : hits.hits.size();
    for (int i = 0; i < last; ++i) {
        HitMenuItem* hit = createHit(input, hits.hits[i]);
        menu->insertItem(hit, i, i);
        menu->connectItem( i, hit, SLOT(open()) );
    }

    if (last) {
        if (menu->isVisible()) {
            menu->hide();
        }
        menu->popup(QPoint(0, 0));
        setFocus();
    } else {
        menu->hide();
    }
    if (animatedWidget) {
        animatedWidget->stop();
        animatedWidget->hide();
    }
}
HitMenuItem*
StrigiLineEdit::createHit(const QString& input,
        const Strigi::IndexedDocument& hit) const {
    // get the appropriate icon
    QString iconname = KMimeType::mimeType(hit.mimetype.c_str())->icon(QString::null,0);
    QPixmap icon = strigiiconloader.loadIcon(iconname, KIcon::Panel);

    // get the filename
    QString text = "<html>";
    QString name;
/*    map<string, string>::const_iterator i = hit.properties.find("audio.title");
    if (i == hit.properties.end()) {
        i = hit.properties.find("email.subject");
    }
    if (i != hit.properties.end()) {
        name = i->second;
    } else {*/
        uint pos = hit.uri.rfind('/');
        if (pos == string::npos) {
            name = hit.uri;
        } else {
            name = hit.uri.substr(pos+1);
        }
//    }
    int index = name.find(input, 0, false);
    if (index >= 0) {
        name = name.left(index) + "<b>" + name.mid(index, input.length())
            + "</b>" + name.mid(index + input.length());
    }
    text += name + " <i>" + hit.fragment.substr(0, 500);
    return new HitMenuItem(icon, text, hit.uri);
}

