/******************************************************************************
 * smartdimmer.c                                                              *
 *                                                                            *
 * SmartDimmer adjustment tool.                                               *
 * Thanks to Thunderbird and sk1p from #nvclock @ freenode for their help :)  *
 *                                                                            *
 * July 23, 2005                                                              *
 * Erik Waling <erikw@acc.umu.se>                                             *
 ******************************************************************************/

#include <stdlib.h>
#include <getopt.h>
#include "nvclock.h"
#include "smartdimmer.h"


static struct option long_opts[] = {
    { "get"     , no_argument      , 0, 'g' },
    { "set"     , required_argument, 0, 's' },
    { "increase", no_argument      , 0, 'i' },
    { "decrease", no_argument      , 0, 'd' },
    { "help"    , no_argument      , 0, 'h' },
    { 0         , 0                , 0, 0   }
};
  
void sd_usage(char *argv0) {
    
    printf("nVIDIA SmartDimmer adjustment tool version %.1f.\n\n", VERSION);
    printf("Usage: %s [OPTION]...\n\n", argv0);
    printf("Options:\n");
    printf("\t-g  --get\t\tQuery brightness level.\n");
    printf("\t-s  --set <level>\tSet brightness level (%d-%d)\n", MIN_LEVEL, MAX_LEVEL);
    printf("\t-i  --increase\t\tIncrease brightness with one level.\n");
    printf("\t-d  --decrease\t\tDecrease brightness with one level.\n");
    printf("\t-h  --help\t\tPrints this help text.\n\n");
            
}

void sd_set_level(int level) {
    
    if (level < MIN_LEVEL)
        level = MIN_LEVEL;
    if (level > MAX_LEVEL)
        level = MAX_LEVEL;
    
    nv_card.PMC[PMC_SMARTDIM_OFFSET/4] = (level << SMARTDIM_SHIFT) | (nv_card.PMC[PMC_SMARTDIM_OFFSET/4] & ~SMARTDIM_MASK); 
}

int sd_get_level() {
    return (nv_card.PMC[PMC_SMARTDIM_OFFSET/4] & SMARTDIM_MASK) >> SMARTDIM_SHIFT;
}

int sd_init() {

    if (!init_nvclock()) 
        return -1;
    
    
    if (!set_card(0)) 
        return -2;
    
    return 0;
}

int main(int argc, char *argv[]) {
    
    int optind = 0, options = 0, setvalue = 0;
    int c;
    
    if (argc < 2) {
        sd_usage(argv[0]);
        return 0;
    }
    
    while ((c = getopt_long(argc, argv, "gs:idh", long_opts, &optind)) != -1) {
        switch (c) {
            case '?':
                fprintf(stderr, "\nTry `%s --help' for help.\n", argv[0]);
                return 1;
            case 'h':
                sd_usage(argv[0]);
                return 0;
            case 'g':
               options |= GET_BIT;
               break;
            case 's':
               if (isdigit(*optarg)) {
                   setvalue = atoi(optarg);
                   options |= SET_BIT;
               } else {
                   fprintf(stderr, "Illegal option value (-s): "
                           "Value has to be a non-negative number.\n");
                   return 1;
               }
               break;
            case 'i':
               options |= INC_BIT;
               break;
            case 'd':
               options |= DEC_BIT;
               break;
        }
    } 
    
    if (!options) {
        sd_usage(argv[0]);
        return 1;
    }
        
    switch (sd_init()) {
        case -1:
            fprintf(stderr, "init_nvclock() failed!\n");
            return 2;
        case -2:
            fprintf(stderr, "set_card() failed!\n");
            return 3;
    }
    
    if (options & INC_BIT)
        sd_set_level(sd_get_level() + 1);
    
    if (options & DEC_BIT)
        sd_set_level(sd_get_level() - 1);
    
    if (options & SET_BIT)
        sd_set_level(setvalue);
    
    if (options & GET_BIT)
        printf("SmartDimmer level: %d\n", sd_get_level());
    
    return 0;

}

