/*  Screem:  screem-helper-manager.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <glib/gi18n.h>

#include <glade/glade.h>

#include <libgnomeui/libgnomeui.h>

#include <string.h>

#include "screem-helper-manager.h"

#include "fileops.h"
#include "support.h"

#include "screem-application.h"
#include "screem-window.h"
#include "screem-session.h"

#include "screem-apps-model.h"

#include "screemmarshal.h"

struct ScreemHelperManagerPrivate {
	GSList *helpers;
};

enum {
	ARG_0
};

enum {
	ADDED_HELPER,
	REMOVED_HELPER,
	LAST_SIGNAL
};
static guint screem_helper_manager_signals[LAST_SIGNAL] = { 0 };

static void screem_helper_manager_finalize( GObject *object );
static void screem_helper_manager_set_prop( GObject *object, 
		guint property_id, const GValue *value, 
		GParamSpec *pspec );
static void screem_helper_manager_get_prop( GObject *object, 
		guint property_id, GValue *value, GParamSpec *pspec );

/* yuk */
extern ScreemApplication *app;

/* G Object stuff */
G_DEFINE_TYPE( ScreemHelperManager, screem_helper_manager, G_TYPE_OBJECT )

static void screem_helper_manager_class_init( ScreemHelperManagerClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_helper_manager_finalize;

	object_class->get_property = screem_helper_manager_get_prop;
	object_class->set_property = screem_helper_manager_set_prop;

	screem_helper_manager_signals[ ADDED_HELPER ] = 
		g_signal_new( "added",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperManagerClass, 
					       added ),
			      NULL, NULL,
			      screem_marshal_VOID__OBJECT,
			      G_TYPE_NONE, 1,
			      G_TYPE_OBJECT );
	screem_helper_manager_signals[ REMOVED_HELPER ] = 
		g_signal_new( "removed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperManagerClass, 
					       removed ),
			      NULL, NULL,
			      screem_marshal_VOID__OBJECT,
			      G_TYPE_NONE, 1,
			      G_TYPE_OBJECT );

}

static void screem_helper_manager_init( ScreemHelperManager *manager )
{
	ScreemHelperManagerPrivate *priv;
	
	priv = manager->priv = g_new0( ScreemHelperManagerPrivate, 1 );

}

static void screem_helper_manager_finalize( GObject *object )
{
	ScreemHelperManager *helper_manager;
	ScreemHelperManagerPrivate *priv;
	
	helper_manager = SCREEM_HELPER_MANAGER( object );
	priv = helper_manager->priv;

	g_free( priv );
	
	G_OBJECT_CLASS( screem_helper_manager_parent_class )->finalize( object );
}

static void screem_helper_manager_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec )
{
	ScreemHelperManager *helper_manager;
	ScreemHelperManagerPrivate *priv;
	
	helper_manager = SCREEM_HELPER_MANAGER( object );
	priv = helper_manager->priv;
	
	switch( property_id ) {
		default:
			break;
	}
}

static void screem_helper_manager_get_prop( GObject *object, guint property_id,
				    GValue *value, GParamSpec *pspec)
{
	ScreemHelperManager *helper_manager;
	ScreemHelperManagerPrivate *priv;
	
	helper_manager = SCREEM_HELPER_MANAGER( object );
	priv = helper_manager->priv;
	
	switch( property_id ) {
		default:
			break;
	}
}

/* Private API */

static gint helper_filter( const GnomeVFSFileInfo *info )
{
	return strcmp( "application/x-desktop", info->mime_type );
}

static void screem_helper_manager_load_helpers_from_dir( ScreemHelperManager *manager,
		const gchar *dir )
{
	GSList *helpers;
	GSList *list;
	ScreemHelper *helper;
	
	/* we recurse so we can have helper packs that can just be
	 * untared into ~/.screem/helpers to add en-mass */

	helpers = screem_vfs_scandir( dir, helper_filter,
			(GCompareFunc)strcmp, TRUE );
	for( list = helpers; list; list = list->next ) {
		helper = screem_helper_new_from_file( list->data );
		if( helper ) {
			screem_helper_manager_add_helper( manager,
					helper );
		}
		g_free( list->data );
	}
	g_slist_free( helpers );

}

static void screem_helper_manager_helper_failed( ScreemHelper *helper,
		const gchar *msg, ScreemHelperManager *manager )
{
	gchar *name;
	gchar *primary;

	g_object_get( G_OBJECT( helper ), "name", &name, NULL );
		
	primary = g_strdup_printf( _( "There was an error running the %s helper" ), name );

	screem_hig_alert( GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
			primary, msg, NULL );
	g_free( primary );
	g_free( name );
}

static void screem_helper_manager_helper_error( ScreemHelper *helper,
		gint status, ScreemHelperManager *manager )
{
	gchar *name;
	gchar *primary;
	const gchar *secondary;

	if( status != 0 ) {
		g_object_get( G_OBJECT( helper ), "name", &name, NULL );
	
		primary = g_strdup_printf( "The %s helper returned errors", 
				name );
		secondary = _( "The error window will contain details of these" );
		screem_hig_alert( GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE,
				primary, secondary, NULL );
		g_free( primary );
		g_free( name );
	}
}

static void screem_helper_manager_helper_select( GtkTreeSelection *sel )
{
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter fit;
	GtkTreeIter it;
	GtkWidget *widget;
	GladeXML *xml;
	static const gchar *iwidgets[] = {
		"stdin_none", "stdin_page", "stdin_user"
	};
	static const gchar *owidgets[] = {
		"stdout_msg", "stdout_page", "stdout_insert"
	};

       	view = gtk_tree_selection_get_tree_view( sel );

	xml = glade_get_widget_tree( GTK_WIDGET( view ) );

	if( gtk_tree_selection_get_selected( sel, &model, &fit ) ) {
		ScreemHelper *helper;
		gchar *name;
		gchar *path;
		gchar *icon;
		gint imode;
		gint omode;
		gboolean exec;
		gchar *exectype;
		GtkTreeIter tit;
		gboolean found;
	
		gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &it, &fit );
		model = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER( model ) );
				
		gtk_tree_model_get( model, &it, 3, &helper, -1 );
		g_object_get( G_OBJECT( helper ), 
				"name", &name,
				"pathname", &path,
				"iconpath", &icon,
				"input", &imode, 
				"output", &omode, 
				"exec_on_save", &exec,
				"exec_type", &exectype,
				NULL );
			
		widget = glade_xml_get_widget( xml, "helpername" );
		widget = GTK_BIN( widget )->child;
		gtk_entry_set_text( GTK_ENTRY( widget ), name );
		g_free( name );

		widget = glade_xml_get_widget( xml, "helperpath" );
		widget = GTK_BIN( widget )->child;
		gtk_entry_set_text( GTK_ENTRY( widget ), path );
		g_free( path );

		widget = glade_xml_get_widget( xml, "helpericon" );
		gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ),
				icon );
		g_free( icon );

		widget = glade_xml_get_widget( xml, iwidgets[ imode ] );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
						TRUE );
		widget = glade_xml_get_widget( xml, owidgets[ omode ] );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
						TRUE );

		widget = glade_xml_get_widget( xml, "helperexec" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), exec );
		widget = glade_xml_get_widget( xml, "helpertype" );
		model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
		found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &tit, 1, exectype );
		if( ! found ) {
			found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &tit, 0, exectype );
		}
		if( found ) {
			gtk_combo_box_set_active_iter( GTK_COMBO_BOX( widget ), &tit );
		} else {
			widget = GTK_BIN( widget )->child;
			gtk_entry_set_text( GTK_ENTRY( widget ), 
					exectype );
			g_free( exectype );
		}
		
		widget = glade_xml_get_widget( xml, "helperremove" );
		gtk_widget_set_sensitive( widget, TRUE );
		widget = glade_xml_get_widget( xml, "helperupdate" );
		gtk_widget_set_sensitive( widget, TRUE );
	} else {
		widget = glade_xml_get_widget( xml, "helpername" );
		widget = GTK_BIN( widget )->child;
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "helperpath" );
		widget = GTK_BIN( widget )->child;
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "helpericon" );
		gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ), "");
		widget = glade_xml_get_widget( xml, "helperremove" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "helperupdate" );
		gtk_widget_set_sensitive( widget, FALSE );
	
	}
}

static void screem_helper_manager_add_helper_cb( GtkWidget *widget,
	ScreemHelperManager *manager )
{
	GladeXML *xml;
	GtkWidget *view;
	ScreemHelper *helper;
	GtkTreeModel *model;
	GtkTreeIter it;

	const gchar *name;
	const gchar *pathname;
	gchar *iconpath;

	HelperInputMode imode;
	HelperOutputMode omode;

	gboolean exec;
	gchar *exectype;
	
	gboolean found;
	ScreemHelper *old;
	gchar *oname;

	xml = glade_get_widget_tree( widget );

	view = glade_xml_get_widget( xml, "helperlist" );

	widget = glade_xml_get_widget( xml, "helpername" );
	widget = GTK_BIN( widget )->child;
	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "helperpath" );
	widget = GTK_BIN( widget )->child;
	pathname = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "helpericon" );
	iconpath = gnome_icon_entry_get_filename( GNOME_ICON_ENTRY( widget ) );

	imode = SCREEM_HELPER_STDIN_NONE;
	widget = glade_xml_get_widget( xml, "stdin_page" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		imode = SCREEM_HELPER_STDIN_SELECTION;
	}
	widget = glade_xml_get_widget( xml, "stdin_user" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		imode = SCREEM_HELPER_STDIN_USER;
	}
	
	omode = SCREEM_HELPER_STDOUT_NONE;
	widget = glade_xml_get_widget( xml, "stdout_page" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		omode = SCREEM_HELPER_STDOUT_SELECTION;
	}
	widget = glade_xml_get_widget( xml, "stdout_insert" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		omode = SCREEM_HELPER_STDOUT_INSERT;
	}
	
	widget = glade_xml_get_widget( xml, "helperexec" );
	exec = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "helpertype" );
	model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
	if( gtk_combo_box_get_active_iter( GTK_COMBO_BOX( widget ), &it ) ) {
		gtk_tree_model_get( model, &it,
				1, &exectype,
				-1 );
	} else {
		widget = GTK_BIN( widget )->child;
		exectype = (gchar*)gtk_entry_get_text( GTK_ENTRY( widget ) );
		exectype = g_strdup( exectype );
	}
	
	helper = screem_helper_new( name, pathname, iconpath,
				    imode, omode );

	g_object_set( G_OBJECT( helper ), 
			"exec_on_save", exec,
			"exec_type", exectype,
			NULL );
	g_free( exectype );
	
	screem_helper_save( helper );
	
	screem_helper_manager_add_helper( manager, helper );
		
	found = FALSE;
	old = NULL;
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
	model = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER( model ) );
	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( model ),
					&it ) ) {
		do {
			old = NULL;
			oname = NULL;
			gtk_tree_model_get( GTK_TREE_MODEL( model ),
					&it, 3, &old, -1 );
			if( old ) {
				g_object_get( G_OBJECT( old ),
					"name", &oname,
					NULL );
			}
			found = ( oname && ! strcmp( name, oname ) );
		} while( ! found &&
			gtk_tree_model_iter_next( GTK_TREE_MODEL( model ),
						&it ) );
	}

	if( ! found ) {
		old = NULL;
		gtk_list_store_append( GTK_LIST_STORE( model ), &it );
	}
	gtk_list_store_set( GTK_LIST_STORE( model ), &it,
			    0, name, 
			    1, pathname,
			    2, iconpath,
			    3, helper,
			    -1 );
	g_free( iconpath );

	if( old ) {
		g_object_unref( old );
	}
}

static void screem_helper_manager_remove_helper_cb( GtkWidget *widget,
	ScreemHelperManager *manager )
{
	GladeXML *xml;
	GtkWidget *view;
	ScreemHelper *helper;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter fit;
	GtkTreeSelection *selection;

	xml = glade_get_widget_tree( widget );

	view = glade_xml_get_widget( xml, "helperlist" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	
	if( gtk_tree_selection_get_selected( selection, &model, &fit ) ) {
		gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &it, &fit );
		model = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER( model ) );
	
		gtk_tree_model_get( GTK_TREE_MODEL( model ),
				&it, 3, &helper, -1 );
	
		screem_helper_manager_remove_helper( manager, helper );

		gtk_list_store_remove( GTK_LIST_STORE( model ), &it );

		g_object_unref( G_OBJECT( helper ) );
	}
}

void screem_helper_manager_response( GtkWidget *dialog,
		gint button )
{
	GladeXML *xml;
	GtkWidget *widget;
	ScreemHelperManager *manager;
	ScreemSession *session;
	GtkWidget *view;
	GtkTreeSelection *selection;
	
	xml = glade_get_widget_tree( dialog );

	widget = glade_xml_get_widget( xml, "helperlist" );
	manager = g_object_get_data( G_OBJECT( widget ), "manager" );
	
	session = screem_application_get_session( app );
	
	switch( button ) {
		case 0: /* add */
			screem_helper_manager_add_helper_cb( widget, manager );			
			break;
		case 1: /* remove */
			screem_helper_manager_remove_helper_cb( widget, manager );			
			break;
		case 2: /* refresh */
			view = glade_xml_get_widget( xml, "helperlist" );
			selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
			g_signal_handlers_block_by_func( G_OBJECT( selection ), G_CALLBACK( screem_helper_manager_helper_select ),  NULL );

			screem_helper_manager_remove_helper_cb( widget, manager );			
			g_signal_handlers_unblock_by_func( G_OBJECT( selection ), G_CALLBACK( screem_helper_manager_helper_select ),  NULL );

			screem_helper_manager_add_helper_cb( widget, manager );			
			break;
		default:
			session = screem_application_get_session( app );
			screem_session_store_dialog( session, dialog );
			gtk_widget_destroy( dialog );
			break;
	}
}

static gboolean visible_func( GtkTreeModel *model, GtkTreeIter *it,
	gpointer data )
{
	gboolean ret;
	ScreemHelper *helper;
	gchar *pathname;
		
	ret = TRUE;
		
	pathname = NULL;
	gtk_tree_model_get( model, it, 3, &helper, -1 );
	if( helper ) {
		g_object_get( G_OBJECT( helper ), 
				"filename", &pathname, NULL );
	}

	if( pathname != NULL ) {
		ret = strncmp( DATADIR"/screem/helpers", pathname,
			strlen( DATADIR"/screem/helpers" ) );
		g_free( pathname );
	}
	
	return ret;	
}

static void get_browsers( ScreemHelperManager *manager,
		ScreemHelper *helper, GSList **ret )
{
	gchar *filename;
	gchar *browsers;
	gchar *remote_browsers;
	
	g_object_get( G_OBJECT( helper ), "filename", &filename, NULL );

	if( filename ) {
		browsers = g_strconcat( G_DIR_SEPARATOR_S, "browsers",
				G_DIR_SEPARATOR_S, NULL );
		remote_browsers = g_strconcat( G_DIR_SEPARATOR_S, 
				"browsers-remote",
				G_DIR_SEPARATOR_S, NULL );
		if( strstr( filename, browsers ) || 
		    strstr( filename, remote_browsers ) ) {
			*ret = g_slist_prepend( *ret, helper );
		}
		g_free( remote_browsers );
		g_free( browsers );
		g_free( filename );
	}
}

/* Public API */

ScreemHelperManager *screem_helper_manager_new( void )
{
	ScreemHelperManager *manager;

	manager = SCREEM_HELPER_MANAGER( g_object_new( SCREEM_TYPE_HELPER_MANAGER, 
				NULL ) );

	return manager;
}

void screem_helper_manager_load_helpers( ScreemHelperManager *manager )
{
	gchar *path;
	gchar *tmp;

	g_return_if_fail( SCREEM_IS_HELPER_MANAGER( manager ) );

	/* load default helpers if have any */
	path = g_build_path( G_DIR_SEPARATOR_S, DATADIR, "screem",
			"helpers", NULL );
	if( uri_exists( path, NULL ) ) {
		screem_helper_manager_load_helpers_from_dir( manager, path );
	}
	g_free( path );
	
	/* load helpers from user path */
	
	tmp = screem_get_dot_dir();
	path = g_build_path( G_DIR_SEPARATOR_S, tmp, "helpers", NULL );
	g_free( tmp );

	if( mkdir_recursive( path, GNOME_VFS_PERM_USER_ALL, 
				screem_application_file_op, app ) ) {
		screem_helper_manager_load_helpers_from_dir( manager,
				path );
	}

	g_free( path );
}

void screem_helper_manager_add_helper( ScreemHelperManager *manager,
		ScreemHelper *helper )
{
	ScreemHelperManagerPrivate *priv;

	g_return_if_fail( SCREEM_IS_HELPER_MANAGER( manager ) );
	g_return_if_fail( SCREEM_IS_HELPER( helper ) );

	priv = manager->priv;

	priv->helpers = g_slist_append( priv->helpers, helper );
	
	/* hookup callbacks */
	g_signal_connect( G_OBJECT( helper ), "helper_failed",
			  G_CALLBACK( screem_helper_manager_helper_failed ),
			  manager );
	g_signal_connect( G_OBJECT( helper ), "helper_error",
			  G_CALLBACK( screem_helper_manager_helper_error ),
			  manager );

	g_signal_emit( G_OBJECT( manager ),
		       screem_helper_manager_signals[ ADDED_HELPER ],
			       0, helper );
}

void screem_helper_manager_remove_helper( ScreemHelperManager *manager,
		ScreemHelper *helper )
{
	ScreemHelperManagerPrivate *priv;
	GSList *hlist;
	gchar *name;
	gchar *dname;

	gchar *path;
	gchar *tmp;
	
	g_return_if_fail( SCREEM_IS_HELPER_MANAGER( manager ) );
	g_return_if_fail( SCREEM_IS_HELPER( helper ) );
	
	priv = manager->priv;
	
	g_object_get( G_OBJECT( helper ), "name", &name, NULL );

	/* first try and find a matching helper in priv->helpers */
	for( hlist = priv->helpers; hlist; hlist = hlist->next ) {

		g_object_get( G_OBJECT( hlist->data ), 
				"name", &dname, NULL );

		if( ! strcmp( name, dname ) ) {
			g_free( dname );
			break;
		}
		g_free( dname );
	}
	tmp = screem_get_dot_dir();
	path = g_strconcat( tmp, G_DIR_SEPARATOR_S,
			"helpers", G_DIR_SEPARATOR_S,
			name, ".desktop", NULL );
	g_free( tmp );
	g_free( name );
	
	if( hlist ) {
		/* found it */
		helper = SCREEM_HELPER( hlist->data );
		priv->helpers = g_slist_remove( priv->helpers,
				hlist->data );

		g_signal_emit( G_OBJECT( manager ),
		       screem_helper_manager_signals[ REMOVED_HELPER ],
			       0, helper );

		delete_file( path, screem_application_file_op, app );
	}
	g_free( path );
}

void screem_helper_manager_exec_all_helpers( ScreemHelperManager *manager,
		ScreemPage *page, ScreemWindow *window )
{
	ScreemHelperManagerPrivate *priv;	
	GSList *helpers;
	const gchar *type;
	gchar *etype;
	gboolean exec;
	ScreemHelper *helper;
	
	g_return_if_fail( SCREEM_IS_HELPER_MANAGER( manager ) );
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	priv = manager->priv;
	
	type = screem_page_get_mime_type( page );

	for( helpers = priv->helpers; helpers; helpers = helpers->next ) {
		helper = SCREEM_HELPER( helpers->data );
		g_object_get( G_OBJECT( helper ),
				"exec_on_save", &exec,
				"exec_type", &etype,
				NULL );
		if( exec && g_pattern_match_simple( etype, type ) ) {
			screem_helper_execute( helper, page, window );
		}
		g_free( etype );
	}
}

void screem_helper_manager_edit_helpers( ScreemHelperManager *manager  )
{
	ScreemHelperManagerPrivate *priv;
	ScreemSession *session;
	GtkTreeModel *model;
	GtkListStore *store;
	GtkTreeModel *filter;
	GSList *helpers;
	GtkWidget *view;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	const gchar *titles[] = { "Name", "Path", "Icon", NULL };
	gint i;

	GladeXML *xml;
	GtkWidget *widget;
	GtkTreeIter it;
	
	g_return_if_fail( SCREEM_IS_HELPER_MANAGER( manager ) );

	priv = manager->priv;
	
	session = screem_application_get_session( app );
	
	xml = glade_xml_new( GLADE_PATH"/screem.glade", "helpers", NULL );
	glade_xml_signal_autoconnect( xml );	

	widget = glade_xml_get_widget( xml, "helpername" );
	screem_gtk_add_history( widget );

	model = screem_apps_model_get_default();
	widget = glade_xml_get_widget( xml, "helperpath" );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), model );
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			AVAILABLE_APP_COLUMN_EXEC );
	screem_apps_model_setup_cell_layout( widget );
	g_object_unref( model );
			
	store = gtk_list_store_new( 4, 
				    G_TYPE_STRING,
				    G_TYPE_STRING, G_TYPE_STRING,
				    G_TYPE_POINTER,
				    NULL );
	filter = gtk_tree_model_filter_new( GTK_TREE_MODEL( store ), NULL );
	gtk_tree_model_filter_set_visible_func( GTK_TREE_MODEL_FILTER( filter ),
		visible_func, NULL, NULL );
	
	/* build model */
	for( i = 0, helpers = priv->helpers; helpers; helpers = helpers->next, ++ i ) {
		ScreemHelper *helper;
		const gchar *name;
		const gchar *pathname;
		const gchar *icon;

		helper = SCREEM_HELPER( helpers->data );

		g_object_get( G_OBJECT( helper ), 
				"name", &name, "pathname", &pathname,
				"iconpath", &icon, NULL );
		
		gtk_list_store_insert_with_values( store, &it, i,
				0, name, 1, pathname, 2, icon, 
				3, helper, -1 );
	}

	/* create tree view */
	view = glade_xml_get_widget( xml, "helperlist" );
	g_object_set_data( G_OBJECT( view ), "manager", manager );

	for( i = 0; titles[ i ]; ++ i ) {
		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, titles[ i ] );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );
		gtk_tree_view_column_set_attributes( col, renderer, 
						     "text", i, NULL );
	}
	
	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
			GTK_TREE_MODEL( filter ) );
	g_object_unref( store );
	g_object_unref( filter );

	widget = glade_xml_get_widget( xml, "helpertype" );
	store = screem_support_get_mime_types_store();
	gtk_list_store_prepend( store, &it );
	gtk_list_store_set( store, &it,
			0, _( "All Types" ),
			1, "*",
			-1 );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), 
			GTK_TREE_MODEL( store ) ); 
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			0 );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
	g_object_unref( store );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_helper_manager_helper_select ),
			  NULL );

	widget = glade_xml_get_widget( xml, "helpers" );
	g_object_set_data( G_OBJECT( widget ), "session", session );

	screem_session_restore_dialog( session, widget );

	gtk_widget_show_all( widget );
}

void screem_helper_manager_foreach( ScreemHelperManager *manager,
		ScreemHelperManagerForeach func,
		gpointer data )
{
	ScreemHelperManagerPrivate *priv;
	GSList *list;

	g_return_if_fail( SCREEM_IS_HELPER_MANAGER( manager ) );
	g_return_if_fail( func != NULL );
	
	priv = manager->priv;

	for( list = priv->helpers; list; list = list->next ) {
		func( manager, list->data, data );
	}
}

GSList *screem_helper_manager_get_browsers( ScreemHelperManager *manager )
{
	GSList *ret;

	ret = NULL;
	screem_helper_manager_foreach( manager, 
			(ScreemHelperManagerForeach)get_browsers, &ret );

	ret = g_slist_reverse( ret );
	
	return ret;
}

