/*
   Unix SMB/CIFS implementation.
   async gid2sid
   Copyright (C) Volker Lendecke 2009

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "includes.h"
#include "winbindd.h"
#include "librpc/gen_ndr/cli_wbint.h"

struct wb_gid2sid_state {
	struct tevent_context *ev;
	char *dom_name;
	struct dom_sid sid;
};

static void wb_gid2sid_done(struct tevent_req *subreq);

struct tevent_req *wb_gid2sid_send(TALLOC_CTX *mem_ctx,
				   struct tevent_context *ev,
				   gid_t gid)
{
	struct tevent_req *req, *subreq;
	struct wb_gid2sid_state *state;
	struct winbindd_domain *domain;
	struct winbindd_child *child;
	bool expired;

	req = tevent_req_create(mem_ctx, &state, struct wb_gid2sid_state);
	if (req == NULL) {
		return NULL;
	}

	if (winbindd_use_idmap_cache()
	    && idmap_cache_find_gid2sid(gid, &state->sid, &expired)) {

		DEBUG(10, ("idmap_cache_find_gid2sid found %d%s\n",
			   (int)gid, expired ? " (expired)": ""));

		if (!expired || idmap_is_offline()) {
			if (is_null_sid(&state->sid)) {
				tevent_req_nterror(req,
						   NT_STATUS_NONE_MAPPED);
			} else {
				tevent_req_done(req);
			}
			return tevent_req_post(req, ev);
		}
	}

	state->dom_name = NULL;

	for (domain = domain_list(); domain != NULL; domain = domain->next) {
		if (domain->have_idmap_config
		    && (gid >= domain->id_range_low)
		    && (gid <= domain->id_range_high)) {
			state->dom_name = domain->name;
			break;
		}
	}

	child = idmap_child();

	subreq = rpccli_wbint_Gid2Sid_send(
		state, ev, child->rpccli, state->dom_name,
		gid, &state->sid);
	if (tevent_req_nomem(subreq, req)) {
		return tevent_req_post(req, ev);
	}
	tevent_req_set_callback(subreq, wb_gid2sid_done, req);
	return req;
}

static void wb_gid2sid_done(struct tevent_req *subreq)
{
	struct tevent_req *req = tevent_req_callback_data(
		subreq, struct tevent_req);
	struct wb_gid2sid_state *state = tevent_req_data(
		req, struct wb_gid2sid_state);
	NTSTATUS status, result;

	status = rpccli_wbint_Gid2Sid_recv(subreq, state, &result);
	TALLOC_FREE(subreq);
	if (!NT_STATUS_IS_OK(status)) {
		tevent_req_nterror(req, status);
		return;
	}
	if (!NT_STATUS_IS_OK(result)) {
		tevent_req_nterror(req, result);
		return;
	}
	tevent_req_done(req);
}

NTSTATUS wb_gid2sid_recv(struct tevent_req *req, struct dom_sid *sid)
{
	struct wb_gid2sid_state *state = tevent_req_data(
		req, struct wb_gid2sid_state);
	NTSTATUS status;

	if (tevent_req_is_nterror(req, &status)) {
		return status;
	}
	sid_copy(sid, &state->sid);
	return NT_STATUS_OK;
}
