/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Arek Korbik <arkadini@gmail.com>
 */

/*
 * OpenGL backend for Mac OS X, using Carbon and AGL.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <Carbon/Carbon.h>
#include "pgmaglbackend.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gl_aglbackend_debug);
#define GST_CAT_DEFAULT pgm_gl_aglbackend_debug

static void set_foreground();
static void init_agl(PgmAglBackend *aglbackend);
static void init_opengl_bundle(PgmAglBackend *backend);
static void dealloc_opengl_bundle(PgmAglBackend *backend);
static void translate_key_event (PgmAglBackend *aglbackend, PgmEventKey *event,
                                 EventRef cevent);
static pascal OSStatus app_event_handler(EventHandlerCallRef handler,
                                         EventRef event, void *user_data);
static pascal OSStatus window_event_handler(EventHandlerCallRef handler,
                                            EventRef event, void *user_data);
static gboolean install_app_event_handler(PgmBackend *backend);
static gboolean install_window_event_handler(PgmBackend *backend);
static gboolean post_create_window_event(PgmBackend *backend);
static gboolean do_create_window(PgmBackend *backend);
static gboolean pgm_agl_backend_dispatch_events (PgmBackend *backend);
static gboolean timed_dispatch_events (PgmBackend *backend);

#define CHECK_MAIN_EVENT_LOOP()                                         \
  (GST_DEBUG("!! IN THE MAIN T?: %c (0x%08lx, 0x%08lx)",                \
             GetCurrentEventLoop() == GetMainEventLoop() ? 'Y' : 'n',   \
             (unsigned long) GetMainEventLoop(),                        \
             (unsigned long) GetCurrentEventLoop()))

static PgmBackendClass *parent_class = NULL;


/* Private functions */

static void
init_agl(PgmAglBackend *aglbackend)
{
  GSource *to_src;
  guint ret = 0;

  GST_DEBUG_OBJECT (aglbackend, "init_agl");

  CHECK_MAIN_EVENT_LOOP();
  GST_DEBUG("!! numevents: c: %ld, m: %ld",
              GetNumEventsInQueue(GetCurrentEventQueue()),
              GetNumEventsInQueue(GetMainEventQueue()));

  set_foreground();

  install_app_event_handler(aglbackend);

  to_src = g_timeout_source_new(40);
  g_source_set_callback(to_src, (GSourceFunc) timed_dispatch_events,
                        aglbackend, NULL);
  GST_DEBUG("ATTACHING!!: %p %p", to_src, PGM_BACKEND(aglbackend)->context);
  ret = g_source_attach(to_src, PGM_BACKEND(aglbackend)->context->render_context);
  GST_DEBUG("ATTACHED!!: %u", ret);
}

static void
init_opengl_bundle(PgmAglBackend *backend)
{
  OSStatus err = noErr;
  CFURLRef bundleURLOpenGL;
  // FIXME: build the path properly
  const UInt8 *glpath = "/System/Library/Frameworks/OpenGL.framework"; // l:43

  bundleURLOpenGL = CFURLCreateFromFileSystemRepresentation(NULL, glpath,
                                                            43, true);
  if (!bundleURLOpenGL) {
    GST_ERROR_OBJECT(backend, "Could not create OpenGL Framework bundle URL");
    return;
  }

  backend->gl_bundle_ref = CFBundleCreate(kCFAllocatorDefault, bundleURLOpenGL);
  if (!backend->gl_bundle_ref) {
    GST_ERROR_OBJECT(backend, "Could not create OpenGL Framework bundle");
    return;
  }

  CFRelease (bundleURLOpenGL);
  if (!CFBundleLoadExecutable(backend->gl_bundle_ref)) {
    GST_ERROR_OBJECT(backend, "Could not load Mach-O executable");
    return;
  }

  //return err;
}

static void
dealloc_opengl_bundle(PgmAglBackend *backend)
{
  if (backend->gl_bundle_ref != NULL) {
    CFBundleUnloadExecutable (backend->gl_bundle_ref);
    CFRelease (backend->gl_bundle_ref);
    backend->gl_bundle_ref = NULL;
  }
}

static void
set_foreground()
{
  OSStatus err;
  ProcessSerialNumber psn;

  GST_DEBUG("set_foreground");

  err = GetCurrentProcess(&psn);
  if (!err) {
    // long live the "No hidden APIs!"
    extern int32_t CPSSetProcessName(ProcessSerialNumber * PSN, char *name);

    GST_DEBUG("enabling foreground operation");
    TransformProcessType(&psn, kProcessTransformToForegroundApplication);
    CPSSetProcessName(&psn, "Pigment app");
    SetFrontProcess(&psn);
  }
}

static gboolean
pgm_agl_backend_dispatch_events (PgmBackend *backend)
{
  OSStatus err = noErr;
  EventRef theEvent;
  EventTargetRef theTarget;

  GST_LOG_OBJECT (backend, "dispatch_events");
  CHECK_MAIN_EVENT_LOOP();

  theTarget = GetEventDispatcherTarget();

  while ((err = ReceiveNextEvent(0, NULL, kEventDurationNoWait, true,
                                 &theEvent)) == noErr)
  {
    GST_LOG_OBJECT(backend, "!!! DISPATCHING");
    SendEventToEventTarget(theEvent, theTarget);
    ReleaseEvent(theEvent);
  }

  if (err != eventLoopTimedOutErr)
    GST_LOG_OBJECT(backend, "ReceiveNextEvent = %ld", err);

  return TRUE;
}

static gboolean
timed_dispatch_events (PgmBackend *backend)
{
  OSStatus err = noErr;
  EventRef theEvent;
  EventTargetRef theTarget;

  //GST_DEBUG_OBJECT (backend, "timed_dispatch_events");

  //CHECK_MAIN_EVENT_LOOP();
  //GST_DEBUG("!! numevents: c: %ld, m: %ld",
  //           GetNumEventsInQueue(GetCurrentEventQueue()),
  //           GetNumEventsInQueue(GetMainEventQueue()));


  // just call the dispatcher method, but always return TRUE
  pgm_agl_backend_dispatch_events(backend);

  return TRUE;
}

static pascal OSStatus
app_event_handler(EventHandlerCallRef handler,
                  EventRef event, void *user_data)
{
  OSStatus err = eventNotHandledErr;
  Rect rectPort;
  WindowRef window = FrontWindow();
  UInt32 class = GetEventClass(event);
  UInt32 kind = GetEventKind(event);
  HICommand command;

  GST_LOG_OBJECT(user_data, "app_event_handler");

  if (window) {
    GetWindowPortBounds(window, &rectPort);
  }

  switch (class) {
  case kEventClassMouse:
    //handleWindowMouseEvents(handler, event);
    break;

  case kEventClassCommand:
    switch (kind) {
    case kEventProcessCommand:
      GetEventParameter(event, kEventParamDirectObject, kEventParamHICommand,
                        NULL, sizeof(command), NULL, &command);
      switch (command.commandID) {
      case kHICommandQuit:
        QuitApplicationEventLoop();
        break;
      }
      err = noErr;
      break;
    }
    break;
  }

  return err;
}

static void
translate_key_event (PgmAglBackend *aglbackend, PgmEventKey *event,
                     EventRef cevent)
{
  guint32 keycode = 0;
  UniChar keychar = 0;

  event->time = GetEventTime(cevent);
  GetEventParameter(cevent, kEventParamKeyCode, typeUInt32, NULL, sizeof(guint32),
                    NULL, &keycode);
  GST_WARNING("translate_key_event:: keycode = %ld, 0x%08lx", keycode, keycode);
  GetEventParameter(cevent, kEventParamKeyUnicodes, typeUnicodeText, NULL,
                    sizeof(UniChar), NULL, &keychar);
  GST_WARNING("translate_key_event:: keychar = %u, 0x%08x", keychar, keychar);

  switch (keycode) {
  case 0x73: // Home
    event->keyval = 0xff50;
    break;

  case 0x74: // Page Up
    event->keyval = 0xff55;
    break;

  case 0x77: // End
    event->keyval = 0xff57;
    break;

  case 0x79: // Page Down
    event->keyval = 0xff56;
    break;

  default:
    switch (keychar) {
    case 0x1c: // Left
      event->keyval = 0xff51;
      break;

    case 0x1d: // Right
      event->keyval = 0xff53;
      break;

    case 0x1e: // Up
      event->keyval = 0xff52;
      break;

    case 0x1f: // Down
      event->keyval = 0xff54;
      break;

    case 0x03: // 'mini' return
      event->keyval = 0xff0d;
      break;

    case 0x01: // BOL
      event->keyval = 0xff58;
      break;

    case 0x08: // Backspace
    case 0x09: // Tab
    case 0x0a: // Linefeed
    case 0x0b: // Clear
    case 0x0d: // Return
    case 0x1b: // Escape
      event->keyval = 0xff00 | keychar;
      break;

    case 0x7f: // Delete
      event->keyval = 0xffff;
      break;

    default:
      event->keyval = keychar;
    }
  }

  event->hardware_keycode = keycode;
}

static pascal OSStatus
window_event_handler(EventHandlerCallRef handler,
                     EventRef event, void *user_data)
{
  PgmAglBackend *aglbackend = (PgmAglBackend *) user_data;
  PgmGlViewport *glviewport = PGM_BACKEND(aglbackend)->context->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT(glviewport);

  OSStatus err = eventNotHandledErr;
  Rect rectPort;
  WindowRef window = FrontWindow();
  UInt32 class = GetEventClass(event);
  UInt32 kind = GetEventKind(event);

  PgmEvent *pgmevent = NULL;

  GST_LOG("window_event_handler");

  switch (class) {
  case kEventClassKeyboard:
    switch (kind) {
    case kEventRawKeyDown:
      GST_LOG("window_event_handler :: kEventRawKeyDown");
      pgmevent = pgm_event_new (PGM_KEY_PRESS);
      translate_key_event (aglbackend, (PgmEventKey *) pgmevent, event);
      break;

    case kEventRawKeyRepeat:
      GST_LOG("window_event_handler :: kEventRawKeyRepeat");
      pgmevent = pgm_event_new (PGM_KEY_PRESS);
      translate_key_event (aglbackend, (PgmEventKey *) pgmevent, event);
      break;

    case kEventRawKeyUp:
      GST_LOG("window_event_handler :: kEventRawKeyUp");
      pgmevent = pgm_event_new (PGM_KEY_RELEASE);
      translate_key_event (aglbackend, (PgmEventKey *) pgmevent, event);
      break;
    }
    break;

  case kEventClassWindow:
    GST_LOG("window_event_handler :: kEventClassWindow");
    GetEventParameter(event, kEventParamDirectObject, typeWindowRef, NULL,
                      sizeof(WindowRef), NULL, &window);
    switch (kind) {
    case kEventWindowCollapsing:
      //GetWindowPortBounds (window, &rectPort);
      //drawGL (window, false); // in this case just draw content
      //CompositeGLBufferIntoWindow(...->ctx, &rectPort, window);
      //err = UpdateCollapsedWindowDockTile(window);
      break;

    case kEventWindowActivated: // called initially and on click activation
      //...
      break;

    case kEventWindowDrawContent: // called before being shown
      //...
      break;

    case kEventWindowClose: // called when window is being closed (close box)
      //...
      //HideWindow(window);
      //TransitionWindow(window, kWindowFadeTransitionEffect,
      //                 kWindowHideTransitionAction, NULL);
      pgmevent = pgm_event_new (PGM_DELETE);
      ((PgmEventDelete *) pgmevent)->time = GetEventTime (event);
      break;

    case kEventWindowBoundsChanged: // called for resize and moves/drags
      //GetWindowPortBounds(window, &rectPort);
      //...
      break;

    case kEventWindowZoomed: // called after window has been zoomed (zoom
                             // button clicked)
      break;
    }
    break;
  }

  if (pgmevent != NULL)
    pgm_viewport_push_event (viewport, pgmevent);

  return err;
}

static gboolean
install_app_event_handler(PgmBackend *backend)
{
  OSStatus err;
  EventHandlerRef ref;
  EventTypeSpec	list[] = {{ kEventClassCommand, kEventProcessCommand },
                          { kEventClassCommand, kEventCommandUpdateStatus },
                          { kEventClassMouse, kEventMouseDown },
                          { kEventClassMouse, kEventMouseUp },
                          { kEventClassMouse, kEventMouseDragged },
                          { kEventClassMouse, kEventMouseWheelMoved }};
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND (backend);
  EventQueueRef cqref, mqref;

  GST_DEBUG_OBJECT (aglbackend, "install_app_event_handler");

  cqref = GetCurrentEventQueue();
  mqref = GetMainEventQueue();

  CHECK_MAIN_EVENT_LOOP();
  GST_DEBUG_OBJECT (aglbackend, "numevents: c: %ld, m: %ld",
                    GetNumEventsInQueue(cqref), GetNumEventsInQueue(mqref));

  err = InstallApplicationEventHandler(aglbackend->app_event_handler,
                                       GetEventTypeCount(list), list,
                                       backend, &ref);

  GST_DEBUG_OBJECT(aglbackend, "InstallApplicationEventHandler() = %ld",
                   err);
  return TRUE;
}

static gboolean
install_window_event_handler(PgmBackend *backend)
{
  OSStatus err;
  EventHandlerRef ref;
  EventTypeSpec list[] = { { kEventClassWindow, kEventWindowCollapsing },
                           { kEventClassWindow, kEventWindowShown },
                           { kEventClassWindow, kEventWindowActivated },
                           { kEventClassWindow, kEventWindowClose },
                           { kEventClassWindow, kEventWindowDrawContent },
                           { kEventClassWindow, kEventWindowBoundsChanged },
                           { kEventClassWindow, kEventWindowZoomed },
                           { kEventClassKeyboard, kEventRawKeyDown },
                           { kEventClassKeyboard, kEventRawKeyRepeat },
                           { kEventClassKeyboard, kEventRawKeyUp } };
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND (backend);

  GST_DEBUG_OBJECT (aglbackend, "install_window_event_handler");
  CHECK_MAIN_EVENT_LOOP();
  GST_LOG("!! numevents: c: %ld, m: %ld",
              GetNumEventsInQueue(GetCurrentEventQueue()),
              GetNumEventsInQueue(GetMainEventQueue()));

  //GST_WARNING("!! ARE WE IN THE MAIN THREAD?: %c",
  //            GetCurrentEventLoop() == GetMainEventLoop() ? 'Y' : 'n');

  err = InstallWindowEventHandler(aglbackend->win, aglbackend->win_event_handler,
                                  GetEventTypeCount (list), list,
                                  (void*) aglbackend, &ref); // add event handler

  GST_LOG_OBJECT(aglbackend, "InstallWindowEventHandler() = %ld", err);
  return TRUE;
}

static gboolean
post_create_window_event(PgmBackend *backend)
{
  OSStatus err;
  EventRef windowEvent;

  GST_DEBUG_OBJECT(backend, "posting synthetic event.");
  err = MacCreateEvent(nil, kEventClassCommand, kEventProcessCommand, 0,
                       kEventAttributeNone, &windowEvent);
  if (err == noErr) {
    HICommand command; // set HI command parameter...

    GST_LOG_OBJECT(backend, "posting synthetic event..");

    command.commandID = kHICommandNew; command.attributes = 0;
    command.menu.menuRef = 0; command.menu.menuItemIndex = 0;

    err = SetEventParameter(windowEvent, kEventParamDirectObject,
                            kEventParamHICommand, sizeof(command), &command);
    if (err == noErr) {
      GST_LOG_OBJECT(backend, "posting synthetic event...");
      err = PostEventToQueue(GetCurrentEventQueue(), windowEvent,
                             kEventPriorityHigh);
    }
    ReleaseEvent(windowEvent);
  }

  return TRUE;
}

static gboolean
do_create_window(PgmBackend *backend)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);
  EventHandlerRef ref;
  EventTypeSpec list[] = { { kEventClassWindow, kEventWindowCollapsing },
                           { kEventClassWindow, kEventWindowShown },
                           { kEventClassWindow, kEventWindowActivated },
                           { kEventClassWindow, kEventWindowClose },
                           { kEventClassWindow, kEventWindowDrawContent },
                           { kEventClassWindow, kEventWindowBoundsChanged },
                           { kEventClassWindow, kEventWindowZoomed },
                           { kEventClassKeyboard, kEventRawKeyDown },
                           { kEventClassKeyboard, kEventRawKeyRepeat },
                           { kEventClassKeyboard, kEventRawKeyUp } };
  //WindowRef window = NULL;
  Rect bounds = {100, 280, 700, 1000};
  OSStatus werr = noErr;

  PgmViewport *viewport;
  gint width, height;

  GST_DEBUG_OBJECT (aglbackend, "create_window");
  CHECK_MAIN_EVENT_LOOP();
  GST_WARNING("!! numevents: c: %ld, m: %ld",
              GetNumEventsInQueue(GetCurrentEventQueue()),
              GetNumEventsInQueue(GetMainEventQueue()));


  viewport = PGM_VIEWPORT (backend->context->glviewport);
  /**/
  pgm_viewport_get_size (viewport, &width, &height);
  bounds.right = bounds.left + width;
  bounds.bottom = bounds.top + height;
  /**/

  werr = CreateNewWindow(kDocumentWindowClass,
                         kWindowStandardDocumentAttributes |
                         kWindowCloseBoxAttribute |
                         kWindowFullZoomAttribute |
                         kWindowCollapseBoxAttribute |
                         kWindowResizableAttribute |
                         kWindowStandardHandlerAttribute |
                         kWindowLiveResizeAttribute,
                         &bounds, &aglbackend->win); // build window from scratch

  if (aglbackend->win)
  {
    install_window_event_handler(backend);

    //ShowWindow(aglbackend->win);
    //TransitionWindow(aglbackend->win, kWindowFadeTransitionEffect,
    //                 kWindowShowTransitionAction, NULL);
  } else {
    GST_ERROR_OBJECT (aglbackend, "CreateNewWindow() = %ld", werr);
  }

  {
    /* OpenGL part */
    OSStatus err = noErr;
    Rect rectPort;
    GLint attrib[] = { AGL_RGBA, AGL_DOUBLEBUFFER, AGL_DEPTH_SIZE, 16, AGL_NONE };
    //!pRecContext pContextInfo = GetCurrentContextInfo(window);
    ProcessSerialNumber psn = { 0, kCurrentProcess };

    aglbackend->pixfmt = aglChoosePixelFormat(NULL, 0, attrib);

    if (aglbackend->pixfmt)
      aglbackend->ctx = aglCreateContext(aglbackend->pixfmt, NULL);

    if (aglbackend->pixfmt) {
      short fNum;
      GLint swap = 1;
      CGRect viewRect = {{0.0f, 0.0f}, {0.0f, 0.0f}};

      GrafPtr portSave = NULL;
      GetPort (&portSave);
      SetPort ((GrafPtr) GetWindowPort(aglbackend->win));

      if(!aglSetDrawable(aglbackend->ctx, GetWindowPort(aglbackend->win)))
        //err = aglReportError();
        ;
      if (!aglSetCurrentContext(aglbackend->ctx))
        //err = aglReportError();
        ;
      aglbackend->currentVS = aglGetVirtualScreen(aglbackend->ctx); // sync renderer

      //aglSetInteger (aglbackend->ctx, AGL_SWAP_INTERVAL, &swap);

      SetPort (portSave);
    }
  }

  aglbackend->created = TRUE;

  return TRUE;
}

/* PgmBackend methods */

static gboolean
pgm_agl_backend_create_window (PgmBackend *backend)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_DEBUG_OBJECT (backend, "create_window");

  init_agl(aglbackend);

  return do_create_window(backend);
}

static gboolean
pgm_agl_backend_destroy_window (PgmBackend *backend)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_DEBUG_OBJECT (backend, "destroy_window");

  if (aglbackend->created) {
    HideWindow(aglbackend->win);

    aglSetDrawable(aglbackend->ctx, NULL);
    aglSetCurrentContext(NULL);
    aglDestroyContext(aglbackend->ctx);
    aglDestroyPixelFormat(aglbackend->pixfmt);
    //FIXME: destroy font lists?

    //SetWRefCon(aglbackend->win, NULL);
    DisposeWindow(aglbackend->win);

    aglbackend->win = NULL;
    aglbackend->created = FALSE;
  }

  return TRUE;
}

static void
pgm_agl_backend_set_title (PgmBackend *backend, const gchar *title)
{
  GST_DEBUG_OBJECT (backend, "set_title");

  /*
    SetWindowTitleWithCFString(aglbackend->win, ...);
   */
}

static gboolean
pgm_agl_backend_set_visibility (PgmBackend *backend,
                                gboolean visible)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_DEBUG_OBJECT (backend, "set_visibility");

  if (visible)
    ShowWindow(aglbackend->win);
  else
    HideWindow(aglbackend->win);

  return TRUE;
}

static gboolean
pgm_agl_backend_set_decorated (PgmBackend *backend,
                               gboolean decorated)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_DEBUG_OBJECT (backend, "set_decorated");

  /* FIXME */

  return FALSE;
}

static void
pgm_agl_backend_swap_buffers (PgmBackend *backend)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_LOG_OBJECT (backend, "swap_buffers");

  aglSwapBuffers(aglbackend->ctx);
}

static gpointer
pgm_agl_backend_get_proc_address (PgmBackend *backend,
                                  const gchar *proc_name)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_LOG_OBJECT (backend, "get_proc_address");


  return CFBundleGetFunctionPointerForName(aglbackend->gl_bundle_ref,
            CFStringCreateWithCStringNoCopy(NULL,
                                            proc_name,
                                            CFStringGetSystemEncoding(), NULL));
}

static gboolean
pgm_agl_backend_set_size (PgmBackend *backend,
                          gint width,
                          gint height)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);
  Rect r;
  gint diff = 0;

  GST_DEBUG_OBJECT (backend, "set_size");

  GetWindowBounds (aglbackend->win, kWindowContentRgn, &r);
  diff = r.right - r.left - width;
  r.left -= diff / 2;
  r.right = r.left + width;
  diff = r.bottom - r.top - height;
  r.top -= diff / 2;
  r.bottom = r.top + height;
  SetWindowBounds (aglbackend->win, kWindowContentRgn, &r);

  return TRUE;
}

static gboolean
pgm_agl_backend_set_fullscreen (PgmBackend *backend,
                                gboolean fullscreen)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_DEBUG_OBJECT (backend, "set_fullscreen");

  if (fullscreen) {
    SetSystemUIMode(kUIModeAllHidden, kUIOptionAutoShowMenuBar);
  } else {
    SetSystemUIMode(kUIModeNormal, 0);
  }

  return TRUE;
}

static void
pgm_agl_backend_get_screen_size_mm (PgmBackend *backend,
                                    gint *width,
                                    gint *height)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_LOG_OBJECT (backend, "get_screen_size_mm");

  *width = aglbackend->size_mm_width;
  *height = aglbackend->size_mm_height;
}

static gboolean
pgm_agl_backend_set_screen_resolution (PgmBackend *backend,
                                       gint width,
                                       gint height)
{
  GST_DEBUG_OBJECT (backend, "set_screen_resolution");

  return FALSE;
}

static void
pgm_agl_backend_get_screen_resolution (PgmBackend *backend,
                                       gint *width,
                                       gint *height)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);

  GST_LOG_OBJECT (backend, "get_screen_resolution");

  *width = aglbackend->resolution_width;
  *height = aglbackend->resolution_height;
}

static gboolean
pgm_agl_backend_build_text_lists (PgmBackend *backend)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);
  PgmContextProcAddress *gl = backend->context->gl;
  gint first;
  gint last;
  short f_num = 0;
  char f_name_pascal[] = {'\6', 'G', 'e', 'n', 'e', 'v', 'a', '\0'};

  GST_DEBUG_OBJECT (backend, "build_text_lists");

  aglbackend->text_lists = gl->gen_lists(256);
  if (!gl->is_list(aglbackend->text_lists))
  {
    GST_WARNING_OBJECT(aglbackend, "unable to build text display lists\n");
    return FALSE;
  }

  // FIXME: check for errors from GetFNum()...?
  GetFNum(f_name_pascal, &f_num);

  first = 0;
  last = 255;

  if (aglUseFont(aglbackend->ctx, f_num, bold, 8, first, last - first + 1,
                 aglbackend->text_lists + first)) {
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
  }

  return TRUE;
}

static gboolean
pgm_agl_backend_destroy_text_lists (PgmBackend *backend)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND(backend);
  PgmContextProcAddress *gl = backend->context->gl;

  GST_DEBUG_OBJECT (backend, "destroy_text_lists");

  if (gl->is_list(aglbackend->text_lists))
    gl->delete_lists(aglbackend->text_lists, 256);

  return TRUE;
}

static void
pgm_agl_backend_raster_text (PgmBackend *backend,
                             const gchar *text,
                             gfloat x,
                             gfloat y,
                             gfloat r,
                             gfloat g,
                             gfloat b)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND (backend);
  PgmContextProcAddress *gl = backend->context->gl;

  GST_DEBUG_OBJECT (backend, "raster_text");

  gl->load_identity ();
  gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
  gl->push_attrib (PGM_GL_LIST_BIT);
  gl->color_4f (r, g, b, 1.0f);
  gl->raster_pos_2f (x, y);
  gl->list_base (aglbackend->text_lists);
  gl->call_lists (strlen (text), PGM_GL_UNSIGNED_BYTE, (PgmGlUbyte *) text);
  gl->pop_attrib ();
}

static void
pgm_agl_backend_wait_for_vblank (PgmBackend *backend)
{
  GST_LOG_OBJECT (backend, "wait_for_vblank");
}

static void
pgm_agl_backend_notify_startup_complete (PgmBackend *backend)
{
  GST_LOG_OBJECT (backend, "notify_startup_complete");
}

static gboolean
pgm_agl_backend_set_cursor (PgmBackend *backend,
                            PgmViewportCursor cursor)
{
  GST_DEBUG_OBJECT (backend, "set_cursor");

  return FALSE;
}

static gboolean
pgm_agl_backend_set_icon (PgmBackend *backend,
                          GdkPixbuf *icon)
{
  GST_DEBUG_OBJECT (backend, "set_icon");

  return FALSE;
}

static void
pgm_agl_backend_set_drag_status (PgmBackend *backend,
                                 gboolean accept)
{
  GST_LOG_OBJECT (backend, "set_drag_status");
}

static gboolean
pgm_agl_backend_is_accelerated (PgmBackend *backend)
{
  GST_DEBUG_OBJECT (backend, "is_accelerated");

  return FALSE;
}

static gboolean
pgm_agl_backend_is_embeddable (PgmBackend *backend)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND (backend);

  GST_LOG_OBJECT (backend, "is_embeddable");

  /* Not supported yet */

  return FALSE;
}

static void
pgm_agl_backend_get_embedding_id (PgmBackend *backend,
                                  gulong *embedding_id)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND (backend);

  GST_LOG_OBJECT (backend, "get_embedding_id");

  *embedding_id = 0;
}

static gboolean
pgm_agl_backend_has_alpha_component (PgmBackend *backend)
{
  GST_DEBUG_OBJECT (backend, "has_alpha_component");

  return FALSE;
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmAglBackend, pgm_agl_backend, PGM_TYPE_BACKEND);

void pgm_agl_backend_register (GTypeModule *module)
{
  pgm_agl_backend_register_type (module);
}

static void
pgm_agl_backend_dispose (GObject *object)
{
  PgmAglBackend *aglbackend = PGM_AGL_BACKEND (object);

  GST_DEBUG_OBJECT (aglbackend, "dispose");

  pgm_agl_backend_destroy_text_lists(PGM_BACKEND(aglbackend));

  if (aglbackend->created)
    pgm_agl_backend_destroy_window(PGM_BACKEND(aglbackend));

  dealloc_opengl_bundle(aglbackend);

  // FIXME: free handler UPPs

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_agl_backend_class_init (PgmAglBackendClass *klass)
{
  GObjectClass *gobject_class;
  PgmBackendClass *backend_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gl_agl_debug, "pgm_gl_aglbackend", 0,
                           "OpenGL plugin: PgmAglBackend");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  backend_class = PGM_BACKEND_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_agl_backend_dispose);

  /* PgmBackend virtual table */
  backend_class->create_window =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_create_window);
  backend_class->destroy_window =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_destroy_window);
  backend_class->set_title = GST_DEBUG_FUNCPTR (pgm_agl_backend_set_title);
  backend_class->set_decorated =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_set_decorated);
  backend_class->swap_buffers =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_swap_buffers);
  backend_class->get_proc_address =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_get_proc_address);
  backend_class->set_size = GST_DEBUG_FUNCPTR (pgm_agl_backend_set_size);
  backend_class->set_fullscreen =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_set_fullscreen);
  backend_class->set_visibility =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_set_visibility);
  backend_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_get_screen_size_mm);
  backend_class->set_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_set_screen_resolution);
  backend_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_get_screen_resolution);
  backend_class->build_text_lists =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_build_text_lists);
  backend_class->destroy_text_lists =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_destroy_text_lists);
  backend_class->raster_text = GST_DEBUG_FUNCPTR (pgm_agl_backend_raster_text);
  backend_class->wait_for_vblank =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_wait_for_vblank);
  backend_class->notify_startup_complete =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_notify_startup_complete);
  backend_class->set_cursor = GST_DEBUG_FUNCPTR (pgm_agl_backend_set_cursor);
  backend_class->set_icon = GST_DEBUG_FUNCPTR (pgm_agl_backend_set_icon);
  backend_class->set_drag_status =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_set_drag_status);
  backend_class->is_accelerated =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_is_accelerated);
  backend_class->is_embeddable =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_is_embeddable);
  backend_class->get_embedding_id =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_get_embedding_id);
  backend_class->has_alpha_component =
    GST_DEBUG_FUNCPTR (pgm_agl_backend_has_alpha_component);
}

static void
pgm_agl_backend_class_finalize (PgmAglBackendClass *klass)
{
  return;
}

static void
pgm_agl_backend_init (PgmAglBackend *aglbackend)
{
  GST_INFO_OBJECT (aglbackend, "init");

  aglbackend->win = NULL;
  aglbackend->pixfmt = NULL;

  aglbackend->gl_bundle_ref = NULL;

  aglbackend->created = FALSE;

  init_opengl_bundle (aglbackend);

  //set_foreground();

  aglbackend->app_event_handler = NewEventHandlerUPP (app_event_handler);
  aglbackend->win_event_handler = NewEventHandlerUPP (window_event_handler);

  //install_app_event_handler(aglbackend);
}

/* Public methods */

PgmBackend *
pgm_agl_backend_new (PgmContext *context)
{
  PgmBackend *backend;

  backend = g_object_new (PGM_TYPE_AGL_BACKEND, NULL);
  GST_DEBUG_OBJECT (PGM_AGL_BACKEND (backend), "created new aglbackend");

  backend->context = context;

  return backend;
}
