/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Guillaume Emont <guillaume@fluendo.com>
 *            Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmmodule
 * @short_description: A class to load and unload plugins at runtime.
 *
 * <refsect2>
 * <para>
 * A #PgmModule object is responsible for loading and unloading plugins at
 * runtime. Most of the work is done in #GTypeModule from which #PgmModule
 * inherits. #GTypeModule decides when the plugin must be loaded or unloaded,
 * and calls pgm_module_load() or pgm_module_unload().
 * The #PgmModule object also keeps a reference to the #PgmPluginDesc structure
 * found in the plugin. All access to pgm_plugin_desc must be made between a
 * call to g_type_module_use() and a call to g_type_module_unuse(). #PgmModule
 * is an object internal to Pigment, and should not be used outside of it. It
 * is only a helper class for #PgmViewportFactory.
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-10-23 (0.3.2)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include "pgmmodule.h"

/* Maximal levels of recursion in the file system to search for plugins */
#define MAX_RECURSION_LEVELS 10

GST_DEBUG_CATEGORY_STATIC (pgm_module_debug);
#define GST_CAT_DEFAULT pgm_module_debug

G_DEFINE_TYPE (PgmModule, pgm_module, G_TYPE_TYPE_MODULE);

/* Private methods */

/* Recursively search for a Pigment plugin name in a directory with a recursion
 * limit. If a plugin is found it calsl the module's init() function and
 * returns TRUE with the module's handle and plugin_desc members filled, in the
 * other case it returns FALSE. */
static gboolean
find_plugin (PgmModule *module,
             const gchar *plugin_directory,
             gint recursion_level)
{
  GModule *_handle;
  PgmPluginDesc *_plugin_desc;
  GDir *dir;
  const gchar *dir_entry;
  gchar *filename;
  gint plugin_name_len;
  gboolean found;

  g_return_val_if_fail (module && module->name && *module->name, FALSE);

  plugin_name_len = strlen (module->name);
  found = FALSE;

  /* Open the directory */
  dir = g_dir_open (plugin_directory, 0, NULL);
  if (G_UNLIKELY (!dir))
    return FALSE;

  /* Parse all the files */
  while ((dir_entry = g_dir_read_name (dir)))
    {
      filename = g_strjoin ("/", plugin_directory, dir_entry, NULL);

      GST_LOG ("looking for plugin named %d at %s", module->name, dir_entry);

      /* Check if it's a directory and recurse */
      if (g_file_test (filename, G_FILE_TEST_IS_DIR))
        {
          if (recursion_level > 0)
            {
              GST_LOG ("found directory, recursing");
              found = find_plugin (module, filename, recursion_level - 1);
            }
          else
            GST_LOG ("found directory, but recursion level is too deep");
          g_free (filename);

          /* Break our parsing if a plugin has been found */
          if (found)
            break;
          else
            continue;
        }

      /* Check if it's a regular file */
      if (!g_file_test (filename, G_FILE_TEST_IS_REGULAR))
        {
          GST_LOG ("not a regular file, ignoring");
          g_free (filename);
          continue;
        }

      /* Check standard shared library extensions */
      if (!g_str_has_suffix (filename, ".so")
          && !g_str_has_suffix (filename, ".sl")
          && !g_str_has_suffix (filename, ".dll")
          && !g_str_has_suffix (filename, ".dynlib"))
        {
          GST_LOG ("extension is not recognized as module file, ignoring");
          g_free (filename);
          continue;
        }

      /* Open the module */
      GST_DEBUG ("opening module %s", dir_entry);
      _handle = g_module_open (filename, G_MODULE_BIND_LOCAL);
      if (G_UNLIKELY (!_handle))
        {
          GST_WARNING ("failed loading module: %s", g_module_error ());
          g_free (filename);
          continue;
        }

      /* Search our "pgm_plugin_desc" entry point in the plugin */
      if (!g_module_symbol (_handle, "pgm_plugin_desc",
                            (gpointer) &_plugin_desc))
        {
          GST_DEBUG ("cannot find plugin entry point, ignoring");

          if (G_UNLIKELY (!g_module_close (_handle)))
            GST_WARNING ("failed closing module %s", filename);
          _handle = NULL;
          g_free (filename);
          continue;
        }
      GST_DEBUG ("found plugin entry point");

      /* Is it the plugin we are looking for? */
      GST_DEBUG ("plugin '%s' version '%s'",
                 _plugin_desc->name, _plugin_desc->version);
      if (!g_ascii_strncasecmp (_plugin_desc->name, module->name,
                                strlen (_plugin_desc->name))
          && !g_ascii_strncasecmp (_plugin_desc->version, VERSION,
                                   strlen (VERSION)))
        {
          /* Check create function pointer */
          if (_plugin_desc->create)
            {
              gboolean init_success = TRUE;
              GST_DEBUG ("using plugin %s", filename);

              /* Initialize the plugin */
              if (_plugin_desc->init)
                {
                  init_success = FALSE;
                  GST_DEBUG ("calling plugin init func");
                  init_success = _plugin_desc->init (G_TYPE_MODULE (module));
                }
              if (init_success)
                {
                  module->handle = _handle;
                  module->plugin_desc = _plugin_desc;
                  g_free (filename);
                  found = TRUE;
                  break;
                }
            }
          else
            GST_WARNING ("plugin %s has a NULL create function pointer",
                         filename);
        }


      /* It's not the plugin we are searching for, let's close the module */
      if (G_UNLIKELY (!g_module_close (_handle)))
        GST_WARNING ("failed closing module %s", filename);

      _handle = NULL;
      g_free (filename);
    }

  g_dir_close (dir);

  return found;
}

/* Calls find_plugin, first on the directories in the PGM_PLUGIN_PATH_NAME
 * environment variable (if not empty), then on PGM_PLUGIN_PATH. */
static gboolean
load_plugin (PgmModule *module)
{
  const gchar *plugin_path = NULL;
  gboolean found = FALSE;

  /* Try to get the plugin from the plugin path environment variable */
  plugin_path = g_getenv (PGM_PLUGIN_PATH_NAME);
  if (plugin_path)
    {
      gchar **list = NULL;
      gint i = 0;

      GST_DEBUG ("PGM_PLUGIN_PATH set to %s", plugin_path);
      list = g_strsplit (plugin_path, G_SEARCHPATH_SEPARATOR_S, 0);
      while (list[i] && !found)
        {
          found = find_plugin (module, list[i], MAX_RECURSION_LEVELS);
          i++;
        }
      g_strfreev (list);
    }
  else
    GST_DEBUG ("PGM_PLUGIN_PATH not set");

  /* Then from our usual plugin path */
  if (!found)
    found = find_plugin (module, PGM_PLUGIN_PATH, MAX_RECURSION_LEVELS);

  return found;
}

/* GTypeModule methods */

static gboolean
pgm_module_load (GTypeModule *module)
{
  PgmModule *pgm_module = PGM_MODULE (module);

  g_return_val_if_fail (module, FALSE);
  g_return_val_if_fail (pgm_module->name && *pgm_module->name, FALSE);

  return load_plugin (pgm_module);
}

static void
pgm_module_unload (GTypeModule *module)
{
  PgmModule *pgm_module = PGM_MODULE (module);

  g_return_if_fail (module);
  g_return_if_fail (pgm_module->handle);

  if (pgm_module->plugin_desc && pgm_module->plugin_desc->shutdown)
    pgm_module->plugin_desc->shutdown (G_TYPE_MODULE (pgm_module));

  g_module_close (pgm_module->handle);

  pgm_module->plugin_desc = NULL;
  pgm_module->handle = NULL;
}

/* GObject stuff */

static void
pgm_module_dispose (GObject *object)
{
  PgmModule *module = PGM_MODULE (object);

  g_free (module->name);

  G_OBJECT_CLASS (pgm_module_parent_class)->dispose (object);
}

static void
pgm_module_class_init (PgmModuleClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;
  GTypeModuleClass *parent_class = G_TYPE_MODULE_CLASS (klass);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_module_dispose);

  parent_class->load = GST_DEBUG_FUNCPTR (pgm_module_load);
  parent_class->unload = GST_DEBUG_FUNCPTR (pgm_module_unload);

  GST_DEBUG_CATEGORY_INIT (pgm_module_debug, "pgm_module", 0, "pigment modules handling");
}

static void
pgm_module_init (PgmModule *module)
{
  module->name = NULL;
  module->handle = NULL;
  module->plugin_desc = NULL;
}

/* Public methods */

PgmModule *
pgm_module_new (const gchar *name)
{
  PgmModule *module = NULL;

  if (!(name && *name))
    return NULL;

  module = g_object_new (PGM_TYPE_MODULE, NULL);
  if (!module)
    return NULL;

  module->name = g_strdup (name);
  g_type_module_set_name (G_TYPE_MODULE (module), module->name);

  return module;
}
