/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmimage
 * @short_description: An image drawable supporting various media.
 * @see_also: #PgmDrawable, #PgmText, #PgmCanvas.
 *
 * <refsect2>
 * <para>
 * #PgmImage is a drawable displaying media. It supports various ways of loading
 * images through buffers, file paths or videos through GStreamer.
 * </para>
 * <title>Loading image data</title>
 * <para>
 * Image data loading can happen with three different functions:
 * <itemizedlist>
 * <listitem>
 * <para>
 * pgm_image_set_from_buffer() takes a pre-loaded data buffer and sets it as
 * the currently displayed image. This is useful when you want to use an image
 * loading library (GdkPixbuf, FreeImage, etc) in your application and just
 * provide the pixels to #PgmImage for display. The data buffer containing the
 * pixels is copied internally, you can free the data buffer from the
 * application side as soon as the function returns.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * pgm_image_set_from_gst_buffer() takes a GStreamer #GstBuffer and sets it as
 * the currently displayed image. This is mostly used to do video rendering.
 * There's no copying of the buffer data to optimize performances, indeed the
 * reference count of the buffer is going to be increased to keep the buffer
 * around while it's needed for rendering. When you call pgm_image_clear() the
 * reference to the buffer will be decreased and the buffer can get freed. Note
 * that this method is used by #PgmSink to render video frames directly in a
 * #PgmImage when the pipeline is playing.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * pgm_image_set_from_file() takes a path to an image file delegating image
 * loading to Pigment. Thus the loading is asynchronous and won't block the
 * Pigment main loop.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * <title>Sharing image data between #PgmImage objects</title>
 * <para>
 * pgm_image_set_from_image() is a convenient system to slave an image to
 * another one. Indeed you might want to load an image data once and then use
 * it in multiple image objects. In that case this image becomes a slave to the
 * one that has the image data loaded internally and each time it needs to draw
 * it will use that data.
 * </para>
 * <para>
 * Layout settings of the drawable are independent from one image to another.
 * That means that even if two image objects are using the same image, they can
 * have different colors, different #PgmDrawableLayoutType or different
 * #PgmDrawableAlignment.
 * </para>
 * <para>
 * Each time a new image data buffer is loaded in the master image object, all
 * the slave image objects are automatically updated. That means you can render
 * a video clip in ten different drawables without doing anything else than
 * slaving nine image objects to the one that's receiving the image data.
 * </para>
 * <title>Image data aspect ratio</title>
 * <para>
 * This rarely happens with normal images but video rendering often has non
 * square pixels video frames coming out of the video decoders (DVD, DV cameras,
 * etc). In that case a calculation has to be done when projecting to the
 * viewport so that we put in adequation both the pixel aspect ratio and the
 * source video aspect ratio. You can set the image aspect ratio using
 * pgm_image_set_aspect_ratio() and be assured that Pigment is going to render
 * that image correctly on the viewport.
 * </para>
 * <title>Benefitting from hardware acceleration</title>
 * <para>
 * Depending on the viewport implementation, some #PgmImagePixelFormat (color
 * space) can be supported or not. When it comes to video rendering, hardware
 * acceleration is very important and you need to know what kind of pixel
 * formats are convenient for the rendering backend. you can get a list of
 * supported (accelerated) pixel formats using
 * pgm_viewport_get_pixel_formats().
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-11-08 (0.3.2)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <glib/gstdio.h> /* g_fopen */
#include "pgmimage.h"
#include "pgmmarshal.h"

/* Number of bytes read at each callback */
#define ASYNC_LOADER_CHUNK_SIZE 1024

GST_DEBUG_CATEGORY_STATIC (pgm_image_debug);
#define GST_CAT_DEFAULT pgm_image_debug

/* Image signals */
enum {
  PIXBUF_LOADED,
  LAST_SIGNAL
};

/* Asynchronous pixbuf loader */
typedef struct {
  PgmImage        *image;    /* Image object */
  FILE            *stream;   /* File handle */
  GdkPixbufLoader *loader;   /* GDK Pixbuf loader */
  GMutex          *lock;     /* Access lock: avoid holding both this and the
                                image lock. Only exception: you must have the
                                image lock held when calling
                                async_loader_free().
                                */
  gboolean         loaded;   /* Is the loading finished? */
  guint            max_size; /* Scaling threshold */
  guint            source;   /* Pigment main loop source id */
} AsyncLoader;

static PgmDrawableClass *parent_class = NULL;
static guint pgm_image_signals[LAST_SIGNAL] = { 0 };

static void async_loader_free (AsyncLoader *async);

/* Private methods */

/* Update the ratio of the slaves.
 * Called with Image LOCK. */
static void
update_slaves_ratio (PgmImage *image)
{
  PgmImage *slave;
  GList *walk;

  walk = image->slaves;
  while (walk)
    {
      slave = walk->data;
      slave->par_n = image->par_n;
      slave->par_d = image->par_d;
      walk = walk->next;
    }
}

/* Do the actual clearing of image.
 * Called with Image LOCK. */
static PgmError
do_clear (PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  switch (image->storage_type)
    {
    case PGM_IMAGE_BUFFER:
      GST_DEBUG_OBJECT (image, "Clearing buffer image");
      g_free (image->data.buffer.buffer);
      break;

    case PGM_IMAGE_GST_BUFFER:
      GST_DEBUG_OBJECT (image, "Clearing GstBuffer image");
      gst_buffer_unref (image->data.gst_buffer.gst_buffer);
      break;

    case PGM_IMAGE_PIXBUF:
      GST_DEBUG_OBJECT (image, "Clearing GdkPixbuf image");
      gdk_pixbuf_unref ((GdkPixbuf *) (image->data.pixbuf.pixbuf));
      break;

    case PGM_IMAGE_IMAGE:
      GST_DEBUG_OBJECT (image, "Clearing slaved image");
      if (image->master)
        {
          /* Remove ourself from the master's slave list. */
          GST_OBJECT_LOCK (image->master);
          image->master->slaves = g_list_remove (image->master->slaves, image);
          GST_OBJECT_UNLOCK (image->master);
          image->master = NULL;
        }
      break;

    default:
      break;
    }

  image->storage_type = PGM_IMAGE_EMPTY;
  image->par_n = 0;
  image->par_d = 1;
  update_slaves_ratio (image);

  return PGM_ERROR_OK;
}

/* Asynchronous loader chunk reader callback */
static gint
async_loader_read_chunk_cb (gpointer data)
{
  AsyncLoader *async = (AsyncLoader*) data;
  PgmImage *image = async->image;
  gchar buffer[ASYNC_LOADER_CHUNK_SIZE];
  gsize bytes_read;
  GdkPixbuf *pixbuf;

  /* we want to avoid a disposal of the image between here and when we use it */
  gst_object_ref (image);

  g_mutex_lock (async->lock);

  /* the source may have been destroyed while we were waiting for the lock (e.g.
   * by async_loader_free() which holds that lock) */
  if (g_source_is_destroyed (g_main_current_source ()))
    {
      if (async->lock)
        g_mutex_unlock (async->lock);
      gst_object_unref (image);
      return FALSE;
    }

  /* Read the chunk */
  bytes_read = fread (buffer, 1, ASYNC_LOADER_CHUNK_SIZE, async->stream);
  if (ferror (async->stream))
    {
      GST_WARNING_OBJECT (image, "cannot read from the file: %s",
                        g_strerror (errno));
      goto error;
    }

  /* Write the chunk */
  if (!gdk_pixbuf_loader_write (async->loader, (const guchar*) buffer,
                                bytes_read, NULL))
    {
      GST_WARNING_OBJECT (image, "GdkPixbufLoader cannot parse the buffer");
      goto error;
    }

  /* Image data not fully loaded */
  if (!feof (async->stream))
    {
      g_mutex_unlock (async->lock);
      gst_object_unref (image);
      return TRUE;
    }
  /* Image data fully loaded */
  else
    {
      /* Closing makes the loader aware that its got the full file: without this
       * it's buggy with small files */
      gdk_pixbuf_loader_close (async->loader, NULL);

      /* Get the pixbuf */
      pixbuf = gdk_pixbuf_ref (gdk_pixbuf_loader_get_pixbuf (async->loader));
      if (!pixbuf)
        {
          GST_WARNING_OBJECT (image, "GdkPixbufLoader cannot get the pixbuf");
          goto error;
        }

      /* Now we don't need to access the async loader anymore, but we want to
       * get a lock on the image, which we should not do with the async loader
       * lock held */
      g_mutex_unlock (async->lock);

      /* Clean up the previous data */
      pgm_image_clear (image);

      GST_OBJECT_LOCK (image);
      /* Fill the new storage informations */
      image->storage_type = PGM_IMAGE_PIXBUF;
      image->par_n = gdk_pixbuf_get_width (pixbuf);
      image->par_d = gdk_pixbuf_get_height (pixbuf);
      image->data.pixbuf.pixbuf = pixbuf;
      update_slaves_ratio (image);
      GST_OBJECT_UNLOCK (image);


      /* And emit signals */
      _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_PIXBUF);
      g_signal_emit (G_OBJECT (image), pgm_image_signals[PIXBUF_LOADED], 0);

      /* We don't need the loader anymore */
      GST_OBJECT_LOCK (image);
      /* we free image->async_loader and not async since it may have been freed
       * and NULL-ified since we unlocked it */
      async_loader_free (image->async_loader);
      image->async_loader = NULL;
      GST_OBJECT_UNLOCK (image);

      gst_object_unref (image);
      return FALSE;
    }

 error:
  fclose (async->stream);
  async->stream = NULL;
  gdk_pixbuf_loader_close (async->loader, NULL);
  g_object_unref (async->loader);
  async->loader = NULL;
  async->loaded = TRUE;
  g_mutex_unlock (async->lock);
  gst_object_unref (image);
  return FALSE;
}

/* GdkPixbufLoader handler called whenever loader's figured out the size */
static void
async_loader_size_prepared_cb (GdkPixbufLoader *loader,
                               gint width,
                               gint height,
                               gpointer data)
{
  AsyncLoader *async = (AsyncLoader*) data;
  gint max_size = (gint) async->max_size;

  /* Request a scaling if the width or the height exceeds the maximum size */
  if (width > max_size || height > max_size)
    {
      if (width >= height)
        {
          height = height * max_size / width;
          width = max_size;
        }
      else
        {
          width = width * max_size / height;
          height = max_size;
        }

      gdk_pixbuf_loader_set_size (loader, width, height);
    }
}

/* Asynchronous loader creation */
static AsyncLoader*
async_loader_new (PgmImage *image,
                  FILE *stream,
                  guint max_size)
{
  AsyncLoader *async;

  async = g_slice_new0 (AsyncLoader);
  if (G_UNLIKELY (!async))
    {
      GST_WARNING_OBJECT (image, "cannot create the asynchronous image loader");
      return NULL;
    }

  async->image = image;
  async->stream = stream;
  async->max_size = max_size;
  async->loaded = FALSE;
  async->lock = g_mutex_new ();
  async->loader = gdk_pixbuf_loader_new ();

  /* Connect to 'size_prepared' signal to adapt image size as requested */
  if (max_size != 0)
    g_signal_connect (async->loader, "size_prepared",
                      G_CALLBACK (async_loader_size_prepared_cb), async);

  /* Add the progressive loading source to the main Pigment loop */
  async->source = g_idle_add (async_loader_read_chunk_cb, async);

  return async;
}

/* Asynchronous loader destruction.
 * Note: This function should be called with the lock on the corresponding image
 * held.
 */
static void
async_loader_free (AsyncLoader *async)
{
  if (async == NULL)
    return;

  if (!async->loaded)
    {
      g_source_remove (async->source);

      /* from this point on, async_loader_read_chunk_cb() cannot be called, we
       * lock in case it has been called before the source removal. */
      g_mutex_lock (async->lock);
      g_mutex_unlock (async->lock);
      fclose (async->stream);
      async->stream = NULL;
      gdk_pixbuf_loader_close (async->loader, NULL);
      g_object_unref (async->loader);
      async->loader = NULL;
    }

  g_mutex_free (async->lock);
  async->lock = NULL;

  g_slice_free (AsyncLoader, async);
}

/* GObject stuff */

G_DEFINE_TYPE (PgmImage, pgm_image, PGM_TYPE_DRAWABLE);

static void
pgm_image_dispose (GObject *object)
{
  PgmImage *image = PGM_IMAGE (object);

  GST_OBJECT_LOCK (image);
  async_loader_free (image->async_loader);
  image->async_loader = NULL;
  GST_OBJECT_UNLOCK (image);

  /* Image is a master */
  if (image->slaves)
    {
      GList *copy = NULL, *walk = NULL;

      /* We need a copy of the slaves list since it gets modified in
       * pgm_image_clear */
      copy = g_list_copy (image->slaves);

      for (walk = copy; walk; walk = walk->next)
        pgm_image_clear (walk->data);

      g_list_free (copy);

      if (image->slaves)
        GST_DEBUG_OBJECT (image, "Slave list not completely cleared");
    }

  GST_OBJECT_LOCK (image);
  do_clear (image);
  GST_OBJECT_UNLOCK (image);

  pgm_mat4x4_free (image->mapping_matrix);
  image->mapping_matrix = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_image_class_init (PgmImageClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  /**
   * PgmImage::pixbuf-loaded:
   * @image: the #PgmImage
   *
   * Will be emitted after @image has finished to load its data from the file
   * path given in the pgm_image_set_from_file() method.
   */
  pgm_image_signals[PIXBUF_LOADED] =
    g_signal_new ("pixbuf-loaded", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmImageClass, pixbuf_loaded),
                  NULL, NULL, pgm_marshal_VOID__VOID, G_TYPE_NONE, 0);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_image_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_image_debug, "pgm_image", 0,
                           "Pigment Image object");
}

static void
pgm_image_init (PgmImage *image)
{
  image->mapping_matrix = pgm_mat4x4_new_identity ();
  image->storage_type = PGM_IMAGE_EMPTY;
  image->par_n = 0;
  image->par_d = 1;
  image->align = PGM_IMAGE_CENTER;
  image->layout = PGM_IMAGE_SCALED;
  image->interp = PGM_IMAGE_BILINEAR;
  image->master = NULL;
  image->slaves = NULL;
  image->border_width = 0.0f;
  image->border_inner_r = 255;
  image->border_inner_g = 255;
  image->border_inner_b = 255;
  image->border_inner_a = 255;
  image->border_outer_r = 255;
  image->border_outer_g = 255;
  image->border_outer_b = 255;
  image->border_outer_a = 255;
  image->async_loader = NULL;
}

/* public methods */

/**
 * pgm_image_new:
 *
 * Creates a new #PgmImage instance.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new (void)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_file:
 * @filename: the filename.
 * @max_size: the maximum size of the image in pixels before loading it in the
 * #PgmImage or 0 to not constrain the size.
 *
 * Creates a new #PgmImage instance loading progressively an image from the
 * given @filename. It optionally pre-scales the image so that it has a maximum
 * width and height of @max_size.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_file (const gchar *filename,
                         guint max_size)
{
  PgmImage *image;
  PgmError ret = PGM_ERROR_OK;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image from file");

  ret = pgm_image_set_from_file (image, filename, max_size);
  if (PGM_ERROR_OK != ret)
    {
      gst_object_unref (image);
      return NULL;
    }

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_buffer:
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the image rowstride in bytes (number of bytes per line).
 * @size: the buffer size in bytes.
 * @data: a pointer to the data buffer.
 *
 * Creates a new #PgmImage instance with the image from the given buffer.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_buffer (PgmImagePixelFormat format,
                           guint width,
                           guint height,
                           guint stride,
                           guint size,
                           gconstpointer data)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  pgm_image_set_from_buffer (image, format, width, height, stride, size, data);

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_image:
 * @src_image: a #PgmImage which will be used as the master image.
 *
 * Creates a new #PgmImage instance with an image slaved from the
 * image of @src_image.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_image (PgmImage *src_image)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  pgm_image_set_from_image (image, src_image);

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_set_from_file:
 * @image: a #PgmImage object.
 * @filename: a filename.
 * @max_size: the maximum size of the image in pixels before loading it in the
 * #PgmImage or 0 to not constrain the size.
 *
 * Loads an image from the file @filename. It optionally pre-scales the image so
 * it has a maximum width and height of @max_size.
 *
 * This function is meant to be asynchronous, it loads the image by small chunks
 * of 1024 bytes using an idle source in the Pigment mainloop. The consequence
 * being that the storage type of @image doesn't change immediately, but only
 * once the whole image is loaded. You can connect a callback to the
 * <link linkend="PgmImage-pixbuf-loaded">pixbuf-loaded</link> signal to know when
 * the loading is done.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_file (PgmImage *image,
                         const gchar *filename,
                         guint max_size)
{
  FILE *stream;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  /* Get a file handle from the filename */
  stream = g_fopen (filename, "rb");
  if (!stream)
    {
      GST_WARNING_OBJECT (image, "cannot open %s: %s", filename,
                        g_strerror (errno));
      return PGM_ERROR_X;
    }

  /* Free the current asynchronous loader and create the new one */
  GST_OBJECT_LOCK (image);
  async_loader_free ((AsyncLoader*) image->async_loader);
  image->async_loader = (gpointer) async_loader_new (image, stream, max_size);
  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_buffer:
 * @image: a #PgmImage object.
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the rowstride of the image in bytes (number of bytes per line).
 * @size: the buffer size in bytes.
 * @data: a pointer to the data buffer.
 *
 * Loads an image in @image from an existing buffer using pixel format @format.
 * If you don't know the rowstride of the image you can set stride to 0. @data
 * is copied internally you can free it right after the function call returns.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_buffer (PgmImage *image,
                           PgmImagePixelFormat format,
                           guint width,
                           guint height,
                           guint stride,
                           guint size,
                           gconstpointer data)
{
  gpointer _data = NULL;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (data != NULL, PGM_ERROR_X);

  /* Let's set the storage data */
  GST_OBJECT_LOCK (image);

  /* The buffer sent is not the first of this size, it's not needed to call
   * a clear, just free the buffer. */
  if (G_LIKELY (image->storage_type == PGM_IMAGE_GST_BUFFER
                && image->data.buffer.width == width
                && image->data.buffer.height == height
                && image->data.buffer.format == format
                && image->data.buffer.stride == stride))
    {
      if (G_LIKELY (image->data.buffer.buffer))
        g_free (image->data.buffer.buffer);
    }

  /* It's the first */
  else
    {
      /* Let's clear if needed */
      if (image->storage_type != PGM_IMAGE_EMPTY)
        {
          do_clear (image);
          GST_OBJECT_UNLOCK (image);
          _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);
          GST_OBJECT_LOCK (image);
        }

      /* Store buffer informations */
      image->storage_type = PGM_IMAGE_BUFFER;
      image->data.gst_buffer.format = format;
      image->data.gst_buffer.width = width;
      image->data.gst_buffer.height = height;
      image->data.gst_buffer.stride = stride;
      image->data.buffer.size = size;

      /* Store ratio */
      image->par_n = width;
      image->par_d = height;
      update_slaves_ratio (image);
    }

  /* Try to copy the given buffer */
  _data = g_memdup (data, size);

  image->data.buffer.buffer = (guint8 *) _data;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_BUFFER);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_gst_buffer:
 * @image: a #PgmImage object.
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the rowstride of the image in bytes (number of bytes per line).
 * @buffer: A #GstBuffer reference containing the video frame.
 *
 * Loads an image in @image from an existing #GstBuffer using the pixel format
 * @format. If you don't know the rowstride of the image you can set stride
 * to 0. @buffer will have its reference count increased by 1 and will not get
 * freed until the drawable gets cleaned up or that a new buffer is loaded.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_gst_buffer (PgmImage *image,
                               PgmImagePixelFormat format,
                               guint width,
                               guint height,
                               guint stride,
                               GstBuffer *buffer)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (GST_IS_BUFFER (buffer), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  /* The GstBuffer sent is not the first, it's not needed to call a clear,
   * just unref the GstBuffer. I mean that the implementation should not be
   * signaled to clear its allocated size/format for the stream of buffers
   * since it's really heavy to clear for each new one. */
  if (G_LIKELY (image->storage_type == PGM_IMAGE_GST_BUFFER
                && image->data.gst_buffer.width == width
                && image->data.gst_buffer.height == height
                && image->data.gst_buffer.format == format
                && image->data.gst_buffer.stride == stride))
    {
      if (G_LIKELY (image->data.gst_buffer.gst_buffer))
        gst_buffer_unref (image->data.gst_buffer.gst_buffer);
    }

  /* It's the first */
  else
    {
      /* Let's clear if needed */
      if (image->storage_type != PGM_IMAGE_EMPTY)
        {
          do_clear (image);
          GST_OBJECT_UNLOCK (image);
          _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);
          GST_OBJECT_LOCK (image);
        }

      /* Store buffer informations */
      image->storage_type = PGM_IMAGE_GST_BUFFER;
      image->data.gst_buffer.format = format;
      image->data.gst_buffer.width = width;
      image->data.gst_buffer.height = height;
      image->data.gst_buffer.stride = stride;

      /* Store ratio */
      image->par_n = width;
      image->par_d = height;
      update_slaves_ratio (image);
    }

  /* Take a ref on the GstBuffer */
  image->data.gst_buffer.gst_buffer = gst_buffer_ref (buffer);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_GST_BUFFER);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_image:
 * @image: a #PgmImage object.
 * @src_image: the source #PgmImage object to use as a master.
 *
 * Slaves @image to @src_image. Every change to @src_image is reflected on
 * @image until you remove @image from the canvas or you call pgm_image_clear()
 * on @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_image (PgmImage *image,
                          PgmImage *src_image)
{
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_IMAGE (src_image), PGM_ERROR_X);

  GST_DEBUG_OBJECT (image, "using image from %s", GST_OBJECT_NAME (src_image));

  GST_OBJECT_LOCK (image);

  /* Trying to deadlock us? :) */
  if (G_UNLIKELY (image == src_image || image == src_image->master))
    {
      GST_WARNING_OBJECT (image, "trying to do a master/slave loop!");
      GST_OBJECT_UNLOCK (image);
      ret = PGM_ERROR_X;
      goto beach;
    }

  do_clear (image);
  GST_OBJECT_UNLOCK (image);
  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);
  GST_OBJECT_LOCK (image);

  image->storage_type = PGM_IMAGE_IMAGE;

  GST_OBJECT_LOCK (src_image);

  /* The master requested is already a slave */
  if (G_UNLIKELY (src_image->master))
    {
      GST_DEBUG_OBJECT (image, "%s is already a slave to %s, using its master",
                        GST_OBJECT_NAME (src_image),
                        GST_OBJECT_NAME (src_image->master));

      GST_OBJECT_LOCK (src_image->master);

      src_image->master->slaves = g_list_append (src_image->master->slaves,
                                                 image);
      image->master = src_image->master;
      GST_OBJECT_UNLOCK (src_image->master);
    }

  /* The master requested is not a slave */
  else
    {
      /* Add ourself to the slave list with an increased reference */
      src_image->slaves = g_list_append (src_image->slaves, image);
      image->master = src_image;
    }

  image->par_n = src_image->par_n;
  image->par_d = src_image->par_d;

  GST_OBJECT_UNLOCK (src_image);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_IMAGE);

beach:
  return ret;
}

/**
 * pgm_image_clear:
 * @image: a #PgmImage object.
 *
 * Removes any image from @image. If @image had some image data loaded, it's
 * cleared, if there was a #GstBuffer used, it's unreffed and if the @image was
 * a slave to another it is not anymore. If @image has slave images they all
 * get cleared but they still are slaves to @image. So if you load a new image
 * to @image, all the slaves will load it too.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_clear (PgmImage *image)
{
  PgmError ret = PGM_ERROR_OK;

  GST_OBJECT_LOCK (image);
  async_loader_free (image->async_loader);
  image->async_loader = NULL;
  ret = do_clear (image);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);

  return ret;
}

/**
 * pgm_image_get_storage_type:
 * @image: a #PgmImage object.
 * @storage: a #PgmImageStorageType where the storage type is going to be
 * stored.
 *
 * Retrieves the type of representation being used by @image to store image
 * data. If @image has no image data, the return value will be
 * #PGM_IMAGE_EMPTY.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_storage_type (PgmImage *image,
                            PgmImageStorageType *storage)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (storage != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *storage = image->storage_type;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_mapping_matrix:
 * @image: a #PgmImage object.
 * @mapping_matrix: a #PgmMat4x4 object.
 *
 * Defines the transformation to apply to the stored image when it is rendered.
 * You can make the stored image slide over the drawable, rotate around it,
 * stretch and shrink, or any combination of the three.
 *
 * Each point in an image can be defined by an (x, y) vector, which we call
 * the source position, representing the horizontal (x) and vertical (y)
 * positions (with values between 0 for left/top and 1 for right/bottom).
 * When the image is drawn on a surface, each point (x, y) is drawn on the
 * (x', y') coordinate of the surface. We call (x', y') the destination
 * position. The default mapping matrix is the identity, you have
 * (x', y') == (x, y). Once you have called the function, the destination
 * position is calculated by multiplying @mapping_matrix with the source
 * position vector. To reset the mapping matrix, just set the identity.
 *
 * @mapping_matrix is a 4x4 matrix since the source and destination positions
 * can be represented as 4 coordinate vectors (x, y, z, w) and (x', y', z', w').
 * Unless you know what you are doing, you should not worry about z and w, and
 * arrange for your matrix to leave them unchanged.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_mapping_matrix (PgmImage *image,
                              PgmMat4x4 *mapping_matrix)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (mapping_matrix != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);
  pgm_mat4x4_set_from_mat4x4 (image->mapping_matrix, mapping_matrix);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_MAPPING_MATRIX);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_mapping_matrix:
 * @image: a #PgmImage object.
 * @mapping_matrix: a pointer to a #PgmMat4x4 pointer where the mapping matrix
 * is going to be stored. pgm_mat4x4_free() after use.
 *
 * Retrieves in @matrix the current mapping matrix applied to @image. 
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_mapping_matrix (PgmImage *image,
                              PgmMat4x4 **mapping_matrix)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (mapping_matrix != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);
  *mapping_matrix = pgm_mat4x4_copy (image->mapping_matrix);
  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_alignment:
 * @image: a #PgmImage object.
 * @align: a #PgmImageAlignment combination of flags.
 *
 * Defines the way @image aligns the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_alignment (PgmImage *image,
                         PgmImageAlignment align)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->align = align;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_ALIGNMENT);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_alignment:
 * @image: a #PgmImage object.
 * @align: a pointer to a #PgmImageAlignment where alignment flags are going
 * to be stored.
 *
 * Retrieves in @align the way @image aligns the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_alignment (PgmImage *image,
                         PgmImageAlignment *align)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (align != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *align = image->align;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_layout:
 * @image: a #PgmImage object.
 * @layout: a #PgmImageLayoutType layout type.
 *
 * Defines the way @image layouts its stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_layout (PgmImage *image,
                      PgmImageLayoutType layout)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->layout = layout;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_LAYOUT);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_layout:
 * @image: a #PgmImage object.
 * @layout: a pointer to a #PgmImageLayoutType where the layout type is going
 * to be stored.
 *
 * Retrieves in @layout the way @image layouts its its stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */

PgmError
pgm_image_get_layout (PgmImage *image,
                      PgmImageLayoutType *layout)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (layout != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *layout = image->layout;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_interp:
 * @image: a #PgmImage object.
 * @interp: the interpolation type.
 *
 * Defines that @image will be rendered using @interp as its interpolation
 * type.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_interp (PgmImage *image,
                      PgmImageInterpType interp)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->interp = interp;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_INTERP);

  return  PGM_ERROR_OK;
}

/**
 * pgm_image_get_interp:
 * @image: a #PgmImage object.
 * @interp: a pointer to a #PgmImageInterpType where the interpolation type
 * is going to be stored.
 *
 * Retrieves in @interp the current interpolation type of @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_interp (PgmImage *image,
                      PgmImageInterpType *interp)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (interp != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *interp = image->interp;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_aspect_ratio:
 * @image: a #PgmImage object.
 * @numerator: the numerator of the aspect ratio fraction.
 * @denominator: the denominator of the aspect ratio fraction.
 *
 * Customizes the aspect ratio of the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_aspect_ratio (PgmImage *image,
                            guint numerator,
                            guint denominator)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  if (numerator == image->par_n && denominator == image->par_d)
    {
      GST_OBJECT_UNLOCK (image);
      return PGM_ERROR_OK;
    }

  image->par_n = numerator;
  image->par_d = MAX (denominator, 1);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_ASPECT_RATIO);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_aspect_ratio:
 * @image: a #PgmImage object.
 * @numerator: a pointer to a #guint where the numerator of the aspect ratio
 * fraction will be stored.
 * @denominator: a pointer to a #guint where the denominator of the aspect
 * ratio fraction will be stored.
 *
 * Retrieves the aspect ratio of the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_aspect_ratio (PgmImage *image,
                            guint *numerator,
                            guint *denominator)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (numerator != NULL, PGM_ERROR_X);
  g_return_val_if_fail (denominator != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *numerator = image->par_n;
  *denominator = image->par_d;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_border_width:
 * @image: a #PgmImage object.
 * @width: the border with. 0.0 by default.
 *
 * Defines the border width drawn around @image.
 *
 * Note that the border is drawn around the image, inside the drawable. When
 * you change the size of the border, the image will be down-scaled.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_border_width (PgmImage *image,
                            gfloat width)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  /* Avoid further computations if the width doesn't change */
  if (width == image->border_width)
    {
      GST_OBJECT_UNLOCK (image);
      return PGM_ERROR_OK;
    }

  /* Clamp width to [0, MAX_FLOAT] */
  image->border_width = MAX (width, 0);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_BORDER_WIDTH);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_border_width:
 * @image: a #PgmImage object.
 * @width: a pointer to a #gfloat where the border width will be stored.
 *
 * Retrieves the width of the border drawn around #image inside the drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_border_width (PgmImage *image,
                            gfloat *width)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *width = image->border_width;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_border_inner_color:
 * @image: a #PgmImage object.
 * @red: the border inner red color. 255 by default.
 * @green: the border inner green color. 255 by default.
 * @blue: the border inner blue color. 255 by default.
 * @alpha: the border inner alpha color. 255 by default.
 *
 * Defines the border inner color drawn around @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_border_inner_color (PgmImage *image,
                                  guchar red,
                                  guchar green,
                                  guchar blue,
                                  guchar alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->border_inner_r = red;
  image->border_inner_g = green;
  image->border_inner_b = blue;
  image->border_inner_a = alpha;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                              PGM_IMAGE_BORDER_INNER_COLOR);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_border_inner_color:
 * @image: a #PgmImage object.
 * @red: a pointer to a #guchar where the border inner red color will be stored.
 * @green: a pointer to a #guchar where the border inner green color will be
 * stored.
 * @blue: a pointer to a #guchar where the border inner blue color will be
 * stored.
 * @alpha: a pointer to a #guchar where the border inner alpha color will be
 * stored.
 *
 * Retrieves the inner color of the border drawn around #image inside the
 * drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_border_inner_color (PgmImage *image,
                                  guchar *red,
                                  guchar *green,
                                  guchar *blue,
                                  guchar *alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (red != NULL, PGM_ERROR_X);
  g_return_val_if_fail (green != NULL, PGM_ERROR_X);
  g_return_val_if_fail (blue != NULL, PGM_ERROR_X);
  g_return_val_if_fail (alpha != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *red = image->border_inner_r;
  *green = image->border_inner_g;
  *blue = image->border_inner_b;
  *alpha = image->border_inner_a;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_border_outer_color:
 * @image: a #PgmImage object.
 * @red: the border outer red color. 255 by default.
 * @green: the border outer green color. 255 by default.
 * @blue: the border outer blue color. 255 by default.
 * @alpha: the border outer alpha color. 255 by default.
 *
 * Defines the border outer color drawn around @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_border_outer_color (PgmImage *image,
                                  guchar red,
                                  guchar green,
                                  guchar blue,
                                  guchar alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->border_outer_r = red;
  image->border_outer_g = green;
  image->border_outer_b = blue;
  image->border_outer_a = alpha;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                              PGM_IMAGE_BORDER_OUTER_COLOR);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_border_outer_color:
 * @image: a #PgmImage object.
 * @red: a pointer to a #guchar where the border outer red color will be stored.
 * @green: a pointer to a #guchar where the border outer green color will be
 * stored.
 * @blue: a pointer to a #guchar where the border outer blue color will be
 * stored.
 * @alpha: a pointer to a #guchar where the border outer alpha color will be
 * stored.
 *
 * Retrieves the outer color of the border drawn around #image inside the
 * drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_border_outer_color (PgmImage *image,
                                  guchar *red,
                                  guchar *green,
                                  guchar *blue,
                                  guchar *alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (red != NULL, PGM_ERROR_X);
  g_return_val_if_fail (green != NULL, PGM_ERROR_X);
  g_return_val_if_fail (blue != NULL, PGM_ERROR_X);
  g_return_val_if_fail (alpha != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *red = image->border_outer_r;
  *green = image->border_outer_g;
  *blue = image->border_outer_b;
  *alpha = image->border_outer_a;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}
