/**
  @file btcond-bt.h

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#ifndef _BTCOND_BT_H_
#define _BTCOND_BT_H_

#include <stdint.h>
#include <dbus/dbus.h>
#include <glib.h>

typedef struct {
    int         __ref;
    uint8_t     type;     /**< 'r' == RFCOMM, 'l' == L2CAP */
    char       *bda;      /**< BT Device the service belongs to */
    char       *name;     /**< Name of service (eg. "DUN") */
    uint8_t     channel;  /**< RFCOMM channel */
    time_t      expiry;   /**< time to live */
    char       *dscr;     /**< description */
} BtService;

/** Callback function type for receiving SDP query result */
typedef void (*btsdp_cb_t)(GError *err, gpointer user_data);

typedef void (*btcon_cb_t)(gint dev_id, GError *err, gpointer data);

/** Get bound RFCOMM device node number
 * @param bda   BDA of device to query
 * @param name  Name of service (eg. "DUN")
 * @returns Device node number or -1 on failure
 */
int bind_dev(const char *bda, const char *name, GError **err);

gboolean release_dev(const char *bda, const char *name, GError **err);

/** Get a list of supported services. List members will be
 *  strings describing each service
 * @returns List of services
 */
GSList *get_service_name_list(const gchar *bda, GError **err);

/** Schedule SDP update for device
 * @param bda       BT address of device
 * @param func      Callback function to call when SDP reply arrives
 * @param svc       Specific service wanted (or NULL for all)
 * @param user_data Optional data to pass to the callback function
 * @param free_data Optional function to free the user_data
 * @param err       Error return location in case of failure (or NULL)
 * @returns FALSE on failure TRUE on success
 */
gboolean schedule_capability_update(const gchar *bda, btsdp_cb_t func,
                                const gchar *svc,
                                gpointer user_data, GFreeFunc free_data,
                                GError **err);

/** Mark device capability as "old" */
void expire_device_capability(const gchar *bda);

/** Disconnect BT connections
 * @param bda           Device to disconnect (NULL to disconnect all)
 * @param clear_dev_info Clear SDP info after disconnect
 */
void bt_disconnect(const gchar *bda, gboolean clear_dev_info);

/** Handle BT error (disconnect and exit) */
void bt_hw_error(uint8_t code);

/** Check if we have info about device and its capabilities
 * @param bda   BT address of device
 * @param svc   Service required (or NULL for any)
 * @returns TRUE if we have device info, else FALSE
 */
gboolean need_capability_update(const gchar *bda, const gchar *svc);

/** Get list of BT devices we have some information about
 * @returns List of strings representing the BT addresses of devices
 */
GSList *get_bda_list(void);

char *bt_devname_from_id(gint dev_num, char *dev, int max_len);

gboolean connect_service(const char *bda, const char *name, const char *owner,
                         gboolean auth, gboolean encr, const char *role,
                         btcon_cb_t func, gpointer user_data, GFreeFunc free_data,
                         GError **err);

gboolean disconnect_service(const char *bda, const char *name, GError **err);

BtService *get_connected_service(uint8_t dev_id);

BtService *get_bound_service(uint8_t dev_id);

BtService *get_service(const char *bda, const char *name, GError **err);

const gchar *bt_service_status(const char *bda, const char *name, GError **err);

void bt_check_owner_disconnect(const char *owner);

#endif /* _BTCOND_BT_H_ */
