/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENSHIVA_KERNEL_SOURCE_H_
#define _OPENSHIVA_KERNEL_SOURCE_H_

#include <list>
#include <OpenShiva/Export.h>

namespace GTLCore {
  class ErrorMessage;
  class String;
}

namespace OpenShiva {
  class Metadata;
  /**
   * @ingroup OpenShiva
   * 
   * Define the source code of a Kernel. Allow to access to information like the metadata,
   * or the source code, or the parameters of the evaluateParam function.
   */
  class OPENSHIVA_EXPORT Source {
    public:
      Source();
      Source(const Source& );
      Source& operator=(const Source& );
      ~Source();
      const GTLCore::String& name() const;
      const GTLCore::String& source() const;
      void setSource( const GTLCore::String& );
      void loadFromFile(const GTLCore::String& _fileName);
      /**
       * @return the metadata for this kernel.
       */
      const Metadata* metadata() const;
      const std::list<GTLCore::ErrorMessage>& metadataCompilationErrors() const;
    public:
      enum SourceType {
        InvalidSource, ///< Not a valid Shiva source code
        Library, ///< \ref Library
        GeneratorKernel, ///< A \ref Kernel with only an output image
        FilterKernel, ///< A \ref Kernel which takes one input
        CompositionKernel ///< A \ref Kernel with multiple input
      };
      /**
       * @return the type of the Kernel/Library
       */
      SourceType sourceType() const;
      enum ImageType {
        InvalidImage,
        Image1,
        Image2,
        Image3,
        Image4,
        Image
      };
      /**
       * @return the type of the output image
       */
      ImageType outputImageType() const;
      /**
       * @return the type of images at index @p idx
       *         (if idx is out of range, return an \ref InvalidImage)
       */
      ImageType inputImageType(int idx) const;
      /**
       * @return the number of input images (-1 indicates the input functions takes an array)
       */
      int countInputImages() const;
    private:
      struct Private;
      Private* const d;
  };
}

#endif
