/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "LightParser_p.h"

#include "Debug.h"
#include "Lexer_p.h"

using namespace OpenShiva;

struct LightParser::Private {
  Lexer* lexer;
  GTLCore::String name;
  Source::SourceType sourceType;
  Source::ImageType outputImageType;
  std::vector<Source::ImageType> inputImageTypes;
};

LightParser::LightParser( Lexer* lexer ) : ParserBase(0, lexer), d(new Private)
{
  d->lexer = lexer;
}

LightParser::~LightParser()
{
  
}

GTLCore::AST::Tree* LightParser::parse()
{
  d->name = "";
  d->sourceType = Source::InvalidSource;
  d->outputImageType = Source::InvalidImage;
  d->inputImageTypes.clear();
  
  getNextToken();
  // Start by ingoring metadata if any
  if( currentToken().type == GTLCore::Token::INFERIOR )
  {
    eatLoopOf( GTLCore::Token::INFERIOR, GTLCore::Token::SUPPERIOR );
    if( isOfType( currentToken(), GTLCore::Token::SEMI ) )
    {
      getNextToken();
    }
  }
  
  // Parse "import"
  while( currentToken().type == GTLCore::Token::IMPORT )
  {
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::STRING_CONSTANT ) )
    {
    }
    checkNextTokenIsSemi();
    getNextToken();
  }
  
  SHIVA_DEBUG("Parse body");
  bool isKernel = false;
  if( currentToken().type == GTLCore::Token::LIBRARY ) {
    d->sourceType = Source::Library;
  } else {
    isKernel = true;
  }
  getNextToken();
  if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
  {
    d->name = currentToken().string;
  }
  
  getNextToken();
  if( isOfType( currentToken(), GTLCore::Token::STARTBRACE ) )
  {
    getNextToken();
    while(true)
    {
      switch(currentToken().type)
      {
        case GTLCore::Token::END_OF_FILE:
        case GTLCore::Token::ENDBRACE:
          return 0;
        case GTLCore::Token::DEPENDENT:
        case GTLCore::Token::CONST:
          eatUntil(GTLCore::Token::SEMI, GTLCore::Token::SEMI);
          getNextToken(); // eath the ';'
          break;
        case GTLCore::Token::STRUCT:
          eatUntil(GTLCore::Token::ENDBRACE, GTLCore::Token::ENDBRACE);
          getNextToken(); // eat the '}'
          getNextToken(); // eat the ';'
          break;
        default:
        {
          getNextToken(); // Return type
          if( currentToken().type == GTLCore::Token::IDENTIFIER )
          {
            bool isEvaluateParam = ("evaluatePixel" == currentToken().string);
            GTL_DEBUG(currentToken().string);
            getNextToken();
            if( isEvaluateParam and isOfType( currentToken(), GTLCore::Token::STARTBRACKET ) )
            {
              getNextToken();
              while(currentToken().type != GTLCore::Token::ENDBRACKET and currentToken().type != GTLCore::Token::END_OF_FILE )
              {
                // Next is either a type or a end bracket
                bool output = false;
                bool varying = false;
                if( currentToken().type == GTLCore::Token::OUTPUT )
                {
                  output = true;
                  getNextToken();
                } else if( currentToken().type == GTLCore::Token::VARYING )
                {
                  varying = true;
                  getNextToken();
                }else if( currentToken().type == GTLCore::Token::INPUT )
                {
                  getNextToken();
                }
                if( currentToken().type == GTLCore::Token::IDENTIFIER )
                {
                  Source::ImageType image;
//                   GTL_DEBUG( currentToken().string );
                  if( currentToken().string == "image" or currentToken().string == "pixel" )
                  {
                    image = Source::Image;
                  } else if( currentToken().string == "image1" or currentToken().string == "pixel1" )
                  {
                    image = Source::Image1;
                  } else if( currentToken().string == "image2" or currentToken().string == "pixel2" )
                  {
                    image = Source::Image2;
                  } else if( currentToken().string == "image3" or currentToken().string == "pixel3" )
                  {
                    image = Source::Image3;
                  } else if( currentToken().string == "image4" or currentToken().string == "pixel4" )
                  {
                    image = Source::Image4;
                  } else {
                    image = Source::InvalidImage;
                  }
                  if( output )
                  {
                    d->outputImageType = image;
                  } else {
                    d->inputImageTypes.push_back(image);
                  }
                  eatUntil( GTLCore::Token::ENDBRACKET, GTLCore::Token::COMA );
                  if(currentToken().type == GTLCore::Token::COMA )
                  {
                    getNextToken(); // eat the ','
                  }
                } else {
                  getNextToken();
                }
              }
              getNextToken(); // eat the ')'
            } else {
              eatUntil( GTLCore::Token::ENDBRACKET, GTLCore::Token::ENDBRACKET );
              getNextToken(); // eath the ')'
            }
            eatLoopOf( GTLCore::Token::STARTBRACE, GTLCore::Token::ENDBRACE );
          }
        }
      }
      
    }
  }
  
  return 0;
}

void LightParser::eatUntil( GTLCore::Token::Type type1,  GTLCore::Token::Type type2)
{
  while( currentToken().type != type1 and currentToken().type != type2 and currentToken().type != GTLCore::Token::END_OF_FILE )
  {
//     GTL_DEBUG( currentToken().type << " " << GTLCore::Token::typeToString(currentToken().type));
    getNextToken();
  }
}

void LightParser::eatLoopOf( GTLCore::Token::Type start, GTLCore::Token::Type end)
{
  int count = 0;
  do
  {
    if( currentToken().type == start )
    {
      ++count;
    } else if( currentToken().type == end ) {
      --count;
    }
    getNextToken();
  } while( count > 0 and currentToken().type != GTLCore::Token::END_OF_FILE  );
}

const GTLCore::String& LightParser::name() const
{
  return d->name;
}

Source::SourceType LightParser::sourceType() const
{
  return d->sourceType;
}

Source::ImageType LightParser::outputImageType() const
{
  return d->outputImageType;
}

const std::vector<Source::ImageType>& LightParser::inputImageTypes() const
{
  return d->inputImageTypes;
}

