/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Compiler_p.h"

#include <llvm/Module.h>
#include <llvm/PassManager.h>

#include "GTLCore/AST/Expression.h"
#include "GTLCore/AST/FunctionDeclaration.h"
#include "GTLCore/AST/Tree.h"
#include "GTLCore/CodeGenerator_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/Optimiser_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/Function_p.h"
#include "GTLCore/Value.h"

#include "Debug.h"
#include "Lexer_p.h"
#include "Library.h"
#include "LibrariesManager.h"
#include "Parser_p.h"

using namespace OpenShiva;

struct Compiler::Private {
  llvm::Module* module;
  GTLCore::ModuleData* moduleData;
  GTLCore::CodeGenerator* codeGenerator;
  Lexer* lexer;
  Parser* parser;
  int channelsNb;
  bool isKernel;
  bool isStdLibrary;
  std::map< GTLCore::String, GTLCore::Value > parameters;
};

Compiler::Compiler( bool _isKernel, int _channelsNb ) : d(new Private)
{
  d->module = 0;
  d->moduleData = 0;
  d->codeGenerator = 0;
  d->channelsNb = _channelsNb;
  d->isKernel = _isKernel;
}

Compiler::~Compiler()
{
  delete d;
}

bool Compiler::isKernel() const
{
  return d->isKernel;
}

bool Compiler::isStdLib() const
{
  return d->isStdLibrary;
}

bool Compiler::compile(bool _isStdLibrary, const GTLCore::String& _sourceCode, const GTLCore::String& _kernelName, GTLCore::ModuleData* _moduleData, GTLCore::String& _nameSpace, const std::map< GTLCore::String, GTLCore::Value >& parameters)
{
  d->isStdLibrary = _isStdLibrary;
  SHIVA_DEBUG("Compile: " << _kernelName << " : " << _sourceCode);
  // Initialise the module structure
  SHIVA_ASSERT( d->module == 0 );
  d->module = _moduleData->llvmModule();
  d->moduleData = _moduleData;
  d->codeGenerator = new GTLCore::CodeGenerator( d->moduleData );
  setModuleData(d->moduleData);
  d->parameters = parameters;

  llvm::LLVMContext& context = d->module->getContext();
  
  // Create Standard Library functions
  // CtlStdLib functions (except print which get a special treatement)
  createStdLibFunction( context, "assert", "assert", GTLCore::Type::Void, 1, GTLCore::Type::Boolean, false);
  createStdLibFunction( context, "isnan_f", "isnan_f", GTLCore::Type::Boolean, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "isnan_h", "isnan_h", GTLCore::Type::Boolean, 1, GTLCore::Type::Float, false);
  // C Math functions
  createStdLibFunction( context, "rand", "rand", GTLCore::Type::Integer32, 1, GTLCore::Type::Integer32, true);
  createStdLibFunction( context, "rand", "intRandomAt", GTLCore::Type::Integer32, 3, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false);
  createStdLibFunction( context, "frand", "floatRandomAt", GTLCore::Type::Float32, 3, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false);
  createStdLibFunction( context, "acos", "acosf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "asin", "asinf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "atan", "atanf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "atan2", "atan2f", GTLCore::Type::Float, 2, GTLCore::Type::Float, false, GTLCore::Type::Float, false);
  createStdLibFunction( context, "cos", "cosf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "sin", "sinf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "tan", "tanf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "cosh", "coshf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "sinh", "sinhf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "tanh", "tanhf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "exp", "expf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "log", "logf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "log10", "log10f", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "pow", "powf", GTLCore::Type::Float, 2, GTLCore::Type::Float, false, GTLCore::Type::Float, false);
  createStdLibFunction( context, "sqrt", "sqrtf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "floor", "floorf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "ceil", "ceilf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "fmod", "fmodf", GTLCore::Type::Float, 2, GTLCore::Type::Float, false, GTLCore::Type::Float, false);
  
  if( not d->isStdLibrary)
  { // of course you don't want to import the standard library when building the standard library
    importModule("shivastdlib");
  }

  // Init the lexer
  std::istringstream iss(_sourceCode);
  d->lexer = new Lexer( &iss );
  d->parser = new Parser( this, d->lexer );
  GTLCore::AST::Tree* tree = d->parser->parse();
  if( tree and errorMessages().size() == 0)
  {
    tree->generate( d->moduleData, d->codeGenerator, &context );
    SHIVA_DEBUG( *d->module );
    for( std::list<GTLCore::AST::FunctionDeclaration*>::const_iterator it = tree->functionsDeclarations().begin();
         it != tree->functionsDeclarations().end(); ++it)
    {
      _moduleData->appendFunction( (*it)->function()->name(), (*it)->function());
    }
  } else {
    // Failure
    SHIVA_DEBUG("failure " << (*errorMessages().begin()).line() << ": " << (*errorMessages().begin()).errorMessage());
//     delete d->module;
    d->module = 0;
  }
  _nameSpace = d->parser->nameSpace();
  // Clean up
  SHIVA_DEBUG("Clean up");
  
  delete tree;

/*  foreach(GTLCore::Function* function, functionsToDelete())
  {
    delete function;
  }*/
  SHIVA_DEBUG("Namespace = " << d->parser->nameSpace() );
  delete d->lexer;
  d->lexer = 0;
  delete d->parser;
  d->parser = 0;
  llvm::Module* mod = d->module;
  delete d->codeGenerator;
  d->codeGenerator = 0;
  d->module = 0;
  d->moduleData = 0;
  SHIVA_DEBUG("Compilation finished");
  return mod;
}

bool Compiler::importModule(const GTLCore::String& name)
{
  Library* library = LibrariesManager::instance()->loadLibrary( name, d->channelsNb );
  if( library )
  {
    if(not library->isCompiled())
    {
      library->compile();
      if(not library->isCompiled())
      {
        SHIVA_DEBUG("Compilation error: " << std::endl << library->compilationErrorsMessage() ); // TODO: report the error
        appendErrors( library->compilationErrors() );
        return false;
      }
    }
    d->moduleData->linkWith( library->data() );
    // Append the function coming from the imported module
    std::list<GTLCore::Function*> functions = library->functions();
    foreach( GTLCore::Function* function, functions )
    {
      SHIVA_DEBUG(function->name());
      GTLCore::Function* newFunction = GTLCore::Function::Private::createExternalFunction( d->moduleData, d->moduleData->llvmModule()->getContext(), function );
      declareFunction( newFunction->name(), newFunction );
      functionsToDelete().push_back( newFunction );
    }
  }
  return library;
}

GTLCore::TypesManager* Compiler::typesManager()
{
  return d->moduleData->typesManager();
}

GTLCore::AST::Expression* Compiler::standardConstant( const GTLCore::String& _name )
{
  // TODO standardConstant
  
/*  // Check if it is a parameter
  std::map< GTLCore::String, GTLCore::Value >::iterator it = d->parameters.find(_name);
  if( it != d->parameters.end() )
  {
    return GTLCore::AST::Expression::fromValue( it->second );
  }*/
  
  return 0;
}

const std::map< GTLCore::String, GTLCore::Value >& Compiler::parameters() const
{
  return d->parameters;
}

int Compiler::channelsNb() const
{
  return d->channelsNb;
}
